package ohd.hseb.hefs.pe.tools;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorTableModel;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepStatusRefreshAllNotice;
import ohd.hseb.hefs.pe.notice.StepUpdatedNotice;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RefreshTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.SelectionMaintainingTableModel;
import ohd.hseb.hefs.utils.status.StatusLabel;

import org.jdesktop.swingx.JXTable;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * This model displays a location id, parameter id, used latitude, and used longitude based on a list of
 * LocationAndDataTypeIdentifier instances. It also allows for displaying status columns, where the status is provided
 * by GenericSummaryTableStatusProvider objects. By default, there are no status columns, with four columns being
 * displayed.<br/>
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class GenericSummaryTableModel extends ParameterEstimatorTableModel implements PreferredWidthsTableModel,
SelectionMaintainingTableModel<LocationAndDataTypeIdentifier>, StepUpdatedNotice.Subscriber,
SelectedIdentifiersChangedNotice.Subscriber, StepStatusRefreshAllNotice.Subscriber
{
    private static final int[] COLUMN_WIDTHS = {55, 25, 55, 55};

    public static String NO_MAPPED_STATION_ID = "-none-";

    protected LocationAndDataTypeIdentifierList _identifiers = new LocationAndDataTypeIdentifierList();

    private final List<GenericSummaryTableStatusProvider> _statusProviders;

    public GenericSummaryTableModel(final ParameterEstimatorRunInfo runInfo,
                                    final Iterable<? extends GenericSummaryTableStatusProvider> statusProviders)
    {
        super(runInfo);
        _statusProviders = Lists.newArrayList(statusProviders);
    }

    public GenericSummaryTableModel(final ParameterEstimatorRunInfo runInfo,
                                    final GenericSummaryTableStatusProvider... statusProviders)
    {
        super(runInfo);
        _statusProviders = Lists.newArrayList(statusProviders);
    }

    public List<GenericSummaryTableStatusProvider> getProviders()
    {
        return _statusProviders;
    }

    /**
     * This copies and sorts the provided identifiers. It will call {@link #fireTableDataChanged()}. If a subclass needs
     * to do something that results in a table structure change when identifiers are set, it will need to override this
     * method and call {@link #fireTableStructureChanged()} instead. Otherwise {@link JXTable} may crash due to data
     * changing when, in reality, the structure changed.
     */
    public void setIdentifiers(final Collection<LocationAndDataTypeIdentifier> identifiers)
    {
        _identifiers = new LocationAndDataTypeIdentifierList(identifiers);
        Collections.sort(_identifiers);
        fireTableDataChanged();
    }

    public LocationAndDataTypeIdentifier getIdentifier(final int row)
    {
        try
        {
            return this._identifiers.get(row);
        }
        catch(final IndexOutOfBoundsException e)
        {
            return null;
        }
    }

    public int getRowOfIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        return _identifiers.indexOf(identifier);
    }

    /**
     * The index of the column which houses the ready indicator for each identifier. -1 for none.
     * 
     * @return index of the indicator column
     */
    public int getStatusColumn(final GenericSummaryTableStatusProvider statusProvider)
    {
        return _statusProviders.indexOf(statusProvider);
    }

    public boolean areStatusColumnsPresent()
    {
        return !_statusProviders.isEmpty();
    }

    public int[] getColumnsForAllStatusProviders()
    {
        final int[] columns = new int[_statusProviders.size()];
        for(int i = 0; i < columns.length; i++)
        {
            columns[i] = 4 + i;
        }
        return columns;
    }

    @Override
    public int getRowCount()
    {
        if(_identifiers == null)
        {
            return 0;
        }
        return _identifiers.size();
    }

    @Override
    public int getColumnCount()
    {
        return 4 + _statusProviders.size();
    }

    @Override
    public String getColumnName(final int column)
    {
        if(column == 0)
        {
            return "Location ID";
        }
        else if(column == 1)
        {
            return "Parameter ID";
        }
        else if(column == 2)
        {
            return "Used Lat";
        }
        else if(column == 3)
        {
            return "Used Lon";
        }
        else if((column > 3) && (column < 4 + _statusProviders.size()))
        {
            return _statusProviders.get(column - 4).getStatusColumnName();
        }
        return null;
    }

    @Override
    public boolean isCellEditable(final int row, final int column)
    {
        return false;
    }

    @Override
    public Object getValueAt(final int row, final int column)
    {
        if(column == 0)
        {
            return this._identifiers.get(row).getLocationId();
        }
        else if(column == 1)
        {
            return this._identifiers.get(row).getParameterId();
        }
        else if(column == 2)
        {
            return this._identifiers.get(row).getUsedLatitude();
        }
        else if(column == 3)
        {
            return this._identifiers.get(row).getUsedLongitude();
        }
        else if((column > 3) && (column < 4 + _statusProviders.size()))
        {
            return StatusLabel.make(_statusProviders.get(column - 4).getStatus(getIdentifier(row)));
        }
        return null;
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex)
    {
        if((columnIndex == 2) || (columnIndex == 3))
        {
            return Double.class;
        }
        else if((columnIndex > 3) && (columnIndex < 4 + _statusProviders.size()))
        {
            return StatusLabel.class;
        }
        return String.class;
    }

    @Override
    public String getColumnHeaderToolTip(final int modelColIndex)
    {
        if((modelColIndex == 2) || (modelColIndex == 3))
        {
            return "If a mapped station is specified, it is the mapped lat and lon.  Otherwise, it is the CHPS location lat and lon.";
        }
        else if((modelColIndex > 3) && (modelColIndex < 4 + _statusProviders.size()))
        {
            return _statusProviders.get(modelColIndex - 4).getToolTipForColumnHeader();
        }
        return null;
    }

    @Override
    public String getCellToolTip(final int rowIndex, final int modelColIndex)
    {
        if(rowIndex == -1)
        {
            return null;
        }
        if((modelColIndex > 3) && (modelColIndex < 4 + _statusProviders.size()))
        {
            return _statusProviders.get(modelColIndex - 4).getStatus(getIdentifier(rowIndex)).getDescription();
        }
        //Return a reprint of the cells value.
        return String.valueOf(getValueAt(rowIndex, modelColIndex));
    }

    @Override
    public Integer getPreferredWidth(final int column)
    {
        if(0 <= column && column < COLUMN_WIDTHS.length)
        {
            return COLUMN_WIDTHS[column];
        }
        return 20;
    }

    @Override
    public LocationAndDataTypeIdentifier getRow(final int index)
    {
        return getIdentifier(index);
    }

    @Override
    public int getRowIndex(final LocationAndDataTypeIdentifier row)
    {
        return _identifiers.indexOf(row);
    }

    @Override
    @Subscribe
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        fireTableDataChanged();
    }

    @Override
    @Subscribe
    public void reactToStepUpdated(final StepUpdatedNotice evt)
    {
        fireTableDataChanged();
    }

    @Override
    @Subscribe
    public void reactToStepStatusRefreshAll(final StepStatusRefreshAllNotice notice)
    {
        //If this implements RefreshTableModel, then call refresh.  Otherwise, just trigger a repaint.
        if(this instanceof RefreshTableModel)
        {
            ((RefreshTableModel)this).refresh();
        }
        else
        {
            this.fireTableDataChanged();
        }
    }

    @Override
    public GenericSummaryTable createDefaultTable()
    {
        return (GenericSummaryTable)super.createDefaultTable();
    }

    @Override
    protected GenericSummaryTable instantiateDefaultTable()
    {
        return new GenericSummaryTable(this);
    }
}
