package ohd.hseb.hefs.pe.tools;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorSubPanel;
import ohd.hseb.hefs.pe.tools.SubEditor.CreatedNotice;
import ohd.hseb.hefs.pe.tools.SubEditor.DestroyedNotice;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.buttons.RefreshButton;
import ohd.hseb.hefs.utils.gui.jtable.buttons.SelectAllButton;
import ohd.hseb.hefs.utils.gui.jtable.buttons.SelectUnreadyRowsButton;
import ohd.hseb.hefs.utils.gui.jtable.buttons.UnselectAllButton;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.tools.ListTools;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;
import com.google.common.primitives.Ints;

/**
 * Displays a generic summary table according to a GenericSummaryTableModel. The panel and model is built based on the
 * arguments to the constructor. Any constructor taking a list of GenericSummaryTableStatusProvider instances will use a
 * standard GenericSummaryTableModel with that list provided to it. It is also possible to create your own subclass of
 * the GenericSummaryTableModel and provide it to a constructor. The list of components at the end of constructors are
 * components to add to the button tool bar displayed below the table.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class GenericSummaryTablePanel extends ParameterEstimatorSubPanel implements SubEditor.CreatedNotice.Subscriber,
SubEditor.DestroyedNotice.Subscriber
{
    public static final Component DEFAULT_BUTTONS_LEFT = new ComponentMarker();
    public static final Component DEFAULT_BUTTONS = new ComponentMarker();
    public static final Component REFRESH_BUTTON = new ComponentMarker();
    public static final Component SELECT_ALL_BUTTON = new ComponentMarker();
    public static final Component SELECT_FALSE_BUTTON = new ComponentMarker();
    public static final Component UNSELECT_BUTTON = new ComponentMarker();
    public static final Component FILLER = new ComponentMarker();
    public static final Component SEPARATOR = new ComponentMarker();

    private final GenericSummaryTableModel _tableModel;
    private GenericSummaryTable _summaryTable;
    private JScrollPane _summaryTablePane;
    private JScrollPane _centerScrollPane;

    private JPanel _centerPanel;

    private final Collection<Component> _centerComponents = Lists.newArrayList();

    public GenericSummaryTablePanel(final ParameterEstimatorRunInfo runInfo,
                                    final String borderTitle,
                                    final GenericSummaryTableModel model,
                                    final List<Component> toolBarComponents)
    {
        super(runInfo);
        _tableModel = model;
        initializeDisplay(borderTitle, toolBarComponents);
        turnOffScrollingForMainPanel();
    }

    public GenericSummaryTablePanel(final ParameterEstimatorRunInfo runInfo,
                                    final String borderTitle,
                                    final GenericSummaryTableModel model,
                                    final boolean includeButtons,
                                    final List<Component> toolBarComponents)
    {
        super(runInfo);
        _tableModel = model;
        if(includeButtons)
        {
            initializeDisplay(borderTitle,
                              ListTools.concat(ImmutableList.of(DEFAULT_BUTTONS_LEFT),
                                               ListTools.nullToEmptyList(toolBarComponents),
                                               ImmutableList.of(FILLER, REFRESH_BUTTON)));
        }
        else
        {
            initializeDisplay(borderTitle, toolBarComponents);
        }
        turnOffScrollingForMainPanel();
    }

    @Override
    protected void initializeDisplay()
    {

    }

    private void initializeDisplay(final String borderTitle, final List<Component> toolBarComponents)
    {
        // Main Panel
        _centerPanel = new JPanel();
        _centerPanel.setLayout(new GridBagLayout());

        _centerScrollPane = new JScrollPane(_centerPanel);

        // Table Panel
        _summaryTable = _tableModel.createDefaultTable();
        _summaryTable.setAutoCreateRowSorter(true);
        _summaryTablePane = new JScrollPane(_summaryTable);
        if(borderTitle != null)
        {
            _summaryTablePane.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1),
                                                                         borderTitle,
                                                                         null));
        }
        addToCenterPanel(_summaryTablePane, 1);
        _centerComponents.clear(); // Don't keep track of center scroll pane.

        setLayout(new BorderLayout());
        add(_centerScrollPane, BorderLayout.CENTER);

        // Lower components panel
        if(!toolBarComponents.isEmpty())
        {
            final JToolBar toolBar = new JToolBar();
            toolBar.setFloatable(false);

            for(int i = 0; i < toolBarComponents.size(); i++)
            {
                final Component comp = toolBarComponents.get(i);

                // One of the table's default components.
                if(comp instanceof ComponentMarker)
                {
                    if(comp == SELECT_ALL_BUTTON || comp == DEFAULT_BUTTONS || comp == DEFAULT_BUTTONS_LEFT)
                    {
                        toolBar.add(new SelectAllButton(_summaryTable));
                    }
                    if(comp == UNSELECT_BUTTON || comp == DEFAULT_BUTTONS || comp == DEFAULT_BUTTONS_LEFT)
                    {
                        toolBar.add(new UnselectAllButton(_summaryTable));
                    }
                    if((comp == SELECT_FALSE_BUTTON || comp == DEFAULT_BUTTONS || comp == DEFAULT_BUTTONS_LEFT)
                        && _tableModel.areStatusColumnsPresent())
                    {
                        toolBar.add(new SelectUnreadyRowsButton(_summaryTable));
                    }
                    if(comp == SEPARATOR || comp == DEFAULT_BUTTONS || comp == DEFAULT_BUTTONS_LEFT)
                    {
                        //Only add a separator if there are two items to separate!
                        if((i < toolBarComponents.size() - 1) && (toolBarComponents.get(i + 1) != FILLER))
                        {
                            toolBar.addSeparator();
                        }
                        continue;
                    }
                    if(comp == FILLER || comp == DEFAULT_BUTTONS)
                    {
                        toolBar.add(HSwingFactory.createFillerJPanel());
                    }
                    if(comp == REFRESH_BUTTON || comp == DEFAULT_BUTTONS)
                    {
                        toolBar.add(new RefreshButton(_summaryTable));
                    }
                }
                // Custom component.
                else
                {
                    toolBar.add(comp);
                }
            }

            add(toolBar, BorderLayout.SOUTH);
        }

        _centerPanel.validate();
        _centerPanel.setMinimumSize(new Dimension(_centerPanel.getSize().width, 0));

        _centerScrollPane.addComponentListener(new ComponentAdapter()
        {
            @Override
            public void componentResized(final ComponentEvent e)
            {
                recomputeSizes();
            }
        });

        recomputeSizes();
    }

    private void recomputeSizes()
    {
        final Dimension extent = _centerScrollPane.getViewport().getExtentSize();
        final Dimension tableDim = _summaryTable.getPreferredSize();
        final Dimension headerDim = _summaryTable.getTableHeader().getPreferredSize();
        final Dimension compMinDim = new Dimension(0, 0);
        final Dimension compPrefDim = new Dimension(0, 0);
        for(final Component comp: _centerComponents)
        {
            if(comp.isVisible())
            {
                compMinDim.width = Math.max(compMinDim.width, comp.getMinimumSize().width);
                compPrefDim.width = Math.max(compPrefDim.width, comp.getPreferredSize().width);
                compMinDim.height += comp.getMinimumSize().height;
                compPrefDim.height += comp.getPreferredSize().height;
            }
        }

        _centerPanel.setPreferredSize(new Dimension(Ints.max(_centerPanel.getPreferredSize().width,
                                                             compPrefDim.width,
                                                             _summaryTable.getPreferredSize().width),
                                                    Math.max(compPrefDim.height + tableDim.height + headerDim.height
                                                        + 1, extent.height)));
        //_summaryTablePane.setPreferredSize(tableDim);
        _centerPanel.validate();
    }

    public void turnOffScrollingForMainPanel()
    {
        remove(_centerScrollPane);
        add(_centerPanel, BorderLayout.CENTER);
    }

    /**
     * Adds a component to the center panel.
     * 
     * @param comp the component to add
     */
    public void addToCenterPanel(final Component comp)
    {
        addToCenterPanel(comp, 0);
    }

    /**
     * Adds a component to the center panel.
     * 
     * @param comp the component to add
     * @param weighty the y weight of the panel
     */
    private void addToCenterPanel(final Component comp, final int weighty)
    {
        final GridBagConstraints con = new GridBagConstraints();
        con.gridx = 0;
        con.gridy = GridBagConstraints.RELATIVE;
        con.weightx = 1;
        con.weighty = weighty;
        if(weighty != 0)
        {
            con.fill = GridBagConstraints.BOTH;
        }
        _centerPanel.add(comp, con);
        _centerComponents.add(comp);
    }

    public void removeFromCenterPanel(final Component comp)
    {
        if(comp != null)
        {
            _centerPanel.remove(comp);
            _centerPanel.validate();
        }
    }

    /**
     * @return The selected {@link LocationAndDataTypeIdentifier} if exactly one is selected; null otherwise (none or
     *         more than one selected).
     */
    public LocationAndDataTypeIdentifier getSelectedIdentifier()
    {
        if(this._summaryTable.getSelectedRowCount() != 1)
        {
            return null;
        }
        return _tableModel.getIdentifier(_summaryTable.getSelectedModelRow());
    }

    /**
     * @return {@link List} of the selected {@link LocationAndDataTypeIdentifier} instances.
     */
    public List<LocationAndDataTypeIdentifier> getSelectedIdentifiers()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        if(_summaryTable.getSelectedRowCount() > 0)
        {
            final int[] rows = this._summaryTable.getSelectedModelRows();
            for(final int row: rows)
            {
                identifiers.add(_tableModel.getIdentifier(row));
            }
        }
        return identifiers;
    }

    public void clearSelection()
    {
        this._summaryTable.clearSelection();
    }

    public void addListSelectionListener(final ListSelectionListener listener)
    {
        _summaryTable.getSelectionModel().addListSelectionListener(listener);
    }

    public void addColumnSelectionListener(final ListSelectionListener listener)
    {
        _summaryTable.getColumnModel().getSelectionModel().addListSelectionListener(listener);
    }

    /**
     * Select the identifier if it can be found in the table model.
     * 
     * @param identifier The identifier to select.
     */
    public void selectIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        final int row = _tableModel.getRowOfIdentifier(identifier);
        if(row >= 0)
        {
            TableTools.selectRowAndScrollToIt(_summaryTable, _summaryTable.convertRowIndexToModel(row));
        }
    }

    /**
     * Select the identifier if it can be found in the table model.
     * 
     * @param identifier The identifier to select.
     */
    public void selectIdentifiers(final Collection<LocationAndDataTypeIdentifier> identifiers)
    {
        if(identifiers.isEmpty())
        {
            _summaryTable.clearSelection();
            return;
        }

        final int selectedRows[] = new int[identifiers.size()];
        int i = 0;
        for(final LocationAndDataTypeIdentifier id: identifiers)
        {
            selectedRows[i] = _tableModel.getRowOfIdentifier(id);
            i++;
        }

        TableTools.selectRows(_summaryTable, selectedRows);
        TableTools.scrollToRow(_summaryTable, selectedRows[0]);
    }

    public void refreshTable()
    {
        this._summaryTable.setVisible(false);
        this._summaryTable.setVisible(true);
    }

    public GenericSummaryTable getTable()
    {
        return this._summaryTable;
    }

    public JScrollPane getSummaryTablePane()
    {
        return this._summaryTablePane;
    }

    public GenericSummaryTableModel getModel()
    {
        return this._tableModel;
    }

    private static class ComponentMarker extends Component
    {
    }

    @Subscribe
    @Override
    public void reactToSubEditorDestroyed(final DestroyedNotice notice)
    {
        removeFromCenterPanel(notice.getSubEditor());
    }

    @Subscribe
    @Override
    public void reactToSubEditorCreated(final CreatedNotice notice)
    {
        if(notice.getSource() == _summaryTable)
        {
            addToCenterPanel(notice.getSubEditor());
            notice.getSubEditor().setVisible(true);
        }
    }
}
