package ohd.hseb.hefs.pe.tools;

import java.util.Calendar;
import java.util.List;

import ohd.hseb.hefs.utils.tools.ParameterId;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

public abstract class HEFSTools
{
    public static final Logger LOG = LogManager.getLogger(HEFSTools.class);

    public static String PRECIPITATION_DATA_TYPE_STRING = "precipitation";
    public static String TEMPERATURE_DATA_TYPE_STRING = "temperature";
    public static String STREAMFLOW_DATA_TYPE_STRING = "streamflow";
    public static String RIVERSTAGE_DATA_TYPE_STRING = "riverstage";

    public static String DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID = "MAP";
    public static String DEFAULT_TEMPERATURE_IDENTIFIER_PARAMETER_ID = "MAT";
    public static String DEFAULT_STREAMFLOW_IDENTIFIER_PARAMETER_ID = "QIN";
    public static String DEFAULT_TMAX_PARAMETER_ID = "TMAX";
    public static String DEFAULT_TMIN_PARAMETER_ID = "TMIN";
    public static String FORECAST_PRECIP_PARAMETER_ID = "FMAP";
    public static String FORECAST_TMAX_PARAMETER_ID = "TFMX";
    public static String FORECAST_TMIN_PARAMETER_ID = "TFMN";
    public static String FORECAST_STREAMFLOW_PARAMETER_ID = "QINE";
    public static String SIMULATED_STREAMFLOW_PARAMETER_ID = "SQIN";
    public static String OBSERVED_STREAMFLOW_PARAMETER_ID = "QIN";
    public static String OBSERVED_QME_PARAMETER_ID = "QME";
    public static String SIMULATED_QME_PARAMETER_ID = "SQME";

    /**
     * @return minimum if true, maximum if false.
     */
    public static String getMinMaxStr(final boolean tmin)
    {
        if(tmin)
        {
            return "minimum";
        }
        return "maximum";
    }

    /**
     * Get the specified parameter id.
     * 
     * @param dataTypeString either {@link #TEMPERATURE_DATA_TYPE_STRING}, {@link #PRECIPITATION_DATA_TYPE_STRING}, or
     *            {@link #STREAMFLOW_DATA_TYPE_STRING}.
     * @param forecast if the parameter id is forecast or not
     * @param maximum True if maximum, false if minimum, null if neither.
     * @return the specified parameter id, or null if unknown
     */
    public static String makeParameterId(final String dataTypeString, final boolean forecast, final Boolean maximum)
    {
        if(TEMPERATURE_DATA_TYPE_STRING.equals(dataTypeString))
        {
            if(forecast)
            {
                if(maximum == null)
                {
                    return "FMAT";
                }
                else if(!maximum)
                {
                    return "TFMN";
                }
                else
                {
                    return "TFMX";
                }
            }
            else
            {
                if(maximum == null)
                {
                    return "MAT";
                }
                else if(!maximum)
                {
                    return "TMIN";
                }
                else
                {
                    return "TMAX";
                }
            }
        }
        else if(PRECIPITATION_DATA_TYPE_STRING.equals(dataTypeString))
        {
            if(maximum != null)
            {
                return null;
            }
            else
            {
                if(forecast)
                {
                    return "FMAP";
                }
                else
                {
                    return "MAP";
                }
            }
        }
        else if(STREAMFLOW_DATA_TYPE_STRING.equals(dataTypeString))
        {
            if(maximum != null)
            {
                return null;
            }
            if(forecast)
            {
                return FORECAST_STREAMFLOW_PARAMETER_ID;
            }
            else
            {
                return DEFAULT_STREAMFLOW_IDENTIFIER_PARAMETER_ID;
            }
        }
        else
        {
            return null;
        }
    }

    public static boolean isForecastDataType(final String parameterId)
    {
        return parameterId.equals(FORECAST_PRECIP_PARAMETER_ID) || parameterId.equals(FORECAST_TMAX_PARAMETER_ID)
            || parameterId.equals(FORECAST_TMIN_PARAMETER_ID) || parameterId.equals(FORECAST_STREAMFLOW_PARAMETER_ID)
            || parameterId.equals(SIMULATED_QME_PARAMETER_ID) || parameterId.equals(SIMULATED_STREAMFLOW_PARAMETER_ID);
    }

    public static boolean isObservedDataType(final String parameterId)
    {
        return parameterId.equals(DEFAULT_TMAX_PARAMETER_ID) || parameterId.equals(DEFAULT_TMIN_PARAMETER_ID)
            || parameterId.equals("TAMX") || parameterId.equals("TAMN")
            || parameterId.equals(OBSERVED_QME_PARAMETER_ID) || parameterId.equals(OBSERVED_STREAMFLOW_PARAMETER_ID)
            || parameterId.equals(DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID)
            || parameterId.equals(DEFAULT_TEMPERATURE_IDENTIFIER_PARAMETER_ID)
            || parameterId.equals(DEFAULT_STREAMFLOW_IDENTIFIER_PARAMETER_ID);
    }

    public static boolean isPrecipitationDataType(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        return isPrecipitationDataType(identifiers.get(0).getParameterId());
    }

    public static boolean isPrecipitationDataType(final LocationAndDataTypeIdentifier identifier)
    {
        return isPrecipitationDataType(identifier.getParameterId());
    }

    public static boolean isPrecipitationDataType(final String parameterId)
    {
        return (parameterId.toUpperCase().contains(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
    }

    public static boolean isTemperatureDataType(final LocationAndDataTypeIdentifier identifier)
    {
        return isTemperatureDataType(identifier.getParameterId());
    }

    public static boolean isTemperatureDataType(final String parameterId)
    {
        return (parameterId.toUpperCase().contains(HEFSTools.DEFAULT_TEMPERATURE_IDENTIFIER_PARAMETER_ID))
            || isTemperatureMaxDataType(parameterId) || isTemperatureMinDataType(parameterId);
    }

    public static boolean isMaximumDataType(final String parameterId)
    {
        return ParameterId.valueOf(parameterId).isMax();
    }

    public static boolean isMinimumDataType(final String parameterId)
    {
        return ParameterId.valueOf(parameterId).isMin();
    }

    public static boolean isTemperatureMaxDataType(final String parameterId)
    {
        return (parameterId.toUpperCase().equals("TAMX"))
            || (parameterId.toUpperCase().equals("TFMX") || (parameterId.toUpperCase().equals(DEFAULT_TMAX_PARAMETER_ID)));
    }

    public static boolean isTemperatureMinDataType(final String parameterId)
    {
        return (parameterId.toUpperCase().equals("TAMN"))
            || (parameterId.toUpperCase().equals("TFMN") || (parameterId.toUpperCase().equals(DEFAULT_TMIN_PARAMETER_ID)));
    }

    public static boolean isStreamflowDataType(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        return isStreamflowDataType(identifiers.get(0).getParameterId());
    }

    public static boolean isStreamflowDataType(final LocationAndDataTypeIdentifier identifier)
    {
        return isStreamflowDataType(identifier.getParameterId());
    }

    public static boolean isStreamflowDataType(final String parameterId)
    {
        return (parameterId.toUpperCase().equals(HEFSTools.DEFAULT_STREAMFLOW_IDENTIFIER_PARAMETER_ID))
            || (parameterId.toUpperCase().equals(HEFSTools.FORECAST_STREAMFLOW_PARAMETER_ID))
            || (parameterId.toUpperCase().equals(HEFSTools.SIMULATED_QME_PARAMETER_ID))
            || (parameterId.toUpperCase().equals(OBSERVED_STREAMFLOW_PARAMETER_ID))
            || (parameterId.toUpperCase().equals(HEFSTools.SIMULATED_STREAMFLOW_PARAMETER_ID) || (parameterId.toUpperCase().equals(HEFSTools.OBSERVED_QME_PARAMETER_ID)));
    }

    /**
     * @param cal The cal for which to determine the day of the year.
     * @return The day of the year, where 2/29 is treated as 3/1 and computations are made on a 365 day calendar.
     */
    public static int computeDayOfYearWithLeapDayBeingMar1(final Calendar cal)
    {
        final int dayOfMonth = cal.get(Calendar.DAY_OF_MONTH);

        final Calendar workingCal = (Calendar)cal.clone();
        workingCal.set(Calendar.YEAR, 1999);
        workingCal.set(Calendar.DAY_OF_MONTH, 1);

        //This should handle cycling 2/29 around to 3/1; subtract 1 because we set the day of month to 1.
        workingCal.add(Calendar.DAY_OF_MONTH, dayOfMonth - 1);

        return workingCal.get(Calendar.DAY_OF_YEAR);
    }

    /**
     * Used for log messaging. This only works for precipitation and temperature data, currently.
     * 
     * @return {@link HEFSTools#PRECIPITATION_DATA_TYPE_STRING} or {@link HEFSTools#TEMPERATURE_DATA_TYPE_STRING} based
     *         on the precipitation flag.
     */
    public static String determineDataTypeString(final boolean precipitation)
    {
        if(precipitation)
        {
            return PRECIPITATION_DATA_TYPE_STRING;
        }
        return TEMPERATURE_DATA_TYPE_STRING;
    }
}
