package ohd.hseb.hefs.pe.tools;

import java.util.ArrayList;
import java.util.List;

import nl.wldelft.util.coverage.Geometry;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.utils.notify.NotifierBase;
import ohd.hseb.hefs.utils.notify.ObjectModifiedNotice;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

/**
 * Provide information required to identify an estimation unit of the EPP3. Specifically, the location id and parameter
 * id.
 * 
 * @author hank.herr
 */
public class LocationAndDataTypeIdentifier extends NotifierBase implements Comparable<LocationAndDataTypeIdentifier>,
XMLWriter, StepUnit
{

    private static Interner<LocationAndDataTypeIdentifier> _interner = Interners.newWeakInterner();

    private final String _locationId;

    private final String _parameterId;

    private String _xmlTagName = "hefsIdentifier";

    private String _mappedLocationId = null;

    private Double _locationLatitude = null;

    private Double _locationLongitude = null;

    private double _mappedLocationLatitude = Double.NaN;

    private double _mappedLocationLongitude = Double.NaN;

    private LocationAndDataTypeIdentifier(final String locationId, final String parameterId)
    {
        _locationId = locationId;
        _parameterId = parameterId;
    }

    private LocationAndDataTypeIdentifier(final LocationAndDataTypeIdentifier base)
    {
        _locationId = base._locationId;
        _parameterId = base._parameterId;
        _mappedLocationId = base._mappedLocationId;
        _locationLatitude = base._locationLatitude;
        _locationLongitude = base._locationLongitude;
        _mappedLocationLatitude = base._mappedLocationLatitude;
        _mappedLocationLongitude = base._mappedLocationLongitude;
    }

    public static LocationAndDataTypeIdentifier get(final LocationAndDataTypeIdentifier sample)
    {
        return _interner.intern(sample);
    }

    public static LocationAndDataTypeIdentifier get(final String locationId, final String parameterId)
    {
        return _interner.intern(new LocationAndDataTypeIdentifier(locationId, parameterId));
    }

    /**
     * @return An identifier with the same locationId, latitude and longitude as the provided base, but with a new
     *         parameterId. Note that the mapped information is not copied over to the new one.
     */
    public static LocationAndDataTypeIdentifier get(final LocationAndDataTypeIdentifier baseIdentifier,
                                                    final String newParameterId)
    {
        final LocationAndDataTypeIdentifier identifier = get(baseIdentifier.getLocationId(), newParameterId);
        identifier.setLocationLatitude(baseIdentifier.getLocationLatitude());
        identifier.setLocationLongitude(baseIdentifier.getLocationLongitude());
        return identifier;
    }

    public static LocationAndDataTypeIdentifier getNonCanonical(final String locationId, final String parameterId)
    {
        _interner.intern(new LocationAndDataTypeIdentifier(locationId, parameterId));
        return new LocationAndDataTypeIdentifier(locationId, parameterId);
    }

    public static LocationAndDataTypeIdentifier get(final TimeSeriesArray ts)
    {
        return get(ts.getHeader());
    }

    public static LocationAndDataTypeIdentifier get(final TimeSeriesHeader header)
    {
        final LocationAndDataTypeIdentifier identifier = get(header.getLocationId(), header.getParameterId());
        identifier.setCoordinates(header);
        return identifier;
    }

    public static void resetInterner()
    {
        _interner = Interners.newWeakInterner();
    }

    public LocationAndDataTypeIdentifier getNonCanonical()
    {
        return new LocationAndDataTypeIdentifier(this);
    }

    public LocationAndDataTypeIdentifier dropForecast()
    {
        return get(_locationId, ParameterId.valueOf(_parameterId).dropForecast().toString());
    }

    public LocationAndDataTypeIdentifier dropExtremum()
    {
        return get(_locationId, ParameterId.valueOf(_parameterId).dropExtremum().toString());
    }

    public void setCoordinates(final TimeSeriesHeader tsHeader)
    {
        final Geometry geometry = tsHeader.getGeometry();
        if(geometry != null)
        {
            this.setLocationLatitude(geometry.getY(0));
            this.setLocationLongitude(geometry.getX(0));
        }
    }

    public boolean isTimeSeriesForThisLocationAndDataType(final TimeSeriesArray ts)
    {
        return ((getLocationId().equals(ts.getHeader().getLocationId())) && (getParameterId().equals(ts.getHeader()
                                                                                                       .getParameterId())));
    }

    public boolean isThisIdentifierForLocationAndDataType(final String locationId, final String parameterId)
    {
        return (_locationId.equals(locationId) && _parameterId.equals(parameterId));
    }

    public boolean isPrecipitationDataType()
    {
        return HEFSTools.isPrecipitationDataType(_parameterId);
    }

    public boolean isTemperatureDataType()
    {
        return HEFSTools.isTemperatureDataType(_parameterId);
    }

    public boolean isStreamflowDataType()
    {
        return HEFSTools.isStreamflowDataType(_parameterId);
    }

    public boolean isForecastDataType()
    {
        return HEFSTools.isForecastDataType(_parameterId);
    }

    public boolean isObservedDataType()
    {
        return HEFSTools.isObservedDataType(_parameterId);
    }

    public boolean isMinimumDataType()
    {
        return HEFSTools.isMinimumDataType(_parameterId);
    }

    public boolean isMaximumDataType()
    {
        return HEFSTools.isMaximumDataType(_parameterId);
    }

    public String buildStringToDisplayInTree()
    {
        return _locationId + " (" + _parameterId + ")";
    }

    public void clearSyntheticLocationId()
    {
        _mappedLocationId = null;
        _mappedLocationLatitude = Double.NaN;
        _mappedLocationLongitude = Double.NaN;
    }

    public void copyMappedLocationInfo(final LocationAndDataTypeIdentifier base)
    {
        setMappedLocation(base.getMappedLocationId(),
                          base.getMappedLocationLatitude(),
                          base.getMappedLocationLongitude());
    }

    public String toStringFull()
    {
        return toString() + "; lat = " + this._locationLatitude + "; lon = " + this._locationLongitude
            + "; synthetic id = " + this._mappedLocationId + "; syn lat = " + this._mappedLocationLatitude
            + "; syn lon = " + this._mappedLocationLongitude;
    }

    public String getLocationId()
    {
        return _locationId;
    }

    public String getParameterId()
    {
        return _parameterId;
    }

    public ParameterId.Type getParameterIdType()
    {
        return ParameterId.valueOf(_parameterId)._type;
    }

    public boolean isMappedLocationSpecified()
    {
        return this._mappedLocationId != null;
    }

    public boolean areMappedLocationCoordinatesValid()
    {
        if(isMappedLocationSpecified())
        {
            if((this._mappedLocationLatitude == Double.NaN) || (this._mappedLocationLongitude == Double.NaN))
            {
                return false;
            }
        }
        return true;
    }

    public String getMappedLocationId()
    {
        return this._mappedLocationId;
    }

    public void setMappedLocationId(final String id)
    {
        this._mappedLocationId = id;
        post(new ObjectModifiedNotice(this));
    }

    public void setMappedLocation(final String id, final double latitude, final double longitude)
    {
        _mappedLocationId = id;
        _mappedLocationLatitude = latitude;
        _mappedLocationLongitude = longitude;
        post(new ObjectModifiedNotice(this));
    }

    public String getUsedLocationId()
    {
        if(this.getMappedLocationId() != null)
        {
            return this.getMappedLocationId();
        }
        return getLocationId();
    }

    public Double getUsedLatitude()
    {
        if(this.getMappedLocationId() != null)
        {
            return this.getMappedLocationLatitude();
        }
        return getLocationLatitude();
    }

    public Double getUsedLongitude()
    {
        if(this.getMappedLocationId() != null)
        {
            return this.getMappedLocationLongitude();
        }
        return getLocationLongitude();
    }

    public Double getLocationLatitude()
    {
        return _locationLatitude == null ? 0 : _locationLatitude;
    }

    /**
     * Sets this identifier's latitude.
     * 
     * @param locationLatitude
     * @throws IllegalStateException if the latitude has already been set and this would change it
     */
    public void setLocationLatitude(final Double locationLatitude)
    {
        if(locationLatitude == null)
        {
            return;
        }
        if(_locationLatitude != null && !NumberTools.nearEquals(locationLatitude, _locationLatitude, 0.01))
        {
            System.err.println(String.format("Warning - overwriting %s's latitude of %06.4f to %06.4f.",
                                             _locationId,
                                             _locationLatitude,
                                             locationLatitude));
        }
        overwriteLocationLatitude(locationLatitude);
    }

    public void overwriteLocationLatitude(final Double locationLatitude)
    {
        this._locationLatitude = locationLatitude;
        post(new ObjectModifiedNotice(this));
    }

    public Double getLocationLongitude()
    {
        return _locationLongitude == null ? 0 : _locationLongitude;
    }

    /**
     * Sets this identifier's longitude.
     * 
     * @param locationLongitude
     * @throws IllegalStateException if the longitude has already been set and this would change it
     */
    public void setLocationLongitude(final Double locationLongitude)
    {
        if(locationLongitude == null)
        {
            return;
        }
        if(_locationLongitude != null && !NumberTools.nearEquals(locationLongitude, _locationLongitude, 0.01))
        {
            System.err.println(String.format("Warning - overwriting %s's longitude of %06.4f to %06.4f.",
                                             _locationId,
                                             _locationLongitude,
                                             locationLongitude));
        }
        overwriteLocationLongitude(locationLongitude);
    }

    public void overwriteLocationLongitude(final Double locationLongitude)
    {
        this._locationLongitude = locationLongitude;
        post(new ObjectModifiedNotice(this));
    }

    public double getMappedLocationLatitude()
    {
        return _mappedLocationLatitude;
    }

    public void setMappedLocationLatitude(final double syntheticLocationLatitude)
    {
        this._mappedLocationLatitude = syntheticLocationLatitude;
        post(new ObjectModifiedNotice(this));
    }

    public double getMappedLocationLongitude()
    {
        return _mappedLocationLongitude;
    }

    public void setMappedLocationLongitude(final double syntheticLocationLongitude)
    {
        this._mappedLocationLongitude = syntheticLocationLongitude;
        post(new ObjectModifiedNotice(this));
    }

    public void setXMLTagName(final String tag)
    {
        this._xmlTagName = tag;
    }

    public boolean checkForFullEquality(final LocationAndDataTypeIdentifier other)
    {
        if(this.equals(other))
        {
            return Objects.equal(getLocationLatitude(), other.getLocationLatitude())
                && Objects.equal(getLocationLongitude(), other.getLocationLongitude())
                && Objects.equal(getMappedLocationId(), other.getMappedLocationId())
                && Objects.equal(getMappedLocationLatitude(), other.getMappedLocationLatitude())
                && Objects.equal(getMappedLocationLongitude(), other.getMappedLocationLongitude());
        }
        else
        {
            return false;
        }
    }

    public boolean isSameTypeOfData(final LocationAndDataTypeIdentifier identifier)
    {
        return ((isPrecipitationDataType() && identifier.isPrecipitationDataType())
            || (isTemperatureDataType() && identifier.isTemperatureDataType()) || (isStreamflowDataType() && identifier.isStreamflowDataType()));
    }

    public boolean isForSameLocation(final LocationAndDataTypeIdentifier identifier)
    {
        return getLocationId().equals(identifier.getLocationId());
    }

    /**
     * Creates a time series header with the default values set in this identifier.
     * 
     * @return
     */
    public DefaultTimeSeriesHeader makeHeader()
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        header.setLocationDescription(this.getLocationId());
        header.setLocationId(this.getLocationId());
        header.setLocationName(this.getLocationId());
        header.setParameterId(this.getParameterId());
        header.setParameterName(this.getParameterId());
        return header;
    }

    @Override
    public int hashCode()
    {
        return buildStringToDisplayInTree().hashCode();
    }

    @Override
    public String toString()
    {
        return "LocationAndDataTypeIdentifier: locationId = " + _locationId + "; parameterId = " + _parameterId;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(obj == this)
        {
            return true;
        }
        if(!(obj instanceof LocationAndDataTypeIdentifier))
        {
            return false;
        }
        final LocationAndDataTypeIdentifier id = (LocationAndDataTypeIdentifier)obj;
        if(!getLocationId().equals(id.getLocationId()) || !getParameterId().equals(id.getParameterId()))
        {
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(final LocationAndDataTypeIdentifier o)
    {
        return buildStringToDisplayInTree().compareTo(o.buildStringToDisplayInTree());
    }

//    @Override
//    public LocationAndDataTypeIdentifier clone()
//    {
//        LocationAndDataTypeIdentifier copy = new LocationAndDataTypeIdentifier(getLocationId(), getParameterId());
//        copy.setLocationLatitude(getLocationLatitude());
//        copy.setLocationLongitude(getLocationLongitude());
//        copy.setMappedLocation(getMappedLocationId(), getMappedLocationLatitude(), getMappedLocationLongitude());
//        return copy;
//    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());
        mainElement.appendChild(XMLTools.createTextNodeElement(request, "locationId", getLocationId()));
        mainElement.appendChild(XMLTools.createTextNodeElement(request, "parameterId", getParameterId()));
        mainElement.appendChild(XMLTools.createTextNodeElement(request, "lat", "" + this.getLocationLatitude()));
        mainElement.appendChild(XMLTools.createTextNodeElement(request, "lon", "" + this.getLocationLongitude()));
        if(this.isMappedLocationSpecified())
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "mappedLocationId",
                                                                   this.getMappedLocationId()));
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "mappedLocationLat",
                                                                   "" + this.getMappedLocationLatitude()));
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "mappedLocationLon",
                                                                   "" + this.getMappedLocationLongitude()));
        }
        return mainElement;
    }

    @Override
    public String getXMLTagName()
    {
        return this._xmlTagName;
    }

    @Override
    public XMLWriter getWriter()
    {
        return this;
    }

    /**
     * @param identifiers List of instances of LocationAndDataTypeIdentifier
     * @return List of location ids that includes all ides among the identifiers.
     */
    public static List<String> createListOfLocationIds(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        final List<String> results = new ArrayList<String>();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            ListTools.addItemIfNotAlreadyInList(results, identifier.getLocationId());
        }
        return results;
    }

    /**
     * @param identifiers List of instances of LocationAndDataTypeIdentifier
     * @return List of buildStringToDisplayInTree results that includes all identifiers.
     */
    public static List<String> createListOfIdentifierStrings(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        final List<String> results = new ArrayList<String>();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            ListTools.addItemIfNotAlreadyInList(results, identifier.buildStringToDisplayInTree());
        }
        return results;
    }

    /**
     * @param identifiers Identifiers to examine
     * @param ts Time series for which to find at least one identifier that matches
     * @return True if at least one identifier matches the time series.
     */
    public static boolean isTimeSeriesForAtLeastOneIdentifier(final List<LocationAndDataTypeIdentifier> identifiers,
                                                              final TimeSeriesArray ts)
    {
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(identifier.isTimeSeriesForThisLocationAndDataType(ts))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @param identifiers List of LocationAndDataTypeIdentifier to search.
     * @param exampleParameterId Parameter id used to determine the data type to be acquired.
     * @return
     */
    public static List<LocationAndDataTypeIdentifier> extractIdentifiers(final List<LocationAndDataTypeIdentifier> identifiers,
                                                                         final DataTypeIdentifier dataType)
    {
        if(dataType.isPrecipitation())
        {
            return extractPrecipitationIdentifiers(identifiers);
        }
        else if(dataType.isTemperature())
        {
            return extractTemperatureIdentifiers(identifiers);
        }
        else if(dataType.isStreamflow())
        {
            return extractStreamflowIdentifiers(identifiers);
        }
        return null;
    }

    /**
     * @param identifiers List of LocationAndDataTypeIdentifier to search.
     * @return List of LocationAndDataTypeIdentifier specifying all those for precipitation data.
     */
    public static List<LocationAndDataTypeIdentifier> extractPrecipitationIdentifiers(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        final List<LocationAndDataTypeIdentifier> results = new ArrayList<LocationAndDataTypeIdentifier>();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(HEFSTools.isPrecipitationDataType(identifier.getParameterId()))
            {
                results.add(identifier);
            }
        }
        return results;
    }

    /**
     * @param identifiers List of LocationAndDataTypeIdentifier to search.
     * @return List of LocationAndDataTypeIdentifier specifying all those for temperature data.
     */
    public static List<LocationAndDataTypeIdentifier> extractTemperatureIdentifiers(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        final List<LocationAndDataTypeIdentifier> results = new ArrayList<LocationAndDataTypeIdentifier>();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(HEFSTools.isTemperatureDataType(identifier.getParameterId()))
            {
                results.add(identifier);
            }
        }
        return results;
    }

    /**
     * @param identifiers List of LocationAndDataTypeIdentifier to search.
     * @return List of LocationAndDataTypeIdentifier specifying all those for streamflow data.
     */
    public static List<LocationAndDataTypeIdentifier> extractStreamflowIdentifiers(final List<LocationAndDataTypeIdentifier> identifiers)
    {
        final List<LocationAndDataTypeIdentifier> results = new ArrayList<LocationAndDataTypeIdentifier>();
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            if(HEFSTools.isStreamflowDataType(identifier.getParameterId()))
            {
                results.add(identifier);
            }
        }
        return results;
    }

    @Override
    public String getMessageName()
    {
        return buildStringToDisplayInTree();
    }

    public final static Function<LocationAndDataTypeIdentifier, String> TO_DISPLAY_STRING = new Function<LocationAndDataTypeIdentifier, String>()
    {
        @Override
        public String apply(final LocationAndDataTypeIdentifier input)
        {
            return input.buildStringToDisplayInTree();
        }
    };

    public String getFiFilename()
    {
        return getLocationId() + "." + getParameterId() + ".fi";
    }
}
