package ohd.hseb.hefs.pe.tools;

import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;

import org.xml.sax.Attributes;

import com.google.common.base.Objects;
import com.google.common.base.Supplier;

public class LocationAndDataTypeIdentifierSupplier implements XMLReader, Supplier<LocationAndDataTypeIdentifier>
{
    public static final String XML_TAG = "hefsIdentifier";

    /**
     * Factory for creating empty LocationAndDataTypeIdentifiers.
     */
    public static final XMLReaderFactory<LocationAndDataTypeIdentifierSupplier> DEFAULT_FACTORY = new XMLReaderFactory<LocationAndDataTypeIdentifierSupplier>()
    {
        @Override
        public LocationAndDataTypeIdentifierSupplier get()
        {
            return new LocationAndDataTypeIdentifierSupplier();
        }
    };

    private final String _xmlTagName;
    private String _locationId = null;
    private String _parameterId = null;
    private String _mappedLocationId = null;
    private Double _locationLatitude = null;
    private Double _locationLongitude = null;
    private double _mappedLocationLatitude = Double.NaN;
    private double _mappedLocationLongitude = Double.NaN;

    public LocationAndDataTypeIdentifierSupplier()
    {
        this((String)null);
    }

    public LocationAndDataTypeIdentifierSupplier(final String tagName)
    {
        _xmlTagName = Objects.firstNonNull(tagName, XML_TAG);
    }

    public LocationAndDataTypeIdentifierSupplier(final LocationAndDataTypeIdentifierSupplier base)
    {
        this._xmlTagName = base._xmlTagName;
        this._locationId = base._locationId;
        this._parameterId = base._parameterId;
        this._mappedLocationId = base._mappedLocationId;
        this._locationLatitude = base._locationLatitude;
        this._locationLongitude = base._locationLongitude;
        this._mappedLocationLatitude = base._mappedLocationLatitude;
        this._mappedLocationLongitude = base._mappedLocationLongitude;
    }

    public LocationAndDataTypeIdentifierSupplier(final String locationId, final String parameterId)
    {
        _xmlTagName = XML_TAG;
        _locationId = locationId;
        _parameterId = parameterId;
    }

    public boolean canCreate()
    {
        return _locationId != null && _parameterId != null;
    }

    public String getLocationId()
    {
        return _locationId;
    }

    public String getParameterId()
    {
        return _parameterId;
    }

    public void setLocationId(final String id)
    {
        _locationId = id;
    }

    public void setParameterId(final String id)
    {
        _parameterId = id;
    }

    public void setMappedLocation(final String id, final double latitude, final double longitude)
    {
        _mappedLocationId = id;
        _mappedLocationLatitude = latitude;
        _mappedLocationLongitude = longitude;
    }

    public boolean isMappedLocationSpecified()
    {
        return this._mappedLocationId != null;
    }

    public boolean areMappedLocationCoordinatesValid()
    {
        if(isMappedLocationSpecified())
        {
            if((this._mappedLocationLatitude == Double.NaN) || (this._mappedLocationLongitude == Double.NaN))
            {
                return false;
            }
        }
        return true;
    }

    public String getMappedLocationId()
    {
        return this._mappedLocationId;
    }

    public double getMappedLocationLatitude()
    {
        return _mappedLocationLatitude;
    }

    public void setMappedLocationLatitude(final double syntheticLocationLatitude)
    {
        this._mappedLocationLatitude = syntheticLocationLatitude;
    }

    public double getMappedLocationLongitude()
    {
        return _mappedLocationLongitude;
    }

    public void setMappedLocationLongitude(final double syntheticLocationLongitude)
    {
        this._mappedLocationLongitude = syntheticLocationLongitude;
    }

    public void setLocationLatitude(final Double latitude)
    {
        _locationLatitude = latitude;
    }

    public void setLocationLongitude(final Double longitude)
    {
        _locationLongitude = longitude;
    }

    @Override
    public LocationAndDataTypeIdentifier get()
    {
        final LocationAndDataTypeIdentifier identifier = LocationAndDataTypeIdentifier.get(_locationId, _parameterId);
        identifier.setLocationLatitude(_locationLatitude);
        identifier.setLocationLongitude(_locationLongitude);
        identifier.setMappedLocationId(_mappedLocationId);
        identifier.setMappedLocationLatitude(_mappedLocationLatitude);
        identifier.setMappedLocationLongitude(_mappedLocationLongitude);
        identifier.setXMLTagName(_xmlTagName);
        return identifier;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("locationId"))
        {
            _locationId = value;
        }
        else if(elementName.equals("parameterId"))
        {
            _parameterId = value;
        }
        else if(elementName.equals("lat"))
        {
            try
            {
                _locationLatitude = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Value of element locationLat, '" + value + "', is not a number.");
            }
        }
        else if(elementName.equals("lon"))
        {
            try
            {
                _locationLongitude = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Value of element locationLon, '" + value + "', is not a number.");
            }
        }
        else if(elementName.equals("mappedLocationId"))
        {
            _mappedLocationId = value;
        }
        else if(elementName.equals("mappedLocationLat"))
        {
            try
            {
                _mappedLocationLatitude = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Value of element mappedLocationLat, '" + value + "', is not a number.");
            }
        }
        else if(elementName.equals("mappedLocationLon"))
        {
            try
            {
                _mappedLocationLongitude = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Value of element mappedLocationLon, '" + value + "', is not a number.");
            }
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader getReader()
    {
        return this;
    }

    @Override
    public String getXMLTagName()
    {
        return _xmlTagName;
    }
}
