package ohd.hseb.hefs.pe.tools;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.SortedSet;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;

/**
 * A collection kept in sorted order. Implemented with an array list.<br/>
 * If you are planning on modifying the objects in place then it will not remain sorted. Call
 * {@link #informChanged(Comparable)} on any changed objects to resort them.<br/>
 * <br/> {@link #remove(int)} is rather slow.<br/>
 * 
 * @author alexander.garbarino
 * @param <E> the element this list stores
 */
public class SortedCollection<E extends Comparable<? super E>> implements Collection<E>
{
    private final ArrayList<E> _delegate;
    private final Comparator<? super E> _comparator;

    public SortedCollection()
    {
        _delegate = Lists.newArrayList();
        _comparator = null;
    }

    public SortedCollection(Collection<E> coll)
    {
        _delegate = Lists.newArrayList();
        _comparator = null;
        this.addAll(coll);
    }

    public SortedCollection(E... elts)
    {
        _delegate = Lists.newArrayList();
        _comparator = null;
        this.addAll(elts);
    }

    public SortedCollection(Comparator<E> comp)
    {
        _delegate = Lists.newArrayList();
        _comparator = comp;
    }

    public SortedCollection(Comparator<E> comp, Collection<E> coll)
    {
        _delegate = Lists.newArrayList();
        _comparator = comp;
        this.addAll(coll);
    }

    public SortedCollection(Comparator<E> comp, E... elts)
    {
        _delegate = Lists.newArrayList();
        _comparator = comp;
        this.addAll(elts);
    }

    /**
     * Used internally to make a sublist view.
     * 
     * @param delegate delegate to use
     */
    private SortedCollection(ArrayList<E> delegate, Comparator<? super E> comp)
    {
        _delegate = delegate;
        _comparator = comp;
    }

    @Override
    public int size()
    {
        return _delegate.size();
    }

    @Override
    public boolean isEmpty()
    {
        return _delegate.isEmpty();
    }

    @Override
    public boolean contains(Object o)
    {
        return _delegate.contains(o);
    }

    @Override
    public Iterator<E> iterator()
    {
        return _delegate.iterator();
    }

    @Override
    public Object[] toArray()
    {
        return _delegate.toArray();
    }

    @Override
    public <T> T[] toArray(T[] a)
    {
        return _delegate.toArray(a);
    }

    @Override
    public boolean add(E e)
    {
        int index = Collections.binarySearch(_delegate, e, _comparator);
        if(index < 0)
        {
            _delegate.add(-index - 1, e);
            return true;
        }
        else
        {
            return false;
        }
    }

    @Override
    public boolean remove(Object o)
    {
        return _delegate.remove(o);
    }

    @Override
    public boolean containsAll(Collection<?> c)
    {
        return _delegate.containsAll(c);
    }

    @Override
    public boolean addAll(Collection<? extends E> c)
    {
        boolean changed = false;
        for(E e: c)
        {
            changed = this.add(e) || changed;
        }
        return changed;
    }

    public boolean addAll(E... elts)
    {
        boolean changed = false;
        for(int i = 0; i < elts.length; i++)
        {
            changed = this.add(elts[i]) || changed;
        }
        return changed;
    }

    @Override
    public boolean removeAll(Collection<?> c)
    {
        return _delegate.removeAll(c);
    }

    @Override
    public boolean retainAll(Collection<?> c)
    {
        return _delegate.retainAll(c);
    }

    @Override
    public void clear()
    {
        _delegate.clear();
    }

    public E get(int index)
    {
        return _delegate.get(index);
    }

    public E remove(int index)
    {
        return _delegate.remove(index);
    }

    public int indexOf(Object o)
    {
        return _delegate.indexOf(o);
    }

    public int lastIndexOf(Object o)
    {
        return _delegate.lastIndexOf(o);
    }

    public ListIterator<E> listIterator()
    {
        return _delegate.listIterator();
    }

    public ListIterator<E> listIterator(int index)
    {
        return _delegate.listIterator(index);
    }

    public SortedCollection<E> subCollection(int fromIndex, int toIndex)
    {
        return new SortedCollection<E>((ArrayList<E>)_delegate.subList(fromIndex, toIndex), _comparator);
    }

    /**
     * As {@link SortedSet}{@link #first()}.
     */
    public E first()
    {
        return _delegate.get(0);
    }

    /**
     * As {@link SortedSet}{@link #last()}.
     */
    public E last()
    {
        return _delegate.get(_delegate.size() - 1);
    }

    /**
     * Informs this list that the given element has changed and may need to be resorted.
     * 
     * @param index the index of the changed element
     */
    public void informChanged(int index)
    {
        E elt = this.remove(index);
        if(elt != null)
        {
            add(elt);
        }
    }

    /**
     * Informs this list that all of its elements may have been changed and need to be resorted.
     */
    public void informChanged()
    {
        Collections.sort(_delegate);
    }

    /**
     * Equal if the other is a collection with the same instances in the same order.
     */
    @Override
    public boolean equals(Object other)
    {
        if(other == null)
        {
            return false;
        }
        if(!(other instanceof Collection))
        {
            return false;
        }
        Collection that = (Collection)other;
        if(this.size() != that.size())
        {
            return false;
        }
        Iterator i0 = this.iterator();
        Iterator i1 = that.iterator();

        while(i0.hasNext())
        {
            if(!i0.next().equals(i1.next()))
            {
                return false;
            }
        }

        return true;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(_delegate, _comparator);
    }

    @Override
    public String toString()
    {
        String str = _delegate.toString();
        if(_comparator != null)
        {
            str = str.substring(0, 1) + _comparator.toString() + "|" + str.substring(1);
        }
        return str;
    }
}
