package ohd.hseb.hefs.pe.tools;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.RepaintManager;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartPanelTools;
import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.panel.ChartEngineChartPanel;
import ohd.hseb.charter.panel.ChartEngineTableModel;
import ohd.hseb.charter.panel.CombinedDomainChartNavigationPanel;
import ohd.hseb.charter.panel.DomainSharingTimeSeriesChartEngineTableModel;
import ohd.hseb.charter.panel.OHDFixedChartPanel;
import ohd.hseb.charter.panel.notices.TableCellControlClickedNotice;
import ohd.hseb.hefs.mefp.sources.cfsv2.CFSv2MonthlyDiagnosticPanel;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.gui.DiagnosticsDisplayPanel;
import ohd.hseb.hefs.utils.gui.components.ClickableJLabel;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.JListUtilities;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.IconTools;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.util.misc.HCalendar;

import org.jdesktop.swingx.JXCollapsiblePane;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;

import com.google.common.base.Supplier;
import com.google.common.collect.Lists;
import com.google.common.collect.TreeMultimap;
import com.google.common.eventbus.Subscribe;

/**
 * Abstract panel to hold chart-based diagnostics. Subclasses must implement the abstract methods, including most
 * importantly {@link #buildChart()}. This only works with time series based charts.<br>
 * <br>
 * The chart displayed is constructed from the {@link ChartEngine} returned by the method {@link #buildChart()}. It
 * calls {@link ChartPanelTools#buildPanelFromChartEngine(ChartEngine, JFreeChart, boolean, Class)} in order to
 * build the displayed {@link ChartEngineChartAndTablePanel}. The return of method {@link #getDataTableClass()} is used
 * to specify the table model within that {@link ChartEngineChartAndTablePanel}. In most cases, the default table model
 * class returned by that method, {@link TimeSeriesChartDiagnosticTableModel}, will work. However, that model requires
 * that the displayed time series be able to share a common domain (see the
 * {@link DomainSharingTimeSeriesChartEngineTableModel} for more info). See {@link CFSv2MonthlyDiagnosticPanel} for an
 * example where that is not the case and how it is handled using a diagnostic specific table model. <br>
 * <br>
 * The {@link ChartEngineChartAndTablePanel}'s displayed {@link ChartEngineChartPanel} includes an associated
 * {@link CombinedDomainChartNavigationPanel} displayed within {@link #_collapsiblePane} below the
 * {@link ChartEngineChartAndTablePanel}. The {@link CombinedDomainChartNavigationPanel} is constructed via calling
 * {@link #constructNavigationPanel(OHDFixedChartPanel)} when this panel is built. It is only made visible via the glass
 * pane button corresponding to the {@link NavigateAction}. <br>
 * <br>
 * Note that these diagnostic panels include two additional buttons visible in the diagnostic panel glass pane in
 * addition to those available by default via {@link DiagnosticsDisplayPanel} (a close button and detach button): a
 * button to toggle data table visibility on/off (see {@link ShowTableAction}; it calls
 * {@link ChartEngineChartAndTablePanel#setTableShowing(boolean)}) and a button to toggle the visibility of the
 * navigation panel (see {@link NavigateAction}; it toggles the collapsed state of {@link #_collapsiblePane}).
 * 
 * @author hankherr
 */
public abstract class TimeSeriesChartDiagnosticPanel extends ParameterEstimatorDiagnosticPanel implements
TableCellControlClickedNotice.Subscriber
{

    public final static long serialVersionUID = 1L;

    /**
     * Container for displaying a {@link ChartEngineChartAndTablePanel} or an error message panel.
     */
    private JPanel _chartContainer;

    /**
     * Indicates if all time series should be displayed. Only included if {@link #hasYearSpinner()} is true.
     */
    private JCheckBox _showAllBox;

    /**
     * Allows for user selection of displayed reforecasts/archived forecasts year. Only included if
     * {@link #hasYearSpinner()} is true.
     */
    private JComboBox _yearChoiceBox;

    /**
     * Allows for user selections of emphasized forecast times. Only included if {@link #hasForecastList()} is true.
     */
    private JList _forecastTimesList;

    /**
     * This list that is displayed within the {@link #_forecastTimesList}.
     */
    private final List<Long> _forecastTimes = Lists.newArrayList();

    /**
     * Map of the year to the forecast times for that year. The tree map keeps things sorted and prevents duplicates,
     * which is a possibilities because both mean and ensemble member time series may be included.
     */
    private final TreeMultimap<Integer, Long> _yearToForecastTimesMap = TreeMultimap.create();

    /**
     * Identifier for which time series are displayed.
     */
    private final LocationAndDataTypeIdentifier _identifier;

    /**
     * All time series, observed and reforecast/archived forecasts, that may need to be displayed.
     */
    private final ArrayList<TimeSeriesArray> _allSeries;

    /**
     * The format to use for the list of forecast times, {@link #_forecastTimesList}. Can be set via
     * {@link #setDateListDateFormat(String)}.
     */
    private String _dateListDateFormat = HCalendar.DEFAULT_DATEONLY_FORMAT;

    /**
     * To hold a navigation panel.
     */
    private final JPanel _navPanelHolder = new JPanel(new BorderLayout());

    /**
     * Holds the navigation panel holder.
     */
    private JXCollapsiblePane _collapsiblePane = null;

    /**
     * @param identifier Identifier for the data.
     * @param series Time series displayed, used to build GUI components.
     */
    protected TimeSeriesChartDiagnosticPanel(final LocationAndDataTypeIdentifier identifier,
                                             final Collection<TimeSeriesArray>... series)
    {
        _identifier = identifier;

        // Collate series and populate the map of year to times.
        _allSeries = new ArrayList<TimeSeriesArray>();
        for(final Collection<TimeSeriesArray> seriesColl: series)
        {
            for(final TimeSeriesArray ts: seriesColl)
            {
                if(!TimeSeriesArrayTools.isAllMissing(ts))
                {
                    _allSeries.add(ts);

                    if(HEFSTools.isForecastDataType(ts.getHeader().getParameterId()))
                    {
                        final Long forecastTime = new Long(ts.getHeader().getForecastTime());
                        final int year = HCalendar.computeCalendarFromMilliseconds(forecastTime).get(Calendar.YEAR);
                        _yearToForecastTimesMap.put(year, forecastTime);
                    }
                }
            }
        }

        createDisplay();
        if(hasForecastList())
        {
            updateForecastTimesList();
        }
    }

    /**
     * @return if the panel has a year spinner.
     */
    protected abstract boolean hasYearSpinner();

    /**
     * @return if the panel has a forecast selector list based on forecast times.
     */
    protected abstract boolean hasForecastList();

    /**
     * If true, then a navigation panel will be available on the southern edge of the diagnostics panel, and the check
     * box will be displayed.
     * 
     * @return False by default.
     */
    protected boolean allowsSouthNavigationPanel()
    {
        return true;
    }

    /**
     * Override as needed.
     * 
     * @return A default {@link Supplier} for use in a {@link ChartEngineChartAndTablePanel}; see that class'
     *         constructors. Returns an instance of {@link TimeSeriesChartDiagnosticTableModel} for which the
     */
    protected Supplier<ChartEngineTableModel> buildTableModelSupplier()
    {
        return new Supplier<ChartEngineTableModel>()
        {
            @Override
            public ChartEngineTableModel get()
            {
                return new TimeSeriesChartDiagnosticTableModel();
            }
        };
    }

    /**
     * Override as needed. Builds the {@link ChartEngine} to be used to construct the displayed chart.
     * 
     * @return the new chart to display
     */
    protected abstract ChartEngine buildChart() throws Exception;

    /**
     * Creates the display based on provided flags from subclassing and overriding methods.
     */
    private void createDisplay()
    {
        setLayout(new BorderLayout());

        // Chart Panel.
        _chartContainer = new JPanel(new BorderLayout());
        add(_chartContainer, BorderLayout.CENTER);

        // East Panel.
        JPanel eastPanel = null;
        if(hasYearSpinner() || hasForecastList())
        {
            eastPanel = new JPanel(new BorderLayout());
            add(eastPanel, BorderLayout.EAST);
        }

        if(hasYearSpinner())
        {
            final JComponent spinner = createYearChoiceBox();
            if(spinner != null)
            {
                eastPanel.add(spinner, BorderLayout.NORTH);
            }
        }
        if(hasForecastList())
        {
            eastPanel.add(createForecastList(), BorderLayout.CENTER);
        }

        if(allowsSouthNavigationPanel())
        {
            _collapsiblePane = new JXCollapsiblePane();
            _collapsiblePane.setLayout(new BorderLayout());
            _collapsiblePane.add(_navPanelHolder, BorderLayout.CENTER);
            _navPanelHolder.setPreferredSize(new Dimension(500, 100));
            _collapsiblePane.setAnimated(false);
            _collapsiblePane.setCollapsed(true);

            final JPanel southPanel = new JPanel(new BorderLayout());
            southPanel.add(_collapsiblePane, BorderLayout.CENTER);

            add(southPanel, BorderLayout.SOUTH);
        }

    }

    /**
     * @return Show all checkbox for combining all data into one.
     */
    private JCheckBox createShowAllBox()
    {
        _showAllBox = new JCheckBox("ShowAll", false);
        _showAllBox.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                reactToYearChange();
            }
        });
        _showAllBox.setBackground(Color.white);
        return _showAllBox;
    }

    /**
     * @return Panel displaying year spinner.
     */
    @SuppressWarnings("serial")
    private JPanel createYearChoiceBox()
    {
        final long range[] = TimeSeriesArraysTools.getRangeOfForecastTimes(new TimeSeriesSorter(_allSeries).restrictViewToForecast(),
                                                                           true);
        if(range == null)
        {
            return null;
        }

        final int minYear = HCalendar.computeCalendarFromMilliseconds(range[0]).get(Calendar.YEAR);
        final int maxYear = HCalendar.computeCalendarFromMilliseconds(range[1]).get(Calendar.YEAR);

        _yearChoiceBox = HSwingFactory.createJComboBox(NumberTools.generateNumberSequence(minYear, 1, maxYear), null);
        _yearChoiceBox.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(final ItemEvent e)
            {
                reactToYearChange();
            }
        });

        //The renderer will indicate if questionable data is found for that year.
        _yearChoiceBox.setRenderer(new DefaultListCellRenderer()
        {
            @Override
            public Component getListCellRendererComponent(final JList list,
                                                          final Object value,
                                                          final int index,
                                                          final boolean isSelected,
                                                          final boolean cellHasFocus)
            {
                final Component c = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);

                //Only change the display component if the _chartContainer is showing a ChartEngineChartAndTablePanel.
                //In some cases, _chartContainer may display an error message; hence the instance of check done here.
                if((_chartContainer.getComponentCount() > 0)
                    && (_chartContainer.getComponent(0) instanceof ChartEngineChartAndTablePanel))
                {
                    if(!_showAllBox.isSelected())
                    {
                        boolean questionable = false;

                        //Check forecasts for questionable data.
                        final Collection<Long> forecastTimes = generateForecastTimesList((Integer)value);
                        final TimeSeriesChartDiagnosticTableModel model = (TimeSeriesChartDiagnosticTableModel)((ChartEngineChartAndTablePanel)_chartContainer.getComponent(0)).getTableModel();
                        final QuestionableMessageMap qMap = model.getQuestionableMap();

                        //Can't do anything with components that are not JLabels, but this should never happen.
                        if(!(c instanceof JLabel))
                        {
                            return c;
                        }

                        if(qMap != null)
                        {
                            if(qMap.isAnyForecastValueQuestionable(forecastTimes))
                            {
                                questionable = true;
                            }

                            //Now check the observed, restricting its date range to be that covered by the forecast time series.
                            final TimeSeriesSorter sorter = new TimeSeriesSorter(_allSeries);
                            //times is based on data, not T0s; forecastInYear(...) which is called below is based on T0s
                            final long[] times = TimeSeriesArraysTools.getRangeOfTimes(sorter.forecastInYear((Integer)value));
                            if((times != null)
                                && (qMap.isAnyObservedValueQuestionable(new Period(new Date(times[0]),
                                                                                   new Date(times[1])))))
                            {
                                questionable = true;
                            }
                        }

                        //Mark any questionable data in red and with a (?).
                        if(questionable)
                        {
                            ((JLabel)c).setText(((JLabel)c).getText() + " (?)");
                            ((JLabel)c).setBackground(Color.RED);
                        }

                        //Mark years that have no data in yellow and with a (X).
                        else if((_yearToForecastTimesMap.get((Integer)value) == null)
                            || (_yearToForecastTimesMap.get((Integer)value).isEmpty()))
                        {
                            ((JLabel)c).setText(((JLabel)c).getText() + " (X)");
                            ((JLabel)c).setBackground(Color.YELLOW);
                        }

                    }
                }
                return c;
            }
        });

        //The year panel contains the year spinner and show all checkbox.
        final JPanel yearPanel = new JPanel(new BorderLayout());
        yearPanel.add(createShowAllBox(), BorderLayout.NORTH);
        yearPanel.add(_yearChoiceBox, BorderLayout.CENTER);
        yearPanel.setBackground(Color.white);
        yearPanel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1), "Year", null));
        return yearPanel;
    }

    /**
     * @return Scroll pane displaying forecasts by forecast time, wrapping the {@link #_forecastTimesList}
     */
    private JScrollPane createForecastList()
    {
        final JScrollPane scrollPane = HSwingFactory.createScrollPanedJList(null);
        _forecastTimesList = (JList)scrollPane.getViewport().getView();
        final ListCellRenderer defaultRenderer = _forecastTimesList.getCellRenderer();

        //Enables toggling of selections by clicking a second time and single selection mode only.
        _forecastTimesList.setSelectionModel(new DefaultListSelectionModel()
        {
            private static final long serialVersionUID = 1L;

            @Override
            public void setSelectionInterval(final int index0, final int index1)
            {
                //Clears the selection if reselected or a negative index0 is provided.  
                if(index0 >= 0)
                {
                    if(isSelectedIndex(index0))
                    {
                        clearSelection();
                    }
                    else
                    {
                        super.setSelectionInterval(index0, index0);
                    }
                }
                else
                {
                    clearSelection();
                }
            }
        });
        _forecastTimesList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        //Renders milliseconds and date strings
        _forecastTimesList.setCellRenderer(new ListCellRenderer()
        {
            @Override
            public Component getListCellRendererComponent(final JList list,
                                                          final Object value,
                                                          final int index,
                                                          final boolean isSelected,
                                                          final boolean cellHasFocus)
            {
                final Long millis = (Long)value;
                final Calendar cal = HCalendar.computeCalendarFromMilliseconds(millis);
                return defaultRenderer.getListCellRendererComponent(list,
                                                                    HCalendar.buildDateStr(cal, _dateListDateFormat),
                                                                    index,
                                                                    isSelected,
                                                                    cellHasFocus);
            }
        });

        //Scrolls the inner panel table upon emphasizing a column
        _forecastTimesList.addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                if(!e.getValueIsAdjusting())
                {
                    if(getInnerPanel() != null)
                    {
                        getInnerPanel().getTable().setScrollForxisLimitsChanges(false);
                    }
                    updateChart(true);
                    if(getInnerPanel() != null)
                    {
                        getInnerPanel().getTable().setScrollForxisLimitsChanges(true);
                    }
                }
            }
        });

        //The scroll pane stuff.
        scrollPane.setBackground(Color.white);
        scrollPane.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1), "T0", null));

        return scrollPane;
    }

    /**
     * @param format An {@link HCalendar} date format (check the constants) or {@link SimpleDateFormat} (note that
     *            {@link HCalendar} converts its own format to {@link SimpleDateFormat} for use in order to take
     *            advantage of better speed).
     */
    public void setDateListDateFormat(final String format)
    {
        this._dateListDateFormat = format;
    }

    /**
     * @return The {@link ChartEngineChartAndTablePanel} displayed in here or null if no such panel is displayed due to
     *         an error.
     */
    private ChartEngineChartAndTablePanel getInnerPanel()
    {
        if(_chartContainer.getComponent(0) instanceof ChartEngineChartAndTablePanel)
        {
            return (ChartEngineChartAndTablePanel)_chartContainer.getComponent(0);
        }
        return null;
    }

    /**
     * @return The year selected in the spinner or null if no spinner or show all is selected.
     */
    protected Integer getSelectedYear()
    {
        if(_yearChoiceBox == null || _showAllBox.isSelected())
        {
            return null;
        }
        return (Integer)_yearChoiceBox.getSelectedItem();
    }

    /**
     * @return The forecast times selected from the {@link #_forecastTimesList}.
     */
    protected Collection<Long> getSelectedDates()
    {
        final Collection<Long> selectedDates = new ArrayList<Long>();
        for(final int selectedIndex: _forecastTimesList.getSelectedIndices())
        {
            selectedDates.add((Long)_forecastTimesList.getModel().getElementAt(selectedIndex));
        }
        return selectedDates;
    }

    /**
     * @return The {@link LocationAndDataTypeIdentifier} for which observed and reforecast/archived time series are
     *         displayed.
     */
    protected LocationAndDataTypeIdentifier getIdentifier()
    {
        return _identifier;
    }

    /**
     * @return All time series that could be displayed.
     */
    protected List<TimeSeriesArray> getAllSeries()
    {
        return _allSeries;
    }

    /**
     * @param desiredYear The year for which to acquire time series.
     * @return {@link List} of forecast times (milliseconds) of time series for which the forecast time year is within
     *         the desired year. This will skip observed time series.
     */
    private Collection<Long> generateForecastTimesList(final Integer desiredYear)
    {
//OLD CODE USED TO REACQUIRE LIST OF TIMES REPEATEDLY WHEN RENDERING THE YEAR CHOICE BOX.  NEW VERSION USES THE ATTRIBUTE MAP.
//        final List<Long> times = Lists.newArrayList();
//        for(final TimeSeriesArray tsa: _allSeries)
//        {
//            if(HEFSTools.isForecastDataType(tsa.getHeader().getParameterId()))
//            {
//                final Long forecastTime = new Long(tsa.getHeader().getForecastTime());
//                final int year = HCalendar.computeCalendarFromMilliseconds(forecastTime).get(Calendar.YEAR);
//
//                if((desiredYear == null) || (year == desiredYear))
//                {
//                    if(!times.contains(forecastTime))
//                    {
//                        times.add(forecastTime);
//                    }
//                }
//            }
//        }
//        return times;
        if(desiredYear == null)
        {
            return _yearToForecastTimesMap.values();
        }
        return _yearToForecastTimesMap.get(desiredYear);
    }

    /**
     * Call to update {@link #_forecastTimes} and pass the new list into {@link #_forecastTimesList}. Note that the
     * times will be in order because {@link #_allSeries} is ordered by forecast time.
     */
    protected void updateForecastTimesList()
    {
        final Integer selectedYear = getSelectedYear();
        _forecastTimes.clear();
        _forecastTimes.addAll(generateForecastTimesList(selectedYear));
        _forecastTimesList.setListData(_forecastTimes.toArray());
    }

    /**
     * Calls {@link #updateChart(boolean)} passing in true to keep the zoom level.
     */
    protected final void updateChart()
    {
        updateChart(true);
    }

    /**
     * Call to update the chart being displayed.
     * 
     * @param keepDomainZoomLevel If true, then the domain axis limits will be remembered when the new chart is created.
     */
    protected final void updateChart(final boolean keepDomainZoomLevel)
    {
        try
        {
            //Build the chart engine and chart.
            final ChartEngine chartEngine = buildChart();
            final JFreeChart newChart = chartEngine.buildChart();

            //A panel already exists and it is a ChartEngineChartPanel, so update it without destroying it.
            //Using the existing panel is important for making the navigation tools work.
            if((_chartContainer.getComponentCount() > 0)
                && (_chartContainer.getComponent(0) instanceof ChartEngineChartAndTablePanel))
            {
                //Get the ChartEngineChartPanel and the OHDFixedChartPanel within it.
                final ChartEngineChartAndTablePanel innerPanel = getInnerPanel();
                final OHDFixedChartPanel chartPanel = innerPanel.getChartPanel();

                //Record the current domain range, if any, so that it can be recovered later.
                final Range range = chartPanel.getChart().getXYPlot().getDomainAxis().getRange();

                //Set the ChartEngine and chart within the inner panel.  At this point, newChart has its
                //auto axis limits.  By setting it now, the navigation panel will draw a new chart image 
                //using the auto range.
                innerPanel.setChartEngineAndChart(chartEngine, newChart, true);

                //Set the domain axis zoom if needed.
                if(keepDomainZoomLevel)
                {
                    ((XYPlot)newChart.getPlot()).getDomainAxis().setRange(range.getLowerBound(), range.getUpperBound());
                }
            }
            //Otherwise, we need to build a new one.
            else
            {
                //Create a new inner panel containing the built chart engine and chart.  The if clause occurs
                //only if an error occurs.
                final JPanel innerPanel = ChartPanelTools.createPanelForChartAndEngine(newChart,
                                                                                              chartEngine,
                                                                                              false,
                                                                                              buildTableModelSupplier());
                if(innerPanel instanceof ChartEngineChartAndTablePanel)
                {
                    final OHDFixedChartPanel chartPanel = ((ChartEngineChartAndTablePanel)innerPanel).getChartPanel();
                    chartPanel.setHorizontalZoom(true);
                }

                //Set the inner panel in its container.
                _chartContainer.removeAll();
                _chartContainer.add(innerPanel, BorderLayout.CENTER);

                //To avoid the squeeze/stretch JFreeChart does for large panels, set the maximum draw size to a large number.
                SwingTools.forceComponentRedraw(_chartContainer);

                //Construct the navigation panel.  Whether visible or not, it needs to be kept up to date.
                if(innerPanel instanceof ChartEngineChartAndTablePanel)
                {
                    final ChartEngineChartAndTablePanel panel = (ChartEngineChartAndTablePanel)innerPanel;

                    constructNavigationPanel(panel.getChartPanel());
                    panel.registerWithCentralBus(this);
                }
            }

        }
        catch(final Exception e)
        {
            e.printStackTrace();
            _chartContainer.removeAll();
            final JScrollPane badResultsScrollArea = new JScrollPane();
            final String headerMessage = "Error building chart:";
            badResultsScrollArea.setViewportView(HSwingFactory.createErrorMessagePane(headerMessage, e.getMessage()));
            _chartContainer.add(badResultsScrollArea, BorderLayout.CENTER);
        }
    }

    /**
     * Updates the navigation panel to display the provided chart given chart rendering info in the provided panel.
     * 
     * @param newChart
     * @param chartPanel
     * @param originalDomainRange If provided, then the chart may be currently zoomed in, so recover these original
     *            domain axis limits, first, before updating the nav panel, and then recover the current limits after.
     */
    private void constructNavigationPanel(final OHDFixedChartPanel chartPanel)
    {
        //Construct a new panel and put it in the holder.
        final CombinedDomainChartNavigationPanel navPanel = new CombinedDomainChartNavigationPanel(chartPanel, null);
        navPanel.setMaintainAspectRatio(false);
        _navPanelHolder.removeAll();
        _navPanelHolder.add(navPanel, BorderLayout.CENTER);
        RepaintManager.currentManager(_navPanelHolder).markCompletelyClean(_navPanelHolder);

        SwingTools.forceComponentRedraw(_navPanelHolder);
    }

    /**
     * Displays an error but placing an error message inside the {@link #_chartContainer}.
     * 
     * @param e Exception containing error message to display.
     */
    protected void error(final Exception e)
    {
        e.printStackTrace();
        _chartContainer.removeAll();
        final JScrollPane badResultsScrollArea = new JScrollPane();
        final String headerMessage = "Failed to build chart (see stack trace dumped to terminal):";
        badResultsScrollArea.setViewportView(HSwingFactory.createErrorMessagePane(headerMessage, e.getMessage()));
        _chartContainer.add(badResultsScrollArea, BorderLayout.CENTER);
    }

    /**
     * Upates the chart for a year spinner value change.
     */
    private void reactToYearChange()
    {
        _yearChoiceBox.setEnabled(!_showAllBox.isSelected());
        updateForecastTimesList();
        updateChart(false);
    }

    @Override
    public Component[] getComponentsForDiagnosticsToolBar()
    {
        if(allowsSouthNavigationPanel())
        {
            //Detach table label
            final ClickableJLabel showTableLabel = new ClickableJLabel(IconTools.getHSEBIcon("viewDataTable14x14"),
                                                                       new ShowTableAction());
            showTableLabel.setIconToUseWhenCursorIsNotOverLabel(IconTools.getHSEBIcon("translucentViewDataTable14x14"));
            showTableLabel.setToolTipText("Toggly Visibility of Data Table");
            showTableLabel.createTransparentMargin(2, 2, 2, 2);

            //Navpanel diagnostic by clicking on a label
            final ClickableJLabel navigateLabel = new ClickableJLabel(IconTools.getHSEBIcon("compass14x14"),
                                                                      new NavigateAction());
            navigateLabel.setIconToUseWhenCursorIsNotOverLabel(IconTools.getHSEBIcon("translucentCompass14x14"));
            navigateLabel.setToolTipText("Toggle Visibility of Navigation Panel");
            navigateLabel.createTransparentMargin(2, 2, 2, 2);

            return new Component[]{showTableLabel, navigateLabel};
        }
        return null;
    }

    /**
     * Action clears the diagnostic by firing a ClearDiagnosticEvent which is listened to above.
     * 
     * @author hank.herr
     */
    @SuppressWarnings("serial")
    private class NavigateAction extends AbstractAction
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            _collapsiblePane.setCollapsed(!_collapsiblePane.isCollapsed());
        }
    }

    /**
     * Action displays/undisplays the table by calling {@link ChartEngineChartAndTablePanel#setTableShowing(boolean)}
     * for the opposite of its current state. It only calls it if the chart container includes a
     * {@link ChartEngineChartAndTablePanel} as its component.
     * 
     * @author hankherr
     */
    @SuppressWarnings("serial")
    private class ShowTableAction extends AbstractAction
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            if((_chartContainer.getComponentCount() > 0)
                && (_chartContainer.getComponent(0) instanceof ChartEngineChartAndTablePanel))
            {
                final ChartEngineChartAndTablePanel panel = (ChartEngineChartAndTablePanel)_chartContainer.getComponent(0);
                panel.setTableShowing(!panel.isTableShowing());
            }
        }
    }

    @Override
    @Subscribe
    public void reactToTableCellControlClicked(final TableCellControlClickedNotice evt)
    {
        //Get the time series for the column and its forecast time.  Select the forecast time in the _forecastTimesList JList
        //to force a reaction of emphasizing the series.
        if((_forecastTimesList != null) && (getInnerPanel() != null))
        {
            final TimeSeriesArray tsa = ((TimeSeriesChartDiagnosticTableModel)getInnerPanel().getTableModel()).getTimeSeries(evt.getModelCol());
            if(tsa == null)
            {
                return; //Do nothing... indicates clicking shared domain column.
            }
            if((tsa.getHeader().getForecastTime() != Long.MIN_VALUE)
                && (tsa.getHeader().getForecastTime() != Long.MAX_VALUE))
            {
                //Setting the selected index in the _forecastTimesList needs to be placed on another thread.  The reason
                //is because an EventBus can only process one event at one time.  The selection below via a call to
                //setChartEngineAndChart will result in firing an event to refresh the nav panel packground image just
                //the domain zoom level is recovered (see the updateChart method above).  However, that event
                //cannot be processed so that the zoom level is reset for the domain BEFORE refreshing the background 
                //image in the nav panel.  Hence, the nav panel has the wrong limits.
                //
                //By doing the invokeLater below, the event bus is freed up and the forecast time selection is processed
                //on the standard Swing/AWT event thread.  Hence, the EventBus is able to properly handle the chart engine
                //and chart changing event (refreshing the nav panel) before the domain zoom is reset.
                SwingUtilities.invokeLater(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        final int index = _forecastTimes.indexOf(tsa.getHeader().getForecastTime());
                        if(_forecastTimesList.getSelectedIndex() == index)
                        {
                            _forecastTimesList.clearSelection();
                        }
                        else
                        {
                            _forecastTimesList.setSelectedIndex(index);
                            JListUtilities.scrollToRow(_forecastTimesList, index);
                        }
                    }
                });
            }
        }
    }
}
