package ohd.hseb.hefs.pe.tools;

import java.awt.Color;
import java.awt.Component;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.TimeSeriesArraysXYChartDataSource;
import ohd.hseb.charter.panel.DomainSharingTimeSeriesChartEngineTableModel;
import ohd.hseb.hefs.mefp.tools.QuestionableMessageMap;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.gui.jtable.GenericTableHeaderRenderer;
import ohd.hseb.hefs.utils.gui.jtable.MarkPanel;
import ohd.hseb.hefs.utils.gui.jtable.models.ColumnHeaderRendererTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.ToolTipTableModel;
import ohd.hseb.hefs.utils.gui.jtable.renderers.PredicateTableCellRenderer;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import org.jdesktop.swingx.renderer.JRendererLabel;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * Extension of {@link DomainSharingTimeSeriesChartEngineTableModel} that overrides
 * {@link DefaultChartEngineCellRenderer}, creating a {@link DiagnosticTableCellRenderer} that includes coloring for
 * whether time series are emphasized. See {@link DomainSharingTimeSeriesChartEngineTableModel} for information about
 * requirements of the data to display.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class TimeSeriesChartDiagnosticTableModel extends DomainSharingTimeSeriesChartEngineTableModel implements
ToolTipTableModel, ColumnHeaderRendererTableModel
{
    public static Color EMPHASIZED_COLOR = new Color(255, 140, 0);
    private QuestionableMessageMap _questionableMap = null;

    /**
     * Stores the questionable status of rows for each source.
     */
    private final HashMap<TimeSeriesArraysXYChartDataSource, boolean[]> _dataSourceToRowQuestionabilityMap = Maps.newHashMap();

    /**
     * Stores the questionable status of columns for each source.
     */
    private final HashMap<TimeSeriesArraysXYChartDataSource, boolean[]> _dataSourceToColumnQuestionabilityMap = Maps.newHashMap();

    @Override
    public TableCellRenderer wrapRenderer(final TableCellRenderer baseRenderer)
    {
        //Compute the number of columns for each color.
        int numberOfColumnsPerColor = 1;
        if(!areAllSeriesXaxisValuesSame())
        {
            numberOfColumnsPerColor = 2; //Each series is a pair of x,y coords.  There is never more than 2 columns per series.
        }

        return new DiagnosticTableCellRenderer(baseRenderer, numberOfColumnsPerColor);
    }

    /**
     * @return True if the time series corresponding to the column is emphasized.
     */
    private boolean isEmphasized(final int column)
    {
        final int seriesIndex = computeSeriesIndex(column);
        if(seriesIndex >= 0)
        {
            return TimeSeriesSorter.isEmphasized(getCurrentDataSource().getTimeSeries().get(seriesIndex));
        }
        return false;
    }

    /**
     * Renderer colors the row and column of emphasized series matching the emphasized coloring in the chart: 255 red,
     * 140 green, 0 blue (dark orange). This builds on top of {@link DefaultChartEngineCellRenderer}.
     * 
     * @author hankherr
     */
    protected class DiagnosticTableCellRenderer extends DefaultChartEngineCellRenderer
    {
        public DiagnosticTableCellRenderer(final TableCellRenderer baseRenderer, final int numberOfColumnsPerColor)
        {
            super(baseRenderer, numberOfColumnsPerColor);

            //Emphasized time series
            addEffect(new Predicate<Dyad<Integer, Integer>>()
            {
                public boolean apply(final Dyad<Integer, Integer> input)
                {
                    return isEmphasized(getTable().convertColumnIndexToModel(input.getSecond()));
                }
            }, PredicateTableCellRenderer.createBackgroundEffect(EMPHASIZED_COLOR, true), null);

            //Questionable data
            addEffect(new Predicate<Dyad<Integer, Integer>>()
            {
                public boolean apply(final Dyad<Integer, Integer> input) // Row, Column
                {
                    final long timeSeriesTime = ((Date)getRawValueAt(input.getFirst(), 0)).getTime();
                    final TimeSeriesArray tsArray = getTimeSeries(input.getSecond());

                    if(tsArray != null)
                    {
                        long forecastTime = tsArray.getHeader().getForecastTime();

                        final String paramId = tsArray.getHeader().getParameterId();

                        if(!HEFSTools.isForecastDataType(TimeSeriesSorter.getParameterIDStringWithoutEmphasized(paramId)))
                        {
                            forecastTime = QuestionableTools.NO_TIME;
                        }

                        if((_questionableMap != null)
                            && (_questionableMap.isAnyValueQuestionable(forecastTime, timeSeriesTime)))
                        {
                            return true;
                        }
                        else
                        {
                            return false;
                        }
                    }
                    else
                    {
                        return false;
                    }
                }
            },
                      PredicateTableCellRenderer.createBackgroundEffect(Color.RED, true),
                      null);

            //Questionable row -- Adds a (?) to the domain cell if all series x-axis values use the same domain.
            addEffect(new Predicate<Dyad<Integer, Integer>>()
            {
                public boolean apply(final Dyad<Integer, Integer> input) // Row, Column
                {
                    //If the column is the shared domain column
                    if(areAllSeriesXaxisValuesSame() && (input.getSecond() == 0)
                        && (!_dataSourceToRowQuestionabilityMap.isEmpty()))
                    {
                        final boolean[] questionableFlags = _dataSourceToRowQuestionabilityMap.get(getCurrentDataSource());
                        return questionableFlags[input.getFirst()];
                    }
                    return false;
                }
            },
                      new Effect<Component>()
                      {
                          @Override
                          public void perform(final Component input)
                          {
                              if(input instanceof JRendererLabel)
                              {
                                  ((JRendererLabel)input).setText(((JRendererLabel)input).getText() + " (?)");
                              }
                          }
                      },
                      null);
        }
    }

    @Override
    public String getColumnName(final int col)
    {
        if((col > 1) && (_questionableMap != null) && (_questionableMap.isAnyValueQuestionable(getTimeSeries(col))))
        {
            return super.getColumnName(col) + " (?)";
        }
        return super.getColumnName(col);
    }

    @Override
    public Dyad<Color, Integer> getRowMarkColor(final int modelRow)
    {
        final Dyad<Color, Integer> superClassReturn = super.getRowMarkColor(modelRow);
        Dyad<Color, Integer> thisReturn = null;

        final Date rowDate = getTimeForRow(modelRow, getCurrentDataSource());
        if(rowDate != null)
        {
            if(!_dataSourceToRowQuestionabilityMap.isEmpty()
                && _dataSourceToRowQuestionabilityMap.get(getCurrentDataSource())[modelRow])
            {
                thisReturn = new Dyad<Color, Integer>(Color.RED, MarkPanel.TIER_1_PRIORITY);
            }
        }

        if(thisReturn == null)
        {
            return superClassReturn;
        }
        if(superClassReturn == null)
        {
            return thisReturn;
        }

        if(thisReturn.getSecond() >= superClassReturn.getSecond())
        {
            return thisReturn;
        }

        return superClassReturn;
    }

    @Override
    public Dyad<Color, Integer> getColumnMarkColor(final int col)
    {
        final Dyad<Color, Integer> superClassReturn = super.getColumnMarkColor(col);

        Dyad<Color, Integer> thisReturn = null;

        if(isEmphasized(col))
        {
            thisReturn = new Dyad<Color, Integer>(EMPHASIZED_COLOR, MarkPanel.TIER_2_PRIORITY);
        }
        else if(col > 0)
        {
            if(!_dataSourceToColumnQuestionabilityMap.isEmpty()
                && _dataSourceToColumnQuestionabilityMap.get(getCurrentDataSource())[col - 1])
            {
                thisReturn = new Dyad<Color, Integer>(Color.RED, MarkPanel.TIER_1_PRIORITY);
            }
        }

        //Its easier for me to follow the logic by breaking the if into three parts...
        //The super class can generate a mark that has higher priorty than the emphasized mark; namely,
        //the selected mark.  This will return that selected mark first, then the emphasized mark, and then
        //the axis limits mark.
        if(thisReturn == null)
        {
            return superClassReturn;
        }
        if(superClassReturn == null)
        {
            return thisReturn;
        }
        if(thisReturn.getSecond() >= superClassReturn.getSecond())
        {
            return thisReturn;
        }
        return superClassReturn;
    }

    @Override
    public String getColumnHeaderToolTip(final int visibleColIndex)
    {
        return null;
    }

    @Override
    public String getCellToolTip(final int rowIndex, final int visibleColIndex)
    {
        final TimeSeriesArray tsArray = getTimeSeries(getTable().convertColumnIndexToModel(visibleColIndex));

        if((tsArray != null) && (_questionableMap != null))
        {
            long forecastTime = tsArray.getHeader().getForecastTime();

            if(HEFSTools.isObservedDataType(tsArray.getHeader().getParameterId()))
            {
                forecastTime = QuestionableTools.NO_TIME;
            }

            final long timeSeriesTime = ((Date)getRawValueAt(getTable().convertRowIndexToModel(rowIndex), 0)).getTime();
            String questionableString = "<HTML>";

            if(getRawValueAt(rowIndex, getTable().convertColumnIndexToModel(visibleColIndex)) instanceof Number)
            {
                final Number rawValue = (Number)getRawValueAt(rowIndex,
                                                              getTable().convertColumnIndexToModel(visibleColIndex));

                if(Double.isNaN(rawValue.doubleValue()))
                {
                    questionableString += "Missing Value<BR>";
                }
                else
                {
                    questionableString += rawValue + "<BR>";
                }
            }

            final List<String> messageList = _questionableMap.getMessagesForAllParameters(forecastTime, timeSeriesTime);

            if(!messageList.isEmpty())
            {
                questionableString += "QUESTIONABLE DATA:<BR>";
                for(final String message: messageList)
                {
                    questionableString += message + "<BR>";
                }

                questionableString += "</HTML>";
                return questionableString;
            }
            else
            {
                return null;
            }
        }
        else
        {
            return null;
        }
    }

    public QuestionableMessageMap getQuestionableMap()
    {
        return _questionableMap;
    }

    @Override
    public void setChartEngineWithoutFiringEvent(final ChartEngine engine)
    {
        super.setChartEngineWithoutFiringEvent(engine);
        populateQuestionabilityMaps();
    }

    private void populateQuestionabilityMaps()
    {
        if((_questionableMap == null) || (_questionableMap.isEmpty()))
        {
            return;
        }

        //For each data source...
        for(final XYChartDataSource source: getDataSources())
        {
            final TimeSeriesArraysXYChartDataSource tsSource = (TimeSeriesArraysXYChartDataSource)source;

            //Populate the row questionable map.
            final boolean[] rowQuestionable = new boolean[computeRowCount(tsSource)];
            for(int i = 0; i < rowQuestionable.length; i++)
            {
                final Date rowDate = getTimeForRow(i, tsSource);
                rowQuestionable[i] = _questionableMap.isAnyValueQuestionable(TimeSeriesArraysTools.convertTimeSeriesArraysToList(tsSource.getTimeSeries()),
                                                                             rowDate.getTime());
            }
            _dataSourceToRowQuestionabilityMap.put(tsSource, rowQuestionable);

            //Populate the column questionable map.
            final boolean[] colQuestionable = new boolean[tsSource.getTimeSeries().size()];
            for(int i = 0; i < colQuestionable.length; i++)
            {
                final TimeSeriesArray ts = tsSource.getTimeSeries().get(i);
                colQuestionable[i] = _questionableMap.isAnyForecastValueQuestionable(Lists.newArrayList(ts.getHeader()
                                                                                                          .getForecastTime()));
            }
            _dataSourceToColumnQuestionabilityMap.put(tsSource, colQuestionable);
        }
    }

    /**
     * After setting the {@link #_questionableMap}, it will populate the {@link #_dataSourceToColumnQuestionabilityMap}
     * and {@link #_dataSourceToRowQuestionabilityMap} maps. If the provided map is null or empty, the two data source
     * maps will be left as empty.
     * 
     * @param questionableMap
     */
    public void setQuestionableMap(final QuestionableMessageMap questionableMap)
    {
        _questionableMap = questionableMap;
        _dataSourceToColumnQuestionabilityMap.clear();
        _dataSourceToRowQuestionabilityMap.clear();
    }

    @Override
    public TableCellRenderer getColumnHeaderRenderer(final int modelColumn)
    {
        return new GenericTableHeaderRenderer()
        {

            @Override
            public Component getTableCellRendererComponent(final JTable table,
                                                           final Object value,
                                                           final boolean isSelected,
                                                           final boolean hasFocus,
                                                           final int row,
                                                           final int column)
            {
                final Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
                if(!_dataSourceToColumnQuestionabilityMap.isEmpty())
                {
                    final int modelColumn = getTable().convertColumnIndexToModel(column);
                    if((modelColumn > 0)
                        && (_dataSourceToColumnQuestionabilityMap.get(getCurrentDataSource())[modelColumn - 1]))
                    {
                        ((JLabel)c).setText(((JLabel)c).getText() + " (?)");
                        c.setBackground(Color.RED);
                    }
                }
                return c;
            }
        };
    }
}
