package ohd.hseb.hefs.pe.tools;

// import static com.google.common.collect.Lists.newArrayList;
// import static com.google.common.collect.Maps.newHashMap;
// import static ohd.hseb.hefs.utils.tools.SetTools.hashSetIntersection;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.graphgen.arguments.GraphGenArgumentsProcessor;
import ohd.hseb.graphgen.inputseries.TimeSeriesSelectionParameters;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.SetTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * Tool to organize a list of {@link TimeSeriesArray} objects primarily for diagnostic purposes. It provides an
 * assortment of tools to extract only observed and forecast time series, as well as emphasized observed and forecast
 * time series. Underlying the list is a {@link LinkedHashMap} ensuring that the series are sorted relative to,
 * first,the {@link LocationAndDataTypeIdentifier} of the time series in added order, and, second, by the order in which
 * the series for each identifier was added.<br>
 * <br>
 * The constants in this class must be kept in line with those in MEFPTools.java.
 * 
 * @author hank.herr
 */
// TODO grab constants from MEFPTools instead of hard-coding them.
public class TimeSeriesSorter implements Collection<TimeSeriesArray>
{
    public final static String EMPHASIZED_SUFFIX = " Emphasized";
    public static final String[] OBSERVED_PARAMETERS = {"MAP", "MAT", "TMAX", "TMIN"};
    public static final String[] FORECAST_PARAMETERS = {"FMAP", "FMAT", "TFMX", "TFMN"};
    public static final String[] EMPHASIZED_OBSERVED_PARAMETERS = {"MAP" + EMPHASIZED_SUFFIX,
        "MAT" + EMPHASIZED_SUFFIX, "TMAX" + EMPHASIZED_SUFFIX, "TMIN" + EMPHASIZED_SUFFIX};
    public static final String[] EMPHASIZED_FORECAST_PARAMETERS = {"FMAP" + EMPHASIZED_SUFFIX,
        "FMAT" + EMPHASIZED_SUFFIX, "TFMX" + EMPHASIZED_SUFFIX, "TFMN" + EMPHASIZED_SUFFIX};

    public static final List<String> ALL_PARAMETERS;
    static
    {
        final List<String> params = new ArrayList<String>();
        Collections.addAll(params, OBSERVED_PARAMETERS);
        Collections.addAll(params, FORECAST_PARAMETERS);
        Collections.addAll(params, EMPHASIZED_OBSERVED_PARAMETERS);
        Collections.addAll(params, EMPHASIZED_FORECAST_PARAMETERS);
        ALL_PARAMETERS = Collections.unmodifiableList(params);
    }

    public static final List<List<String>> PARAMETER_OF_PAIRS; // O.F., not of
    static
    {
        final List<List<String>> params = new ArrayList<List<String>>();
        params.add(Lists.newArrayList("MAP", "FMAP"));
        params.add(Lists.newArrayList("MAT", "FMAT"));
        params.add(Lists.newArrayList("TMAX", "TFMX"));
        params.add(Lists.newArrayList("TMIN", "TFMN"));
        params.add(Lists.newArrayList("MAP" + EMPHASIZED_SUFFIX, "FMAP" + EMPHASIZED_SUFFIX));
        params.add(Lists.newArrayList("MAT" + EMPHASIZED_SUFFIX, "FMAT" + EMPHASIZED_SUFFIX));
        params.add(Lists.newArrayList("TMAX" + EMPHASIZED_SUFFIX, "TFMX" + EMPHASIZED_SUFFIX));
        params.add(Lists.newArrayList("TMIN" + EMPHASIZED_SUFFIX, "TFMN" + EMPHASIZED_SUFFIX));
        PARAMETER_OF_PAIRS = Collections.unmodifiableList(params);
    }

    /**
     * Map identifiers to the respective time series.
     */
    private final LinkedHashMap<LocationAndDataTypeIdentifier, List<TimeSeriesArray>> _identifierToTimeSeriesMap;

    /**
     * Set of allowed parameters. If null, allow all.
     */
    private final Set<String> _allowedParameters;

    /**
     * Set of allowed locations. If null, allow all.
     */
    private final Set<String> _allowedLocations;

    public TimeSeriesSorter()
    {
        _identifierToTimeSeriesMap = Maps.newLinkedHashMap();
        _allowedParameters = null;
        _allowedLocations = null;
    }

    public TimeSeriesSorter(final Collection<? extends TimeSeriesArray> coll)
    {
        this();
        this.addAll(coll);
    }

    /**
     * Creates a view of another sorter, narrowing the allowed parameters and locations.
     * 
     * @param other the sorter to view
     * @param params collection of allowed parameters - may be null
     * @param locations collection of allowed locations - may be null
     */
    @SuppressWarnings("unchecked")
    private TimeSeriesSorter(final TimeSeriesSorter other,
                             final Collection<String> params,
                             final Collection<String> locations)
    {
        _identifierToTimeSeriesMap = Maps.newLinkedHashMap();

        // Allowed Parameters
        if(params == null)
        {
            _allowedParameters = other._allowedParameters;
        }
        else if(other._allowedParameters == null)
        {
            _allowedParameters = new HashSet<String>(params);
        }
        else
        {
            _allowedParameters = SetTools.hashSetIntersection(params, other._allowedParameters);
        }

        // Allowed Locations
        if(locations == null)
        {
            _allowedLocations = other._allowedLocations;
        }
        else if(other._allowedLocations == null)
        {
            _allowedLocations = new HashSet<String>(locations);
        }
        else
        {
            _allowedLocations = SetTools.hashSetIntersection(locations, other._allowedLocations);
        }

        // Add lists
        for(final Map.Entry<LocationAndDataTypeIdentifier, List<TimeSeriesArray>> entry: other._identifierToTimeSeriesMap.entrySet())
        {
            final LocationAndDataTypeIdentifier identifier = entry.getKey();
            if((_allowedParameters == null || _allowedParameters.contains(identifier.getParameterId()))
                && (_allowedLocations == null || _allowedLocations.contains(identifier.getLocationId())))
            {
                _identifierToTimeSeriesMap.put(identifier, entry.getValue());
            }
        }
    }

    /**
     * @return a list of all time series in this sorter sorted by forecast time.
     */
    public List<TimeSeriesArray> toList()
    {
        //There is no guarantee that the map will return values in order, apparently, after emphasizing and 
        //deemphasizing.
        return TimeSeriesArraysTools.sortListByForecastTime(ListTools.flatten(_identifierToTimeSeriesMap.values()));
    }

    /**
     * Returns all contained series sorted by identifier. In reality just returns an unmodifiable view of the sorter's
     * underlying map.
     * 
     * @return
     */
    public Map<LocationAndDataTypeIdentifier, List<TimeSeriesArray>> sortedByIdentifier()
    {
        return Collections.unmodifiableMap(_identifierToTimeSeriesMap);
    }

    /**
     * Retrieves the only time series array in this sorter.
     * 
     * @return the only array in this sorter
     * @throws IllegalStateException if this sorter does not have exactly one series
     */
    public TimeSeriesArray getOnly() throws IllegalStateException
    {
        final Iterator<TimeSeriesArray> iter = this.iterator();
        final TimeSeriesArray only = iter.next();
        if(iter.hasNext() || only == null)
        {
            throw new IllegalStateException("This sorter is supposed to have exactly one series.");
        }
        return only;
    }

    /**
     * Returns a view that only allows the specified parameters.
     * 
     * @param parameters the parameters to allow.
     */
    public TimeSeriesSorter restrictViewToParameters(final String... parameters)
    {
        return this.restrictViewToParameters(Lists.newArrayList(parameters));
    }

    /**
     * Returns a view that only allows the specified parameters.
     * 
     * @param parameters the {@link ParameterId} instances to allow.
     */
    public TimeSeriesSorter restrictViewToParameters(final ParameterId... parameters)
    {
        final List<String> paramStrs = Lists.newArrayList();
        for(final ParameterId id: parameters)
        {
            paramStrs.add(id.name());
        }
        return this.restrictViewToParameters(paramStrs);
    }

    /**
     * Returns a view that only allows the specified parameters.
     * 
     * @param parameters the parameters to allow.
     * @return
     */
    public TimeSeriesSorter restrictViewToParameters(final Collection<String> parameters)
    {
        return new TimeSeriesSorter(this, parameters, null);
    }

    /**
     * @return {@link TimeSeriesSorter} that includes the subset of the time series in this that match the provided
     *         selection parameters.
     */
    public TimeSeriesSorter extractMatchingTimeSeries(final TimeSeriesSelectionParameters parms)
    {
        final TimeSeriesSorter results = new TimeSeriesSorter();
        for(final TimeSeriesArray ts: this)
        {
            if(parms.doesTimeSeriesMatchParameters(GraphGenArgumentsProcessor.createEmpty(), ts))
            {
                results.add(ts);
            }
        }
        return results;
    }

    /**
     * Returns a view that only allows the specified locations.
     * 
     * @param locations the locations to allow.
     * @return
     */
    public TimeSeriesSorter restrictViewToLocations(final Collection<String> locations)
    {
        return new TimeSeriesSorter(this, null, locations);
    }

    /**
     * Returns a view that only allows series without forecast times.
     * 
     * @return
     */
    public TimeSeriesSorter restrictViewToObserved()
    {
        return new TimeSeriesSorter(this, ListTools.concat(OBSERVED_PARAMETERS, EMPHASIZED_OBSERVED_PARAMETERS), null);
    }

    /**
     * Returns a view that only allows forecast series.
     * 
     * @return
     */
    public TimeSeriesSorter restrictViewToForecast()
    {
        return new TimeSeriesSorter(this, ListTools.concat(FORECAST_PARAMETERS, EMPHASIZED_FORECAST_PARAMETERS), null);
    }

    /**
     * Returns a view that only allows unemphasized series.
     * 
     * @return
     */
    public TimeSeriesSorter restrictViewToUnemphasized()
    {
        return new TimeSeriesSorter(this, ListTools.concat(OBSERVED_PARAMETERS, FORECAST_PARAMETERS), null);
    }

    /**
     * Returns a view that only allows emphasized series.
     * 
     * @return
     */
    public TimeSeriesSorter restrictViewToEmphasized()
    {
        return new TimeSeriesSorter(this, ListTools.concat(EMPHASIZED_OBSERVED_PARAMETERS,
                                                           EMPHASIZED_FORECAST_PARAMETERS), null);
    }

    /**
     * Returns a copy that only contains forecast series in the given year.
     * 
     * @param year the year to restrict forecasts to, or null to get ones with no forecast date
     * @return
     */
    public TimeSeriesSorter forecastInYear(final Integer year)
    {
        final TimeSeriesSorter sorter = new TimeSeriesSorter(new TimeSeriesSorter(), _allowedParameters, null).restrictViewToForecast();
        for(final Map.Entry<LocationAndDataTypeIdentifier, List<TimeSeriesArray>> entry: this.restrictViewToForecast()._identifierToTimeSeriesMap.entrySet())
        {
            sorter.addAll(TimeSeriesArrayTools.filterByForecastYear(entry.getValue(), year));
        }
        return sorter;
    }

    /**
     * Returns a list of views split by observed/forecast pairs and location for the varying data types. i.e. the first
     * might be a view of MAP/FMAP, the second TMAX/FTMX, etc.
     * 
     * @return
     */
    public List<TimeSeriesSorter> splitByOFPairsAndLocation()
    {
        final List<TimeSeriesSorter> list = Lists.newArrayList();
        for(final List<String> params: PARAMETER_OF_PAIRS)
        {
            list.addAll(this.restrictViewToParameters(params).splitByLocation());
        }
        return list;
    }

    /**
     * Returns a list of views, each viewing a single location.
     * 
     * @return
     */
    public List<TimeSeriesSorter> splitByLocation()
    {
        final Set<String> locations = Sets.newHashSet();
        for(final LocationAndDataTypeIdentifier identifier: _identifierToTimeSeriesMap.keySet())
        {
            locations.add(identifier.getLocationId());
        }

        final List<TimeSeriesSorter> list = Lists.newArrayList();
        for(final String location: locations)
        {
            list.add(this.restrictViewToLocations(Lists.newArrayList(location)));
        }
        return list;
    }

    /**
     * Sets the dates to be emphasized.
     * 
     * @param dates the collection of dates to be emphasized
     */
    public void setEmphasizedDates(final Collection<Long> dates)
    {
        // Grab all dates.
        final List<TimeSeriesArray> seriesList = this.toList();

        // Clear current dates.
        this.clear();

        // (De)Emphasize dates.
        for(final TimeSeriesArray tsa: seriesList)
        {
            if(dates.contains(tsa.getHeader().getForecastTime()))
            {
                emphasize(tsa);
            }
            else
            {
                deemphasize(tsa);
            }
        }

        // Re-add to map.
        this.addAll(seriesList);
    }

    public void clearEmphasis()
    {
        setEmphasizedDates(Lists.<Long>newArrayList());
    }

    public TimeSeriesArrays toTimeSeriesArrays()
    {
        if(isEmpty())
        {
            return null;
        }

        final List<TimeSeriesArray> ts = this.toList();
        final Iterator<TimeSeriesArray> iter = ts.iterator();
        final TimeSeriesArrays arrays = new TimeSeriesArrays(iter.next());
        while(iter.hasNext())
        {
            arrays.add(iter.next());
        }
        return arrays;
    }

    @Override
    public String toString()
    {
        String s = "TimeSeriesSorter [\n";
        for(final TimeSeriesArray tsa: this)
        {
            s += tsa + "\n";
        }
        return s + "]\n";
    }

    /**
     * @param ts Time series to check.
     * @return True if the the time series is emphasized, following how the method {@link #emphasize(TimeSeriesArray)}
     *         emphasizes it (i.e., adds the {@link #EMPHASIZED_SUFFIX} to the parameter id of the time series header).
     */
    public static boolean isEmphasized(final TimeSeriesArray ts)
    {
        return ts.getHeader().getParameterId().endsWith(EMPHASIZED_SUFFIX);
    }

    /**
     * Emphasizes the given time series by adding the {@link #EMPHASIZED_SUFFIX} to the parameter id.
     * 
     * @param tsa series to be emphasized
     */
    public static void emphasize(final TimeSeriesArray tsa)
    {
        final DefaultTimeSeriesHeader header = (DefaultTimeSeriesHeader)tsa.getHeader();
        if(!header.getParameterId().endsWith(EMPHASIZED_SUFFIX))
        {
            header.setParameterId(header.getParameterId() + EMPHASIZED_SUFFIX);
        }
    }

    /**
     * Deemphasizes the time series, removing the {@link #EMPHASIZED_SUFFIX} suffix from the parameter id.
     * 
     * @param tsa Time series to deemphasize.
     */
    public static void deemphasize(final TimeSeriesArray tsa)
    {
        final DefaultTimeSeriesHeader header = (DefaultTimeSeriesHeader)tsa.getHeader();
        header.setParameterId(header.getParameterId().replaceFirst(EMPHASIZED_SUFFIX, ""));
    }

    /**
     * Returns an iterator that is still viable after this list has been modified.
     */
    public Iterator<TimeSeriesArray> concurrentIterator()
    {
        return this.toList().iterator();
    }

    /**
     * Returns the list which the given series should be put into. Returns a null if the given series is not allowed in
     * this sorter.
     * 
     * @param tsa time series to place
     * @param addIfNeeded If true, then if a list is not found, it will be added.
     * @return where the series should go, or null if it is not allowed
     */
    private List<TimeSeriesArray> pickList(final TimeSeriesArray tsa, final boolean addIfNeeded)
    {
        // Check for legal parameter.
        final String parameter = tsa.getHeader().getParameterId();
        if(_allowedParameters != null && !_allowedParameters.contains(parameter))
        {
            return null;
        }

        // Check for legal location.
        final String location = tsa.getHeader().getLocationId();
        if(_allowedLocations != null && !_allowedLocations.contains(location))
        {
            return null;
        }

        final LocationAndDataTypeIdentifier identifier = LocationAndDataTypeIdentifier.get(location, parameter);
        List<TimeSeriesArray> list = _identifierToTimeSeriesMap.get(identifier);
        if((list == null) && addIfNeeded)
        {
            list = Lists.newArrayList();
            _identifierToTimeSeriesMap.put(identifier, list);
        }
        return list;
    }

    /**
     * Just tests for containing the same series.
     */
    @Override
    public boolean equals(final Object other)
    {
        if(other instanceof Collection<?>)
        {
            final Collection<?> o1 = new ArrayList<Object>((Collection<?>)other);
            final Collection<TimeSeriesArray> o2 = this.toList();
            o1.removeAll(this);
            o2.removeAll((Collection<?>)other);
            return o1.isEmpty() && o2.isEmpty();
        }
        return false;
    }

    //////////////////////////////////////
    /////////////// Collection Interface

    @Override
    public boolean add(final TimeSeriesArray tsa)
    {
        final List<TimeSeriesArray> list = pickList(tsa, true);
        if(list != null)
        {
            list.add(tsa);
            return true;
        }
        else
        {
            throw new IllegalArgumentException("Illegal location or parameter.");
        }
    }

    @Override
    public boolean addAll(final Collection<? extends TimeSeriesArray> coll)
    {
        for(final TimeSeriesArray tsa: coll)
        {
            add(tsa);
        }
        return !coll.isEmpty();
    }

    public boolean addAllNotPresent(final Collection<? extends TimeSeriesArray> coll)
    {
        for(final TimeSeriesArray tsa: coll)
        {
            if(!contains(tsa))
            {
                add(tsa);
            }
        }
        return !coll.isEmpty();
    }

    @Override
    public void clear()
    {
        _identifierToTimeSeriesMap.clear();
//This used to only clear the lists and I'm not sure why:
//        for(final List<TimeSeriesArray> list: _identifierToTimeSeriesMap.values())
//        {
//            list.clear();
//        }
    }

    @Override
    public boolean contains(final Object o)
    {
        if(!(o instanceof TimeSeriesArray))
        {
            return false;
        }
        final TimeSeriesArray tsa = (TimeSeriesArray)o;
        final List<TimeSeriesArray> list = pickList(tsa, false);
        if(list == null)
        {
            return false;
        }
        else
        {
            return list.contains(tsa);
        }
    }

    @Override
    public boolean containsAll(final Collection<?> coll)
    {
        for(final Object o: coll)
        {
            if(!this.contains(o))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * @return True if a time series with that forecast time already exists for the time series provided. It uses
     *         {@link #pickList(TimeSeriesArray)} to identify which time series to look at.
     */
    public boolean containsTimeSeriesWithSameForecastTime(final TimeSeriesArray tsa)
    {
        final List<TimeSeriesArray> list = pickList(tsa, false);
        if(list == null)
        {
            return false;
        }
        return TimeSeriesArraysTools.searchByForecastTime(list, tsa.getHeader().getForecastTime()) >= 0;
    }

    @Override
    public boolean isEmpty()
    {
        for(final List<TimeSeriesArray> list: _identifierToTimeSeriesMap.values())
        {
            if(!list.isEmpty())
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public Iterator<TimeSeriesArray> iterator()
    {
        return new TimeSeriesSorterIterator(this);
    }

    @Override
    public boolean remove(final Object o)
    {
        if(!(o instanceof TimeSeriesArray))
        {
            return false;
        }

        final TimeSeriesArray tsa = (TimeSeriesArray)o;
        final List<TimeSeriesArray> list = pickList(tsa, false);
        if(list == null)
        {
            return false;
        }

        return list.remove(tsa);
    }

    @Override
    public boolean removeAll(final Collection<?> coll)
    {
        boolean changed = false;
        for(final Object o: coll)
        {
            changed = changed || this.remove(o);
        }
        return changed;
    }

    @Override
    public boolean retainAll(final Collection<?> coll)
    {
        boolean changed = false;
        for(final List<TimeSeriesArray> list: _identifierToTimeSeriesMap.values())
        {
            changed = changed || list.retainAll(coll);
        }
        return changed;
    }

    @Override
    public int size()
    {
        int size = 0;
        for(final List<TimeSeriesArray> list: _identifierToTimeSeriesMap.values())
        {
            size += list.size();
        }
        return size;
    }

    @Override
    public Object[] toArray()
    {
        return this.toList().toArray();
    }

    @Override
    public <T> T[] toArray(final T[] a)
    {
        return this.toList().toArray(a);
    }

    /**
     * Iterates over the series, ordered by {@link LocationAndDataTypeIdentifier}.
     * 
     * @author alexander.garbarino
     */
    private class TimeSeriesSorterIterator implements Iterator<TimeSeriesArray>
    {
        private final TimeSeriesSorter _sorter;
        private final List<LocationAndDataTypeIdentifier> _keys;
        private int _nextKeyIndex = 0;
        private List<TimeSeriesArray> _list = null;
        private int _nextListIndex = 0;
        private TimeSeriesArray _next = null;
        private boolean _hasNext = true;

        private TimeSeriesSorterIterator(final TimeSeriesSorter sorter)
        {
            _sorter = sorter;
            _keys = new ArrayList<LocationAndDataTypeIdentifier>(_sorter._identifierToTimeSeriesMap.keySet());
//            Collections.sort(_keys); -- removed the sorting, because I'd rather it use the order of the LinkedHashMap underlying TimeSeriesSorter.
            loadNext();
        }

        private void loadNext()
        {
            while(_list == null || _nextListIndex >= _list.size())
            {
                if(_nextKeyIndex == _keys.size())
                {
                    _next = null;
                    _hasNext = false;
                    return;
                }
                _list = _sorter._identifierToTimeSeriesMap.get(_keys.get(_nextKeyIndex));
                _nextListIndex = 0;
                _nextKeyIndex++;
            }
            _next = _list.get(_nextListIndex);
            _nextListIndex++;
        }

        @Override
        public boolean hasNext()
        {
            return _hasNext;
        }

        @Override
        public TimeSeriesArray next()
        {
            final TimeSeriesArray tsa = _next;
            loadNext();
            return tsa;
        }

        @Override
        public void remove()
        {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * @param timeSeries
     * @param dataType
     * @return All time series for which the header specifies a parameter matching dataType. The method
     *         {@link ParameterId#of(nl.wldelft.util.timeseries.TimeSeriesHeader)} is used.
     */
    public static List<TimeSeriesArray> subListByParameterId(final Collection<TimeSeriesArray> timeSeries,
                                                             final ParameterId dataType)
    {
        final List<TimeSeriesArray> subTS = Lists.newArrayList();
        for(final TimeSeriesArray ts: timeSeries)
        {
            if(ParameterId.of(ts.getHeader()).equals(dataType))
            {
                subTS.add(ts);
            }
        }
        return subTS;
    }

    /**
     * @param paramId The parameter ID to be "trimmed"
     * @return the Parameter ID minus the EMPHASIZED_SUFFIX
     */
    public static String getParameterIDStringWithoutEmphasized(final String paramId)
    {
        return paramId.replace(EMPHASIZED_SUFFIX, "").trim();
    }

}
