package ohd.hseb.hefs.utils;

import java.util.Iterator;

import com.google.common.base.Predicate;

/**
 * {@link Predicate} that requires two inputs to generate an output boolean. The two inputs can take the form of a
 * {@link Dyad} or be independently passed into apply.
 * 
 * @author Hank.Herr
 * @param <A> First input type.
 * @param <B> Second input type.
 */
public abstract class BinaryPredicate<A, B> implements Predicate<Dyad<A, B>>
{
    @Override
    public boolean apply(Dyad<A, B> input)
    {
        return apply(input.getFirst(), input.getSecond());
    }

    /**
     * Must return a result given the two inputs.
     * 
     * @param inputA
     * @param inputB
     * @return Boolean.
     */
    public abstract boolean apply(A inputA, B inputB);

    /**
     * Tests the predicate against two iterables of arguments, for all pairs being true.<br>
     * If the two iterables differ in length, this does not consider the extra elements of the longer one.
     * 
     * @param predicate the predicate to test
     * @param as the list of left arguments
     * @param bs the list of right arguments
     * @return True if all paired elements in the {@link Iterable} arguments return true when
     *         {@link #apply(Object, Object)} is called.
     */
    public boolean areAllTrue(Iterable<? extends A> as, Iterable<? extends B> bs)
    {
        Iterator<? extends A> aIter = as.iterator();
        Iterator<? extends B> bIter = bs.iterator();

        while(aIter.hasNext() && bIter.hasNext())
        {
            if(!apply(aIter.next(), bIter.next()))
            {
                return false;
            }
        }
        return true;
    }
}
