package ohd.hseb.hefs.utils;

import static java.util.concurrent.TimeUnit.DAYS;
import static java.util.concurrent.TimeUnit.HOURS;
import static java.util.concurrent.TimeUnit.MICROSECONDS;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.concurrent.TimeUnit.NANOSECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import java.util.concurrent.TimeUnit;

import ohd.hseb.hefs.utils.tools.EnumTools;

import com.google.common.base.Objects;

/**
 * A duration of time. Note that upward conversions (e.g. minutes to hours) will lose information, as no fractional
 * units are kept.
 * 
 * @author alexander.garbarino
 */
public class Duration implements Comparable<Duration>
{
    private final long _amount;
    private final TimeUnit _unit;

    public Duration(long amount, TimeUnit unit)
    {
        _amount = amount;
        _unit = unit;
    }

    public Duration toDays()
    {
        return new Duration(_unit.toDays(_amount), DAYS);
    }

    public Duration toHours()
    {
        return new Duration(_unit.toHours(_amount), HOURS);
    }

    public Duration toMinutes()
    {
        return new Duration(_unit.toMinutes(_amount), MINUTES);
    }

    public Duration toSeconds()
    {
        return new Duration(_unit.toSeconds(_amount), SECONDS);
    }

    public Duration toMillis()
    {
        return new Duration(_unit.toMillis(_amount), MILLISECONDS);
    }

    public Duration toMicros()
    {
        return new Duration(_unit.toMicros(_amount), MICROSECONDS);
    }

    public Duration toNanos()
    {
        return new Duration(_unit.toNanos(_amount), NANOSECONDS);
    }

    public long getAmount()
    {
        return _amount;
    }

    @Override
    public String toString()
    {
        return String.format("%d %s", _amount, _unit.name().toLowerCase());
    }

    @Override
    public boolean equals(Object other)
    {
        if(this == other)
        {
            return true;
        }

        if(!(other instanceof Duration))
        {
            return false;
        }

        return compareTo((Duration)other) == 0;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(_amount, _unit);
    }

    @Override
    public int compareTo(Duration that)
    {
        TimeUnit minUnit = EnumTools.min(this._unit, that._unit);
        Long thisAmount = minUnit.convert(this._amount, this._unit);
        Long thatAmount = minUnit.convert(that._amount, that._unit);
        return thisAmount.compareTo(thatAmount);
    }
}
