package ohd.hseb.hefs.utils;

import java.io.File;
import java.io.IOException;

import com.google.common.io.Files;

/**
 * Manages a temporary {@link File} that is connected to a target {@link File}.
 * 
 * @author Hank.Herr
 */
@SuppressWarnings("serial")
public class TempFile extends File
{
    private final File _target;

    /**
     * @param file The file to create a temp version of with the same name but ".tmp" added.
     * @return The TempFile to where a temp file can be written.
     * @throws IOException
     */
    public static TempFile make(final File file) throws IOException
    {
        final File temp = File.createTempFile(TempFile.class.getCanonicalName() + "_" + file.hashCode(), ".tmp");
        return new TempFile(temp, file);
    }

    /**
     * @param tempFile The temp file.
     * @param file The target file.
     * @throws IOException
     */
    private TempFile(final File tempFile, final File file) throws IOException
    {
        super(tempFile.getAbsolutePath());
        _target = file;
    }

    /**
     * Overwrites the target file if the temp file is not identical to it.
     * 
     * @return True if the file was overwritten.
     * @throws IOException
     */
    public boolean overwriteIfDifferent() throws IOException
    {
        if(!exists())
        {
            throw new IOException("Temp file does not exist.");
        }
        else if(_target.exists() && Files.equal(this, _target))
        {
            return false;
        }
        else
        {
            overwrite();
            return true;
        }
    }

    public void overwrite() throws IOException
    {
        Files.copy(this, _target);
    }

    public void move() throws IOException
    {
        Files.move(this, _target);
    }

    public File getTarget()
    {
        return _target;
    }
}
