package ohd.hseb.hefs.utils;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import ohd.hseb.hefs.utils.tools.StreamTools;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;

/**
 * Modifies zip files by copying them to another location.<br>
 * 
 * @author alexander.garbarino
 */
public class ZipTransformer
{
    private final List<Function<ZipEntry, ZipEntry>> _entryTransformers;
    private final List<Function<InputStream, InputStream>> _dataTransformers;

    public ZipTransformer()
    {
        _entryTransformers = Lists.newArrayList();
        _dataTransformers = Lists.newArrayList();
    }

    public void addEntryTransformer(final Function<ZipEntry, ZipEntry> transformer)
    {
        _entryTransformers.add(transformer);
    }

    public void addDataTransformer(final Function<InputStream, InputStream> transformer)
    {
        _dataTransformers.add(transformer);
    }

    /**
     * Creates a transformer that sets the date of entry to the specified time.
     * 
     * @param time the new time
     * @return time transformer
     */
    public static Function<ZipEntry, ZipEntry> getSetTimeTransformer(final long time)
    {
        return new Function<ZipEntry, ZipEntry>()
        {
            @Override
            public ZipEntry apply(final ZipEntry input)
            {
                input.setTime(time);
                return input;
            }
        };
    }

    public static Function<ZipEntry, ZipEntry> getSkipFileTransformer(final File file)
    {
        return new Function<ZipEntry, ZipEntry>()
        {
            @Override
            public ZipEntry apply(final ZipEntry input)
            {
                if(input.getName().equals(file.getName()))
                {
                    return null;
                }
                else
                {
                    return input;
                }
            }
        };
    }

    public void transform(final File file) throws IOException
    {
        final File tmpFile = File.createTempFile("ziptrans", ".zip");
        transform(file, tmpFile);
        Files.move(tmpFile, file);
    }

    /**
     * Copies zip file <code>inFile</code> to <code>outFile</code>, performing any specified transformations in the
     * process.
     * 
     * @param inFile the in file
     * @param outFile the out file
     */
    public void transform(final File inFile, final File outFile) throws IOException
    {
        final ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(outFile)));

        ZipFile zip = null;
        try
        {
            zip = new ZipFile(inFile);
            final Enumeration<? extends ZipEntry> entries = zip.entries();
            while(entries.hasMoreElements())
            {
                ZipEntry entry = entries.nextElement();
                InputStream in = zip.getInputStream(entry);

                for(final Function<ZipEntry, ZipEntry> function: _entryTransformers)
                {
                    entry = function.apply(entry);
                    if(entry == null)
                    {
                        continue;
                    }
                }

                for(final Function<InputStream, InputStream> function: _dataTransformers)
                {
                    in = function.apply(in);
                }

                out.putNextEntry(entry);
                ByteStreams.copy(in, out);
                out.closeEntry();
            }
        }
        finally
        {
            StreamTools.closeStream(zip);
            out.close();
        }
    }

}
