package ohd.hseb.hefs.utils;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import com.google.common.io.ByteStreams;

/**
 * Convenience class for writing files to a zip.<br>
 * Usage Example:<br>
 * <code>ZipWriter writer = new ZipWriter(zipFile);<br>
 * writer.write(file1);<br>
 * writer.write(file2);<br>
 * writer.close();</code>
 * 
 * @author alexander.garbarino
 */
public class ZipWriter
{
    private final File _zipFile;
    private final ZipOutputStream _out;

    public ZipWriter(File zipFile) throws FileNotFoundException
    {
        _zipFile = zipFile;
        _out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(_zipFile)));
    }

    /**
     * Writes <code>file</code> to this zip file.
     * 
     * @param file the file to write
     * @throws IOException
     */
    public void writeFile(File file) throws IOException
    {
        writeFile(file, false);
    }

    /**
     * Writes <code>file</code> to this zip file. Can specify whether to use the full path within the zip file, or just
     * the filename itself.
     * 
     * @param file the file to write
     * @param useFullPath If <code>true</code>, use the full path of <code>file</code> in the resulting zip file.<br/>
     *            If <code>false</code>, just use the local filename.
     * @throws IOException
     */
    public void writeFile(File file, boolean useFullPath) throws IOException
    {
        File path;
        if(useFullPath)
        {
            path = file;
        }
        else
        {
            path = new File(file.getName());
        }
        this.writeFile(file, path);
    }

    /**
     * Writes the contents of <code>sourceFile</code> to this zip file under the filename <code>zipFile</code>.
     * 
     * @param sourceFile the file whose contents this will write
     * @param zipFile the filename to write in the zip file as
     * @throws IOException
     */
    public void writeFile(File sourceFile, File zipFile) throws IOException
    {
        writeFile(sourceFile, zipFile, null);
    }

    /**
     * Writes the contents of <code>sourceFile</code> to this zip file under the filename <code>zipFile</code>, with the
     * timestamp <code>time</code>.
     * 
     * @param sourceFile the file whose contents this will write
     * @param zipFile the filename to write in the zip file as
     * @param time the timestamp to use, or null for the current time
     * @throws IOException
     */
    public void writeFile(File sourceFile, File zipFile, Long time) throws IOException
    {
        ZipEntry entry = new ZipEntry(zipFile.getPath());
        if(time != null)
        {
            entry.setTime(time);
        }
        _out.putNextEntry(entry);
        InputStream in = new FileInputStream(sourceFile);
        ByteStreams.copy(in, _out);
        in.close();
    }

    public void close() throws IOException
    {
        _out.close();
    }

}
