package ohd.hseb.hefs.utils.adapter;

import java.io.File;

import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;

/**
 * Uses an {@link XMLFile} to store a property that specifies a file. Flags allow for this to check for file
 * readability, writability, and create it if missing. Those checks are performed with
 * {@link #processAfterSuccessfulRead()}, which is called after ALL properties are read in without error.
 * 
 * @author hankherr
 */
public class DirectoryPropertyVariable extends SimplePropertyVariable<File>
{
    private final boolean _checkForWritability;
    private final boolean _checkForReadability;
    private final boolean _createIfMissing;

    /**
     * @param directory An {@link XMLFile}; its check for readability flag should be set to false, since it is redundant
     *            with the checks in here. If it is true, then checking will be done immediately upon reading. That is
     *            allowable, but in this class checking is done after all properties are processed, which may be
     *            preferable in some cases.
     * @param required
     * @param checkForWritability
     * @param checkForReadability
     * @param createIfMissing
     */
    public DirectoryPropertyVariable(final XMLFile directory,
                                     final boolean required,
                                     final boolean checkForWritability,
                                     final boolean checkForReadability,
                                     final boolean createIfMissing)
    {
        super(directory, required);
        _checkForWritability = checkForWritability;
        _checkForReadability = checkForReadability;
        _createIfMissing = createIfMissing;
    }

    @Override
    public void processAfterSuccessfulRead() throws Exception
    {
        final File dir = this.getValue();

        //Create the directory if missing.
        if(_createIfMissing)
        {
            try
            {
                FileTools.mkdirIfItDoesNotExist(dir);
            }
            catch(final Throwable t)
            {
                throw new Exception("Unable to create the directory specified by " + getPropertyName() + ", "
                    + dir.getAbsolutePath() + ": " + t.getMessage());
            }
        }

        //If the export directory exists, it better be a direcotry and at least readable.
        if((_checkForReadability || _checkForWritability) && !dir.exists())
        {
            throw new Exception("The directory specified by " + this.getPropertyName() + ", " + dir.getAbsolutePath()
                + ", does not exist.");
        }
        if(_checkForReadability && !dir.canRead())
        {
            throw new Exception("The directory specified by " + this.getPropertyName() + ", " + dir.getAbsolutePath()
                + ", cannot be read from.");
        }
        if(_checkForWritability && !dir.canWrite())
        {
            throw new Exception("The directory specified by " + this.getPropertyName() + ", " + dir.getAbsolutePath()
                + ", cannot be written to.");
        }
    }

}
