package ohd.hseb.hefs.utils.adapter;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import nl.wldelft.util.logging.LoggerContextWrapper;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.gui.about.OHDConfigInfo;
import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.tools.PropertiesTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandlerException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.IDriver;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.io.ExceptionParser;
import ohd.hseb.util.misc.HStopWatch;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.ConsoleAppender.Target;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.apache.logging.log4j.LogManager;
import com.google.common.collect.Maps;

/**
 * Super class for adapters that do not follow the mold of OHDFewsAdapter. Specifically, if the model uses
 * {@link TimeSeriesArrays} to store time series, this should be used.<br>
 * <br>
 * When subclassing, the subclass should have its own main method with the following single line ofcode in it:<br>
 * <br>
 * HEFSModelAdapter.runAdapter(args, [subclass].class.getName());<br>
 * <br>
 * By doing so, module can be configured to call the adatpter class directory instead of calling this and specifying a
 * model name.
 * 
 * @author hank.herr
 */
public abstract class HEFSModelAdapter
{
    //private static final Logger LOG = Logger.getLogger(HEFSModelAdapter.class);
    private static final Logger LOG = LogManager.getLogger(HEFSModelAdapter.class);

    /**
     * For use with the default {@link #extractRunInfo(RunInfo)} method. It is a map of property name to variables used
     * to store those properties. This should be setup in the constructor for a subclass or in an override of the
     * {@link #extractRunInfo(RunInfo)} method if not enough information is available at construction to do it then.
     */
    private final HashMap<String, PropertyVariable> _runInfoPropertyVariables = Maps.newLinkedHashMap();

    /**
     * Since this is so often needed, I've added it in this top level adapter: an attribute to store the forecast time
     * specified in the run-info properties.
     */
    private long _forecastTime;

    /**
     * Override to extract needed information from the {@link RunInfo}. By default, this method calls the
     * {@link PropertyVariable#read(java.util.Properties)} method for each entry in the
     * {@link #_runInfoPropertyVariables} list. If other information in runInfomation must be read in, then subclass
     * this, adding other processing as needed.
     * 
     * @param runInformation {@link RunInfo} to process. Its {@link RunInfo#getProperties()} is used for default
     *            property extraction.
     * @throws Exception
     */
    protected void extractRunInfo(final RunInfo runInformation) throws Exception
    {
        _forecastTime = runInformation.getTime0Long();
        PropertiesTools.readPropertyVariables(_runInfoPropertyVariables.values(), runInformation.getProperties());
    }

    protected long getForecastTime()
    {
        return _forecastTime;
    }

    protected void setForecastTime(final long time)
    {
        _forecastTime = time;
    }

    /**
     * Clears the {@link #_runInfoPropertyVariables} map of property variables.
     */
    protected void clearPropertyVariables()
    {
        _runInfoPropertyVariables.clear();
    }

    /**
     * Adds property storing variables to the {@link #_runInfoPropertyVariables} list. This list is used by the default
     * {@link #extractRunInfo(RunInfo)} method.
     * 
     * @param variables Variables to add.
     */
    protected void addPropertyVariables(final PropertyVariable... variables)
    {
        for(final PropertyVariable propVar: variables)
        {
            if(_runInfoPropertyVariables.get(propVar.getPropertyName()) != null)
            {
                throw new IllegalArgumentException("The PropertyVariable with property name "
                    + propVar.getPropertyName() + " is a repeat, which is not allowed.");
            }
            _runInfoPropertyVariables.put(propVar.getPropertyName(), propVar);
        }
    }

    /**
     * @param property The property for which to acquire the variable storage object.
     * @return The {@link PropertyVariable} associated with the property. Note that if the property was not found in the
     *         run information properties, the {@link PropertyVariable} will not undergo any changes during run info
     *         extraction.
     */
    protected PropertyVariable getPropertyVariable(final String property)
    {
        return _runInfoPropertyVariables.get(property);
    }

    /**
     * Override to load states. Default behavior is to output a warning: if this is called, then states description
     * files were provided it the run-information. However, if this is not overridden, then the model must not expect
     * states, so why were the description files provided? The warning message explains this to the users.
     * 
     * @param stateMetaInfo {@link List} of {@link StateMetaInformation} objects providing the contents of all state
     *            description files specified in the run information.
     */
    protected void loadStates(final List<StateMetaInformation> stateMetaInfo)
    {
        LOG.warn("State description files were specified in the run-information file, but no states are used by this model.");
    }

    /**
     * Override as needed. This method is called immediately after the about information is output to the log and before
     * extracting run-time information. It is useful for loading settings upon which the run-time information may
     * depend.<br>
     * <br>
     * By default, this does nothing.
     * 
     * @param runInfo Specifying, currently, the work directory only, {@link RunInfo#getWorkDir()}. Do NOT use the
     *            method {@link RunInfo#getInputDataSetDir()}, because that value is never set when reading the run-time
     *            information (no one coded the ability to extract it).
     */
    protected void loadModuleDataSetsBeforeExtractingRunTimeInformation(final RunInfo runInfo) throws Exception
    {
    }

    /**
     * Override as needed. This method is called immediately after the run-time information is extracted and before the
     * state information is extracted. This is for loading module data sets that depend upon run-time information in
     * order to parse/validate.<br>
     * <br>
     * By default, this does nothing.
     * 
     * @param runInfo Specifying, currently, the work directory only, {@link RunInfo#getWorkDir()}. Do NOT use the
     *            method {@link RunInfo#getInputDataSetDir()}, because that value is never set when reading the run-time
     *            information (no one coded the ability to extract it).
     */
    protected void loadModuleDataSetsAfterExtractingRunTimeInformation(final RunInfo runInfo) throws Exception
    {
    }

    /**
     * @return A displayable name for the model. Must be overridden.
     */
    protected abstract String getModelName();

    /**
     * Override to perform the main work of the model. This method is called after {@link #extractRunInfo(RunInfo)} and,
     * if necessary, {@link #loadStates(List)} are called.
     * 
     * @param inputTS The input time series, read in from inputs.xml. If the run information does not include
     *            inputTimeSeriesFile, then null will be passed in.
     * @return Output time series generated by the model. If null is returned, it is assumed the the model does not
     *         generate output time series.
     * @throws Exception If an error occurs that must halt execution, throw an exception.
     */
    protected abstract TimeSeriesArrays executeModel(TimeSeriesArrays inputTS) throws Exception;

    /**
     * @param runInformation The run information to use.
     * @param inputTS The input time series loaded from all files specified in the run info.
     * @param stateMetaInfo The state meta-information loaded from all state description files specified in the run
     *            info.
     * @return Output time series generated by the model. Null is a possibility; see
     *         {@link #executeModel(TimeSeriesArrays)}.
     * @throws Exception
     */
    private TimeSeriesArrays execute(final RunInfo runInformation,
                                     final TimeSeriesArrays inputTS,
                                     final List<StateMetaInformation> stateMetaInfo) throws Exception
    {
        printAboutInfo();
        loadModuleDataSetsBeforeExtractingRunTimeInformation(runInformation);
        extractRunInfo(runInformation);
        loadModuleDataSetsAfterExtractingRunTimeInformation(runInformation);
        if(!stateMetaInfo.isEmpty())
        {
            loadStates(stateMetaInfo);
        }
        return executeModel(inputTS);
    }

    /**
     * Print version/about information to the LOG file.
     */
    private void printAboutInfo()
    {
        final OHDConfigInfo aboutInfo = OHDConfigInfo.loadOHDConfigInfo(getClass());
        if(aboutInfo != null)
        {
            aboutInfo.printVersionInformation(LOG);
        }
        else
        {
            LOG.warn("Unable to load version information for " + getClass().getName()
                + " either due to no appropriate annotation or no version file found.");
        }
    }

    /**
     * Loads the state meta information from the files specified in the run information. Uses
     * {@link XMLTools#readXMLFromFile(File, ohd.hseb.hefs.utils.xml.XMLReadable)} to do the reading.
     * 
     * @param runInfo The run information.
     * @throws GenericXMLReadingHandlerException
     */
    private static List<StateMetaInformation> loadStateMetaInformation(final RunInfo runInfo) throws GenericXMLReadingHandlerException
    {
        final List<StateMetaInformation> results = new ArrayList<StateMetaInformation>();
        final List<String> fileNames = runInfo.getInputStateDescriptionFileList();
        for(final String fileName: fileNames)
        {
            final StateMetaInformation stateMetaInfo = new StateMetaInformation();
            final File stateFile = new File(fileName);
            if(!stateFile.exists())
            {
                LOG.warn("The run-information specified state description (meta-info) file, "
                    + stateFile.getAbsolutePath() + ", does not exist." + "  States will not be loaded.");
            }
            else
            {
                XMLTools.readXMLFromFile(stateFile, stateMetaInfo);
                results.add(stateMetaInfo);
            }
        }
        return results;
    }

    /**
     * Loads the input time series from all files specified in the run information. Uses
     * {@link TimeSeriesArraysTools#readFromFiles(List)} to do the reading.
     * 
     * @param runInfo The run information.
     * @return The loaded time series.
     * @throws IOException
     * @throws InterruptedException
     */
    private static TimeSeriesArrays loadInputTimeSeries(final RunInfo runInfo) throws IOException, InterruptedException
    {
        TimeSeriesArrays inputTS = null;
        final List<File> inputFiles = new ArrayList<File>();
        for(final String name: runInfo.getInputTimeSeriesFileList())
        {
            inputFiles.add(new File(name));
        }
        inputTS = TimeSeriesArraysTools.readFromFiles(inputFiles);
        return inputTS;
    }

    /**
     * Parse the run info file and stores the properties within _runProperties.
     * 
     * @param runInfoFileName
     * @throws ModelException
     */
    private static RunInfo parseRunInfoFile(final String runInfoFileName) throws Exception
    {
        FewsXMLParser parser;
        final Diagnostics logger = new Diagnostics();
        try
        {
            final RunInfo modelRunInfo = new RunInfo(logger);
            parser = new FewsXMLParser(logger);
            parser.parseRunInfoFile(runInfoFileName, modelRunInfo);
            return modelRunInfo;
        }
        catch(final Exception e)
        {
            final String message = "Unable to parse run information file, " + runInfoFileName + ": " + e.getMessage();

            LOG.error(message);
            LOG.debug("Details of Error: " + logger.getListOfDiagnosticsAsString());
            throw new Exception(message);
        }
    }

    /**
     * Output the diagnostics file.
     * 
     * @param runInfo The adapter {@link RunInfo} object.
     * @param appender The {@link HEFSAdapterLogAppender} containing the log messages.
     */
    private static void outputDiagnostics(final RunInfo runInfo, final HEFSAdapterLogAppender appender) throws IOException
    {
        try
        {
            //No diag file or run info is not defined
            if((runInfo == null) || (runInfo.getDiagnosticFile() == null) || (runInfo.getDiagnosticFile().isEmpty()))
            {
                appender.outputDiagnostics();
            }
            //Diag file is defined. Try to output diagnostics.
            else
            {
                try
                {
                    appender.outputDiagnostics(runInfo.getDiagnosticFile());
                }
                catch(final Exception e2)
                {                	
                    LOG.error("Unable to dump diagnostics to file " + runInfo.getDiagnosticFile()
                        + "; outputting to standard out.");
                    appender.outputDiagnostics();
                }
            }
        }
        finally
        {
            //Remove the appender for this run.
            //Logger.getRootLogger().removeAppender(appender);
            // FB 1239
            //Logger.getLogger("ohd.hseb").removeAppender(appender);
 
        	Configuration config;
        	LoggerContext ctx;
        	Object contextReturn = LogManager.getContext(false);
        	if (contextReturn instanceof LoggerContextWrapper)
        	{
        	    config = ((LoggerContext)((LoggerContextWrapper)contextReturn).getContext()).getConfiguration();
        	    ctx = (LoggerContext)((LoggerContextWrapper)contextReturn).getContext();
        	}
        	else if (contextReturn instanceof LoggerContext)
        	{
        	    config = ((LoggerContext)contextReturn).getConfiguration();
        	    ctx = (LoggerContext)contextReturn;
        	}
        	else
        	{
        		throw new IOException("Log4j tool not initialized for loggin!!!");
        	}
        	//Do what it currently does after that.            
            //LoggerConfig loggerConfig = config.getLoggerConfig(LogManager.ROOT_LOGGER_NAME);
            LoggerConfig loggerConfig = config.getLoggerConfig("ohd.hseb");
            if (appender != null)
            {
                appender.stop();
            }
            loggerConfig.removeAppender(appender.getName());
            config.removeLogger("ohd.hseb");
            ctx.updateLoggers();
            
        }
    }
    
    
   

    /**
     * Execute to run a model adapter. Note that if an error occurs in processing before the diagnostic file can be
     * identified, messages are output to standard out.
     * 
     * @param args Standard main-line arguments specified in configuration.
     * @param classNameOfModelToExecute The class name of the model to execute. If null is given, it is assumed that the
     *            run-time information file specifies the model.
     */
    public static void runAdapter(final String[] args, final String classNameOfModelToExecute) throws IOException
    {
    	
        //TODO: The below should be changed to work similarly to LoggingTools.initializeLogFileLogger.  
        //Please look into creating a utility for the below code because some of it appears in several places,
        //here and in LoggingTools.  
    	
        final HStopWatch stopWatch = new HStopWatch();
        RunInfo runInfo = null;
        
        //Get the configuration and logger context.
    	Configuration config;
    	LoggerContext ctx;
    	Object contextReturn = LogManager.getContext(false);
    	if (contextReturn instanceof LoggerContextWrapper)
    	{
    	    config = ((LoggerContext)((LoggerContextWrapper)contextReturn).getContext()).getConfiguration();
    	    ctx = (LoggerContext)((LoggerContextWrapper)contextReturn).getContext();
    	}
    	else if (contextReturn instanceof LoggerContext)
    	{
    	    config = ((LoggerContext)contextReturn).getConfiguration();
    	    ctx = (LoggerContext)contextReturn;
    	}
    	else
    	{
    		throw new IOException("Log4j tool not initialized for loggin!!!");
    	}      
        
    	
    	LoggerConfig loggerConfig = null;
    	if ( config.getLoggerConfig("ohd.hseb").getName() != null && config.getLoggerConfig("ohd.hseb").getName().length() > 0)
    	{
    		// For testing in developemet environment only - junit test
    		loggerConfig = config.getLoggerConfig("ohd.hseb");
    	} else {
    		loggerConfig = new LoggerConfig("ohd.hseb", Level.DEBUG, false);
    	}         	
    	
      //Construct an OHDConsoleAppender with a specific pattern and start it.
        PatternLayout layout = PatternLayout.newBuilder()
        		.withConfiguration(config)
                .withPattern("%d{HH:mm:ss.SSS} %level %msg%n")
                .build();
        HEFSAdapterLogAppender appender = new HEFSAdapterLogAppender("HEFSAdapterLogAppender", layout, Target.SYSTEM_OUT);        
        appender.start();
        
        config.addAppender(appender);
        
        //Add the appender to the logger and the logger to the configuration, again as specified in that XML file.
        loggerConfig.addAppender(appender, Level.DEBUG, null);
        config.addLogger("ohd.hseb", loggerConfig);
        
        //Update the loggers.
        ctx.updateLoggers();
                              
        if(args.length != 1)
        {
            LOG.error("Invalid number of arguments for HEFSModelAdapter main method. "
                + "Expected only 1, the run info file name, but got " + args.length + ".");
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Parse run info
        try
        {
            runInfo = parseRunInfoFile(args[0]);
        }
        catch(final Exception e)
        {
            //Error and details are already output in parse method.
            LoggingTools.outputDebugLines(LOG, ExceptionParser.multiLineStackTrace(e));
            LOG.error("Unable to parse run-time information from " + args[0] + ". Aborting run of model adapter.");
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Set the debug message level
        final String debugLevelStr = (String)runInfo.getProperties().get("printDebugInfo");

        try
        {                               	
            loggerConfig.setLevel(Level.INFO);            
            if((debugLevelStr != null) && (Integer.parseInt(debugLevelStr) > 0))
        	{        		
        		appender.turnOnDebugLogging();
        		loggerConfig.setLevel(Level.DEBUG);
//        		int debugLevel = Integer.parseInt(debugLevelStr);
//                switch (debugLevel) {
//                 case ohd.hseb.util.Logger.DEBUG :
//                	 loggerConfig.setLevel(Level.DEBUG);   
//                	break;
//                 case  ohd.hseb.util.Logger.INFO :
//                	 loggerConfig.setLevel(Level.INFO);
//                	 break;
//                 case ohd.hseb.util.Logger.WARNING :
//                	 loggerConfig.setLevel(Level.WARN);
//                	 break;
//                 case ohd.hseb.util.Logger.ERROR :
//                	 loggerConfig.setLevel(Level.ERROR);
//                	 break;
//                 case ohd.hseb.util.Logger.FATAL :
//                	 loggerConfig.setLevel(Level.FATAL);
//                	 break; 
//                 default:
//                	 loggerConfig.setLevel(Level.FATAL);
//                	 break;
//                }                  
        	}                        
            ctx.updateLoggers();
        }
        catch(final NumberFormatException e)
        {
            LoggingTools.outputDebugLines(LOG, ExceptionParser.multiLineStackTrace(e));
            LOG.error("The printDebugInfo level, " + debugLevelStr
                + ", is invalid: it is not a number. Aborting run of model adapter.");
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Get the subclass name.
        String modelClassName = classNameOfModelToExecute;
        if(modelClassName == null) //If it is null, use the runinfo model property.
        {
            modelClassName = (String)runInfo.getProperties().get("model");
        }
        if(modelClassName == null) //If it is still null, we have a problem.
        {
            LOG.error("The model name is not provided in the run information file. " + "Aborting run of model adapter.");
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Load the adapter.
        HEFSModelAdapter modelAdapter = null;
        try
        {
            modelAdapter = (HEFSModelAdapter)Class.forName(modelClassName).getDeclaredConstructor().newInstance();                                   
        }
        catch(final Exception e)
        {
            LoggingTools.outputDebugLines(LOG, ExceptionParser.multiLineStackTrace(e));
            LOG.error("Error loading model adapter: " + e.getMessage());
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Load the state meta-information.  After calling loadStateMetaInformation, the list will not be null,
        //but may be empty.
        List<StateMetaInformation> stateMetaInfo = null;
        try
        {
            stateMetaInfo = loadStateMetaInformation(runInfo);
        }
        catch(final GenericXMLReadingHandlerException e)
        {
            LoggingTools.outputDebugLines(LOG, ExceptionParser.multiLineStackTrace(e));
            LOG.error("Error loading state meta-information prior to adapter execution: " + e.getMessage());
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Load the input TS.
        TimeSeriesArrays inputTS = null;
        try
        {
            inputTS = loadInputTimeSeries(runInfo);
        }
        catch(final Exception e)
        {
            LoggingTools.outputDebugLines(LOG, ExceptionParser.multiLineStackTrace(e));
            LOG.error("Error loading input time series: " + e.getMessage());
            outputDiagnostics(runInfo, appender);
            return;
        }

        //Execute adapter.
        try
        {
            LOG.info("Executing model " + modelAdapter.getModelName() + " (" + modelClassName + ")...");
            final TimeSeriesArrays outputTS = modelAdapter.execute(runInfo, inputTS, stateMetaInfo);            
            
            //Generate output TS file if necessary.
            if((outputTS != null) && (!outputTS.isEmpty()))
            {
                if(runInfo.getOutputTimeSeriesFile() == null)
                {
                    LOG.warn("The model generated " + outputTS.size()
                        + " output time series, but the run information did not include outputTimeSeriesFile; "
                        + "no output time series generated.");
                }
                else
                {
                    LOG.debug("Writing output time series to " + runInfo.getOutputTimeSeriesFile() + "...");
                    TimeSeriesArraysTools.writeToFile(new File(runInfo.getOutputTimeSeriesFile()), outputTS);
                    LOG.debug("Done writing output time series.");
                }
            }

            LOG.info("Done executing model " + modelClassName + ".  Completed run in "
                + (stopWatch.getElapsedMillis() / 1000D) + " seconds.");
        }
        //Need to catch ALL throwables, because CHPS does a lousy job of handling those throwables that are not
        //exceptions, such as NoSuchMethodErrors.  
        catch(final Throwable t)
        {
            if(t instanceof Exception)
            {
                final Exception e = (Exception)t;
                LoggingTools.outputStackTraceAsDebug(LOG, e);
                LOG.error("Error executing model: " + t.getMessage());
            }
            else
            {
                t.printStackTrace();
                LOG.fatal("Fatal error executing model (see stack trace dumped to terminal): " + t.getMessage());
            }
            outputDiagnostics(runInfo, appender);
            return;
        }

        outputDiagnostics(runInfo, appender);
        return;
    }

    public static void main(final String[] args) throws IOException 
    {
        runAdapter(args, null);
    }

}
