package ohd.hseb.hefs.utils.adapter;

import java.util.HashMap;
import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ohd.hseb.hefs.utils.xml.XMLVariable;
import ohd.hseb.util.misc.SegmentedLine;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;

/**
 * A {@link PropertyVariable} that can include a qualifier on the key. For example, if the property has the same name
 * for all locations but must be allowed to vary by location, then use this type of variable. It allows for a
 * {@link #QUALIFIER_SEGMENTER} to be used to further qualify the property key and then stores the value in a map where
 * the key is the qualifier. If no qualifier on the key is defined, then it is assumed that the qualifier is null, which
 * indicates a default value.<br>
 * <br>
 * Example: Supposed the properties are:<br>
 * <br>
 * aaaa.Property = value1<br>
 * bbbb.Property = value2<br>
 * Property = value3<br>
 * <br>
 * Then the default value of Property for any location other than aaaa and bbbb is value3. The value for aaaa is value1
 * and for bbbb is value2.<br>
 * <br>
 * A {@link #_qualifierIsSuffix} is provided that allows for the qualifier to be either before or after the property
 * name. Both the base property name and suffix boolean are specified in the constructor along with an
 * {@link XMLVariable} for storing the qualifier and an {@link XMLVariable} for storing the value (as standard). In both
 * cases, the tag name of the {@link XMLVariable} instances are NOT used. Furthermore, when a new entry is placed in the
 * map, those two variables are cloned and the clone is used to store the new value. The {@link XMLVariable} instances
 * provided to the constructor are never used directly except as the source of a clone call.
 * 
 * @author hankherr
 * @param T The type for all values stored in the map (not keys!). This controls the return value of
 *            {@link #getValue(Object)} and is specified by the value {@link XMLVariable} type passed into the
 *            constructor.
 */
public class MappedPropertyVariable<T> implements PropertyVariable<T>
{
    private static final Logger LOG = LogManager.getLogger(MappedPropertyVariable.class);

    private final static String QUALIFIER_SEGMENTER = ".";

    private final HashMap<XMLVariable, XMLVariable<T>> _qualifierToValueMap = Maps.newHashMap();

    private final String _propertyKey;
    private final XMLVariable _keyQualfierVariable;
    private final XMLVariable _valueVariable;
    private final boolean _qualifierIsSuffix;

    /**
     * Calls {@link #MappedPropertyValue(String, XMLVariable, XMLVariable, boolean)} with the suffix flag set to true
     * (its default behavior).
     */
    public MappedPropertyVariable(final String key, final XMLVariable keyQualifier, final XMLVariable<T> value)
    {
        this(key, keyQualifier, value, true);
    }

    /**
     * @param key The key {@link String} to look for.
     * @param keyQualifier An {@link XMLVariable} defining the type of qualifier. Note that this variable is stored
     *            herein but is never used directly except to call its {@link XMLVariable#clone()} method to create keys
     *            for {@link #_qualifierToValueMap}.
     * @param value An {@link XMLVariable} defining the type of value expected. If the return of
     *            {@link XMLVariable#get()} for this variable is NOT null, then it is assumed to specify a default
     *            value; i.e., a value to store with no qualifier. Note that this variable is stored herein but is never
     *            used directly except to call its {@link XMLVariable#clone()} method to create values for
     *            {@link #_qualifierToValueMap}. This specifies the type the generic type T of the return of
     *            {@link #getValue(Object)}.
     * @param qualifierIsSuffix If true, then the qualifier is AFTER the '.'; otherwise, it is before the '.'.
     */
    @SuppressWarnings("unchecked")
    public MappedPropertyVariable(final String key,
                                  final XMLVariable keyQualifier,
                                  final XMLVariable<T> value,
                                  final boolean qualifierIsSuffix)
    {
        _propertyKey = key;
        _keyQualfierVariable = keyQualifier;
        _valueVariable = value;
        _qualifierIsSuffix = qualifierIsSuffix;

        if(value.get() != null)
        {
            final XMLVariable tmpVar = _keyQualfierVariable.clone();
            tmpVar.set(null);
            _qualifierToValueMap.put(tmpVar, value.clone());
        }
    }

    public T getDefaultValue() throws Exception
    {
        return getValue(null);
    }

    /**
     * @param keyQualifier The qualifier for which to acquire the value. Pass in null to acquire the default value.
     * @return The value. If null is returned, then nothing was specified in the properties for that key.
     * @throws Exception If the provided keyQualifier is the wrong type of object.
     */
    @SuppressWarnings("unchecked")
    public T getValue(final Object keyQualifier)
    {
        final XMLVariable tmpVar = _keyQualfierVariable.clone();
        tmpVar.set(keyQualifier);
        if(_qualifierToValueMap.get(tmpVar) == null)
        {
            return null;
        }
        return _qualifierToValueMap.get(tmpVar).get();
    }

    /**
     * Clones both {@link #_keyQualfierVariable} and {@link #_valueVariable} and attempts to call their
     * {@link XMLVariable#setValueOfElement(String, String)} methods for the passed in {@link String}s. If successful,
     * then an entry is added to the map.
     * 
     * @param qualifierStr If null, then the value will be assigned to the default map entry, or that entry for which
     *            the qualifier {@link XMLVariable} contains a null.
     * @param valueStr The value; cannot be null.
     * @throws Exception
     */
    @SuppressWarnings("unchecked")
    private void instantiateMapEntry(final String qualifierStr, final String valueStr) throws Exception
    {
        if(valueStr == null)
        {
            throw new IllegalStateException("For property " + _propertyKey + " and qualifier " + qualifierStr
                + ", the value is null.  That should not be possible.");
        }
        final XMLVariable qualVar = _keyQualfierVariable.clone();
        final XMLVariable valueVar = _valueVariable.clone();

        if(Strings.isNullOrEmpty(qualifierStr))
        {
            qualVar.set(null);
        }
        else
        {
            qualVar.setValueOfElement(qualVar.getXMLTagName(), qualifierStr);
            qualVar.finalizeReading();
        }

        valueVar.setValueOfElement(valueVar.getXMLTagName(), valueStr);
        valueVar.finalizeReading();

        if(_qualifierToValueMap.get(qualVar) != null)
        {
            LOG.warn("Property " + _propertyKey + " with the qualifier '" + qualVar.get()
                + "' has been defined more than once; new value '" + valueStr + "' will override old value '"
                + _qualifierToValueMap.get(qualVar).get() + "'.");
        }

        isValid(qualVar, valueVar);
        _qualifierToValueMap.put(qualVar, valueVar);
    }

    /**
     * Override to check a combination of qualifier and value for validity prior to storing.
     * 
     * @throws Exception If the combination is invalid.
     */
    protected void isValid(final XMLVariable qualVar, final XMLVariable valueVar) throws Exception
    {
    }

    @Override
    public void read(final Properties props) throws Exception
    {
        for(final Object key: props.keySet())
        {
            final String keyStr = (String)key;

            //Accept the property if it starts with the needed key.
            if((_qualifierIsSuffix && (keyStr.startsWith(_propertyKey)))
                || (!_qualifierIsSuffix && (keyStr.endsWith(_propertyKey))))
            {
                //Segment the key.  If there are more than two segments, that's bad.
                final SegmentedLine segLine = new SegmentedLine(keyStr,
                                                                QUALIFIER_SEGMENTER,
                                                                SegmentedLine.MODE_NO_EMPTY_SEGS);
                if(segLine.getNumberOfSegments() > 2)
                {
                    throw new Exception("For property " + _propertyKey + ", improperly formatted key: '" + keyStr
                        + "' contains more than one '" + QUALIFIER_SEGMENTER + "'.");
                }

                //The qualifier is the second segment, or null if not specified.
                String qualifierStr = null;
                if(segLine.getNumberOfSegments() == 2)
                {
                    if(_qualifierIsSuffix)
                    {
                        qualifierStr = segLine.getSegment(1);
                    }
                    else
                    {
                        qualifierStr = segLine.getSegment(0);
                    }
                }

                //Add the map entry.
                instantiateMapEntry(qualifierStr, props.getProperty(keyStr));
            }
        }
    }

    @Override
    public String getPropertyName()
    {
        return this._propertyKey;
    }

    @Override
    public void processAfterSuccessfulRead() throws Exception
    {
        //Does nothing by default.
    }
}
