package ohd.hseb.hefs.utils.adapter;

import java.io.File;
import java.sql.Date;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLEnum;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;
import ohd.hseb.hefs.utils.xml.vars.XMLFixedDate;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import org.xml.sax.Attributes;

/**
 * Stores state meta-information contained in the states.xml file output by CHPS. This class provides an XML reader and
 * writer.
 * 
 * @author hank.herr
 */
public class StateMetaInformation implements XMLReadable, XMLWritable
{
    private final static File NONE = new File("-unspecified-");

    public static enum StateType
    {
        file, directory;
    }

    private final XMLString _stateId = new XMLString("stateId", "cold");
    private final XMLString _stateName = new XMLString("stateName", "time series used to build lagged ensembles");
    private final XMLInteger _timeZoneInt = new XMLInteger("timeZone", 0); //default to GMT
    private final XMLFixedDate _dateTime = new XMLFixedDate("dateTime", new Date(0L));
    private final List<StateLocationInformation> _stateLocations = new ArrayList<StateLocationInformation>();

//State XML example from ohdfewsadapter
//    <stateId>warm</stateId>
//    <stateName>warm</stateName>
//    <timeZone>-8</timeZone>
//    <dateTime date="2008-04-11" time="04:00:00"/>
//    <stateLoc type="file">
//      <readLocation>Modules/snow17/forecast_isac1upperaescchngtopmods/states/statesI.txt</readLocation>
//      <writeLocation>Modules/snow17/forecast_isac1upperaescchngtopmods/output/statesO.txt</writeLocation>
//    </stateLoc>

    public String getXMLTagName()
    {
        return "State";
    }

    public String getStateId()
    {
        return _stateId.get();
    }

    public String getStateName()
    {
        return _stateName.get();
    }

    public int getTimeZoneInt()
    {
        return _timeZoneInt.get();
    }

    public long getDateTime()
    {
        return _dateTime.get();
    }

    public StateLocationInformation getStateLocation()
    {
        if(_stateLocations.size() > 1)
        {
            throw new IllegalStateException("Method getStateLocation() was called, but more than one state location is available.");
        }
        if(_stateLocations.isEmpty())
        {
            return null;
        }
        return _stateLocations.get(0);
    }

    public StateLocationInformation getStateLocation(final int index)
    {
        if(_stateLocations.isEmpty())
        {
            return null;
        }
        return _stateLocations.get(index);
    }

    public int getStateLocationCount()
    {
        return _stateLocations.size();
    }

//Is there ever a reason to set these values other than during reading?
//    public void setStateId(String stateId)
//    {
//        _stateId.set(stateId);
//    }
//
//    public void setStateName(String stateName)
//    {
//        _stateName.set(stateName);
//    }
//
//    public void setTimeZoneInt(int timeZoneInt)
//    {
//        _timeZoneInt.set(timeZoneInt);
//    }
//
//    public void setDateTime(long dateTimeInMillis)
//    {
//        _dateTime.set(dateTimeInMillis);
//    }

    @Override
    public XMLWriter getWriter()
    {
        final List<XMLWritable> components = new ArrayList<XMLWritable>();
        components.add(_stateId);
        components.add(_stateName);
        components.add(_timeZoneInt);
        components.add(_dateTime);
        for(final StateLocationInformation info: this._stateLocations)
        {
            components.add(info);
        }
        final CompositeXMLWriter writer = new CompositeXMLWriter(getXMLTagName(), components);
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        final List<XMLReadable> components = new ArrayList<XMLReadable>();
        components.add(_stateId);
        components.add(_stateName);
        components.add(_timeZoneInt);
        components.add(_dateTime);
        final CompositeXMLReader reader = new CompositeXMLReader(getXMLTagName(), components)
        {
            @Override
            public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
            {
                //This grows the list of state locations as the element is read.  Note that this only works
                //if the CompositeXMLReader containing this has its _readInOrder set to true!  If not set the true,
                //then the first instance of stateLoc in the list of elements to read will always be used, not this
                //added one.
                if(elementName.equals("stateLoc"))
                {
                    //Add a component to hold the stateLoc
                    final StateLocationInformation stateLoc = new StateLocationInformation();
                    StateMetaInformation.this._stateLocations.add(stateLoc);
                    addComponent(stateLoc);
                }
                return super.readInPropertyFromXMLElement(elementName, attr);
            }
        };
        reader.setReadInOrder(true);
        return reader;
    }

    /**
     * Stores a pair: readLocation and writeLocation
     * 
     * @author hank.herr
     */
    public class StateLocationInformation implements XMLReadable, XMLWritable
    {
        private final XMLFile _readLocationFile = new XMLFile("readLocation", NONE, true);
        private final XMLFile _writeLocationFile = new XMLFile("writeLocation", NONE, false);
        @SuppressWarnings({"unchecked", "rawtypes"})
        private final XMLEnum<StateType> _type = new XMLEnum(StateType.class, "type");

        public File getReadLocationFile()
        {
            return _readLocationFile.get();
        }

        public File getWriteLocationFile()
        {
            return _writeLocationFile.get();
        }

        public StateType getType()
        {
            return _type.get();
        }

        public void setReadLocationFile(final File readLocationFile)
        {
            _readLocationFile.set(readLocationFile);
        }

        public void setWriteLocationFile(final File writeLocationFile)
        {
            _writeLocationFile.set(writeLocationFile);
        }

        public void setType(final StateType type)
        {
            _type.set(type);
        }

        @Override
        public XMLWriter getWriter()
        {
            final CompositeXMLWriter stateLocWriter = new CompositeXMLWriter("stateLoc",
                                                                             _readLocationFile,
                                                                             _writeLocationFile);
            stateLocWriter.addAttribute(_type, true);
            return stateLocWriter;
        }

        @Override
        public XMLReader getReader()
        {
            final CompositeXMLReader stateLocReader = new CompositeXMLReader("stateLoc",
                                                                             _readLocationFile,
                                                                             _writeLocationFile);
            stateLocReader.addAttribute(_type, true);
            stateLocReader.setReadInOrder(true);
            return stateLocReader;
        }

        @Override
        public String toString()
        {
            return "State Location (" + _type.get() + "): read = " + _readLocationFile.get() + "; write = "
                + _writeLocationFile.get() + ".";
        }
    }
}
