package ohd.hseb.hefs.utils.arguments;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Generic class records an argument name with function parameters, if applicable.
 * 
 * @author herrhd
 */
public class Argument
{
    public static char PARAMETER_DELIMITER = ';';

    public Argument()
    {

    }

    private String _argumentName;

    private final List<String> _functionParameters = new ArrayList<String>();

    public void setArgumentName(final String name)
    {
        this._argumentName = name;
    }

    public String getArgumentName()
    {
        return this._argumentName;
    }

    public void addFunctionParameterValue(final String functionParameter)
    {
        this._functionParameters.add(functionParameter);
    }

    /**
     * @return List of parameter values in order of appearance within the parentheses of a function.
     */
    public List<String> getFunctionParameterValues()
    {
        return this._functionParameters;
    }

    public void clearFunctionParameterValues()
    {
        this._functionParameters.clear();
    }

    /**
     * Does argument replacement for the function parameters using the passed in argProc.
     * 
     * @param argProc ArgumentProcessor that will do the replacement for function parameters.
     */
    public void replaceArgumentsInFunctionParameters(final ArgumentsProcessor argProc)
    {
        for(int i = 0; i < this._functionParameters.size(); i++)
        {
            final String replacedValue = argProc.replaceArgumentsInString(this._functionParameters.get(i));
            this._functionParameters.set(i, replacedValue);
        }
    }

    /**
     * @return A string that specifies the current argument when executing a replacement on it.
     */
    public String buildArgumentString(final char argumentChar)
    {
        String result = argumentChar + this.getArgumentName();
        if(this._functionParameters.size() > 0)
        {
            result += "(";
            for(int i = 0; i < _functionParameters.size(); i++)
            {
                result += _functionParameters.get(i);
                if(i < _functionParameters.size() - 1)
                {
                    result += PARAMETER_DELIMITER;
                }
            }
            result += ")";
        }
        result += argumentChar;
        return result;
    }

    /**
     * Extracts one parameter at a time from the searched string.
     * 
     * @param searchedString String to search.
     * @param startIndex Index of the open parentheses initiating the parameter extraction.
     * @return Index of a comma or closing parentheses, whichever is found at the end of the parameter just extracted. A
     *         -1 is returned if none is found before reaching the end of the string.
     */
    private int extractFunctionParameter(final String searchedString, final int startIndex)
    {
        int index = startIndex + 1;
        while(index < searchedString.length())
        {
            if((searchedString.charAt(index) == PARAMETER_DELIMITER) || (searchedString.charAt(index) == ')'))
            {
                this._functionParameters.add(searchedString.substring(startIndex + 1, index).trim());
                return index;
            }
            index++;
        }
        return -1;
    }

    /**
     * Extracts this argument and function parameters from the passed in searched string.
     * 
     * @param searchedString The string from which to extract argument information.
     * @param startIndex Index of the opening ARG_CHAR initiating the argument extraction.
     * @return Index of the chosing ARG_CHAR or -1 if none was found before reaching the end of the string.
     */
    protected int populateArgumentNameAndParameterIfOne(final String searchedString,
                                                        final int startIndex,
                                                        final char argumentChar)
    {
        int index = startIndex + 1;
        _argumentName = null;
        this._functionParameters.clear();
        while(index < searchedString.length())
        {
            if(searchedString.charAt(index) == '(')
            {
                //Move through the searched string until we reach the end (workingIndex < 0) or a closing parentheses.
                int workingIndex = extractFunctionParameter(searchedString, index);
                while((workingIndex >= 0) && (searchedString.charAt(workingIndex) != ')'))
                {
                    workingIndex = extractFunctionParameter(searchedString, workingIndex);
                }

                //Went off the end of the string without any close parentheses.  This implies either
                //no argument was found or it was badly formatted; either way just quit.
                if(workingIndex < 0)
                {
                    return -1;
                }

                _argumentName = searchedString.substring(startIndex + 1, index).trim();
                index = workingIndex;
            }
            else if(searchedString.charAt(index) == argumentChar)
            {
                if(_argumentName == null)
                {
                    _argumentName = searchedString.substring(startIndex + 1, index).trim();
                }
                break;
            }
            index++;
        }

        //No closing ARG_CHAR was found.
        if(index == searchedString.length())
        {
            return -1;
        }

        return index;
    }

    @Override
    public String toString()
    {
        return "argumentName = " + this._argumentName + "; functionParameters = "
            + Arrays.toString(this._functionParameters.toArray());
    }
}
