package ohd.hseb.hefs.utils.arguments;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import nl.wldelft.fews.common.config.GlobalProperties;
import ohd.hseb.hefs.utils.plugins.GenericParameter;
import ohd.hseb.hefs.utils.plugins.UniqueGenericParameterList;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriter;

/**
 * Used to replace arguments in string. This can be used to replace arguments in a generic string that is meant to apply
 * to any location, such that the resulting String is location specific (after argument replacement). Arguments are
 * identified by an ARG_CHAR both before and after the argument. For example, within the string,<br>
 * <br>
 * "My name is \@name\@"<br>
 * <br>
 * The argument name is 'name' and it is surrounded by \@ specifying it as a potential argument. This string can be
 * passed into an ArgumentProcessor replaceArgumentsInString method to replace name with the value of the name argument,
 * for example "Hank". However, if there are no arguments specified with the name "name", then no replacement is made.
 * <br>
 * <br>
 * Arguments may also specify functions that accept parameters. Open and closing parentheses surround the parameter. For
 * example,<br>
 * <br>
 * "My height is \@height(Hank)\@"<br>
 * <br>
 * specifies the argument function as 'height' with parameter 'Hank'. Its up to the subclass of ArgumentsProcessor to
 * recognize height as a function and then find the height of 'Hank'. Note that as soon as the open parentheses is
 * found, the argument name is set to the string between ARG_CHAR and the open parentheses. After the closing
 * parentheses, characters between it and the closing ARG_CHAR are discarded.<br>
 * <br>
 * Arguments are not recognized as such until the closing ARG_CHAR is found. If there is no closing ARG_CHAR, then there
 * are no more arguments. Also, if a parameter open parentheses is found, but no closing is found afterwards, then the
 * string is assumed to not contain any more arguments.
 * 
 * @author herrhd
 */
public abstract class ArgumentsProcessor
{
    public static char DEFAULT_ARGUMENT_CHAR = '@';

    private static final Logger LOG = LogManager.getLogger(ArgumentsProcessor.class);

    private UniqueGenericParameterList _arguments = null;
    private ProductSpecificArguments _requiredArguments = null;

    /**
     * The character used to surround an argument. By default, {@link #DEFAULT_ARGUMENT_CHAR}.
     */
    private char _argumentChar = DEFAULT_ARGUMENT_CHAR;

    /**
     * If not null, then any argument not found in the list _arguments will be acquired via this embedded
     * ArgumentsProcessor. If null, then it will be acquire from the _predefinedArguments. Note that it is assumed the
     * predefined arguments list herein has the same entries (by name, not necessarily value) as the
     * _higherLevelProcessor's predefined list.<br>
     * <br>
     * Graphics Generator specific note: For references, this should not be null.
     */
    private ArgumentsProcessor _higherLevelProcessor = null;

    /**
     * List of used arguments.
     */
    private final UniqueGenericParameterList _argumentsAccessedThusFar = new UniqueGenericParameterList();

    /**
     * If true, then the used arguments will be recorded with every call to getArgument.
     */
    private boolean _recording = false;

    /**
     * Stack of the names of arguments currently being replaced. If an argument name is present in this stack and a
     * replacement is being made involving that name, then a cyclical definition has occurred (an argument value that
     * specifies its own argument name).
     */
    private final List<String> _workingArguments = new ArrayList<String>();

    /**
     * @param arguments List of arguments.
     * @param requiredArguments List of required arguments.
     */
    protected void initialize(final UniqueGenericParameterList arguments,
                              final ProductSpecificArguments requiredArguments)
    {
        _arguments = arguments;
        _requiredArguments = requiredArguments;
    }

    /**
     * @param arguments List of arguments.
     * @param requiredArguments List of required arguments.
     * @param higherLevelProcessor High level ArgumentsProcessor that will be accessed for unfound parameters.
     */
    protected void initialize(final UniqueGenericParameterList arguments,
                              final ProductSpecificArguments requiredArguments,
                              final ArgumentsProcessor higherLevelProcessor)
    {
        initialize(arguments, requiredArguments);
        _higherLevelProcessor = higherLevelProcessor;
    }

    public void startRecordingUsedArguments()
    {
        _recording = true;
    }

    public void clearRecording()
    {
        this._argumentsAccessedThusFar.clearParameters();
    }

    public boolean isRecording()
    {
        return _recording;
    }

    public void stopRecordingUsedArguments()
    {
        _recording = false;
    }

    public UniqueGenericParameterList getArgumentsAccessedThusFar()
    {
        return this._argumentsAccessedThusFar;
    }

    public boolean isArgumentPredefined(final String arg)
    {
        return (getPredefinedArguments().getListOfPredefinedArgumentNames().indexOf(arg) >= 0);
    }

    /**
     * Special version of {@link #replaceArgumentsInString(String)} that also resolves properties via
     * {@link GlobalProperties#resolvePropertyTags(String)}. This allows for CHPS global properties to be part of file
     * names, with GraphGen arguments being given priority.
     * 
     * @param fileName File name that may have arguments or global props to replace.
     * @return Full resolved file name.
     */
    public String replaceArgumentsInFileName(final String fileName)
    {
        final String originalResults = replaceArgumentsInString(fileName);
        String finalResults = originalResults;
        try
        {
            //TODO This is probably not the best of solutions.  I should probably have a FEWSArgumentsProcessor that makes
            //use of GlobalProperties and then a superclass ArgumentsProcessor that is the same as this class but
            //without the use of GlobalProperties.  Way to do that would be to rename this to FEWSArgumentsProcessor,
            //create a new ArgumentsProcessor, and extract all other code into the new ArgumentsProcessor (massive
            //copy-and-paste).  
            //OLD if(Class.forName("nl.wldelft.fews.system.data.config.GlobalProperties") != null)
            if(Class.forName("nl.wldelft.fews.common.config.GlobalProperties") != null)
            {
                finalResults = GlobalProperties.resolvePropertyTags(originalResults);
            }
        }
        catch(final ClassNotFoundException e)
        {
            //just skip it.
            finalResults = originalResults; //Making sure!
        }
        catch(final ParseException e)
        {
            LOG.debug("Failed to apply global properties to file name after arguments are replaced, " + originalResults
                + ": " + e.getMessage());
            finalResults = originalResults; //Making sure!
        }
        return finalResults;
    }

    /**
     * @param inStr String to be replaced.
     * @return String with all arguments replaced that are defined in this ArgumentsProcessor.
     */
    public String replaceArgumentsInString(final String inStr)
    {
        return replaceArgumentsInString(inStr, null);
    }

    /**
     * This method is recursive: when inserting an argument value, it will replace the arguments in the value, first,
     * allowing one argument to refer to another argument or function. If a cycle occurs, the returned string will state
     * such.
     * 
     * @param inStr String to be replaced.
     * @param argumentReplacementToBeMade The argument replacement to make in the form of a GenericParameter. If null,
     *            then all args are to be replaced. If not null only replace this one argument value.
     * @return A string with either all arguments or only a single argument replaced.
     */
    public String replaceArgumentsInString(String inStr, final GenericParameter argumentReplacementToBeMade)
    {
        if(inStr == null)
        {
            return null;
        }
        final int startingIndex = 0;
        int foundIndex = inStr.indexOf(getArgumentChar(), startingIndex);
        int nextIndex = -1;
        String argumentValue;
        final Argument argument = new Argument();
        GenericParameter replacement;

        while(foundIndex >= 0)
        {
            //Have the Argument popuplate its name and function parameters.  Then try to find the closing
            //ARG_CHAR.  
            nextIndex = argument.populateArgumentNameAndParameterIfOne(inStr, foundIndex, getArgumentChar());
            if(nextIndex > 0)
            {
                nextIndex = inStr.indexOf(getArgumentChar(), nextIndex);
            }

            //Only if the closing ARG_CHAR is found is the argument considered valid.
            if(nextIndex >= 0)
            {
                replacement = null;

                //Find the argument for replacement.  If only one argument is to be replaced, check it against
                //the passed in GenericParameter.  Otherwise, try to find it via getArgument.
                if(argumentReplacementToBeMade != null)
                {
                    if(argumentReplacementToBeMade.isForParameter(argument.getArgumentName()))
                    {
                        replacement = argumentReplacementToBeMade;
                    }
                }

                //If the function parameters list has size 0, then it is a regular argument.  Try to find it.
                else if(argument.getFunctionParameterValues().size() == 0)
                {
                    replacement = getArgument(argument.getArgumentName());
                }

                //otherwise, it is a function.
                else
                {
                    argument.replaceArgumentsInFunctionParameters(this);
                    final String value = evaluateFunctionValue(argument);
                    if(value != null)
                    {
                        replacement = new GenericParameter(argument.getArgumentName(), value);
                    }
                }

                //Replacement is made if the argument is found, and we are either not doing one argument or, if we are,
                //then the one argument is the current found argument.
                if(replacement != null)
                {
                    //Check for a cyclical reference.
                    if(this._workingArguments.indexOf(replacement.getName()) >= 0)
                    {
                        return "ERROR: cyclical argument found: " + replacement.getName() + " requires itself.!";
                    }

                    //The work is done here.  First, add the argument to the list of working arguments.
                    //This assumes what is inside will never have an exception!
                    this._workingArguments.add(replacement.getName());

                    argumentValue = replacement.getValue();
                    if(argumentValue == null)
                    {
                        argumentValue = GenericParameter.NULL_VALUE_STRING;
                    }
                    argumentValue = this.replaceArgumentsInString(argumentValue); //recursion here!
                    inStr = inStr.substring(0, foundIndex) + argumentValue
                        + inStr.substring(nextIndex + 1, inStr.length());
                    nextIndex = nextIndex + (argumentValue.length() - (nextIndex - foundIndex)) - 1;

                    //Remove the working argument.
                    this._workingArguments.remove(replacement.getName());
                }
                //Otherwise, increment nextIndex to the next character so that it is not pointing to the ARG_CHAR that
                //ended the just found argument reference.
                else
                {
                    nextIndex++;
                }

                foundIndex = inStr.indexOf(getArgumentChar(), nextIndex);
            }
            else
            {
                foundIndex = nextIndex;
            }

        }
        return inStr;
    }

    public boolean checkForFullEqualityOfStrings(final String first, final String second, final boolean caseSensitive)
    {
        final String firstCheck = this.replaceArgumentsInString(first);
        final String secondCheck = this.replaceArgumentsInString(second);
        return StringTools.checkForFullEqualityOfStrings(firstCheck, secondCheck, caseSensitive);
    }

    public void checkParametersForValidity() throws ArgumentsProcessorException
    {
        //First, make sure all required arguments are specified.
        try
        {
            _requiredArguments.areAllRequiredArgumentsPresent(_arguments, getPredefinedArguments());
        }
        catch(final RequiredArgumentsException e)
        {
            throw new ArgumentsProcessorException("Incomplete argument list: " + e.getMessage());
        }

        //Make sure every argument has a non-null value.
        for(int i = 0; i < _arguments.getParameters().size(); i++)
        {
            if(_arguments.getParameters().get(i).getValue() == null)
            {
                throw new ArgumentsProcessorException("Argument " + _arguments.getParameters().get(i).getName()
                    + " exists but does not have a specified value.");
            }
        }
    }

    /**
     * This assumes that all used arguments within the parameter object are contained in its written XML string.
     * 
     * @param parameterObject Must be an XMLWriter.
     * @return List of arguments used in the parameterObject, gathered by writing out the XML and replacing the
     *         arguments while recording.
     */
    public UniqueGenericParameterList determineUsedArguments(final XMLWriter parameterObject)
    {
        try
        {
            final String xmlString = XMLTools.writeXMLStringFromXMLWriter(parameterObject);
            clearRecording();
            startRecordingUsedArguments();
            replaceArgumentsInString(xmlString);
            stopRecordingUsedArguments();
            return getArgumentsAccessedThusFar();
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.warn("Unable to generate xml string in order to determine used arguments.  Make list empty.");
            return new UniqueGenericParameterList();
        }
    }

    /**
     * @param checkArguments List of arguments to check.
     * @return Checks every argument in list to make sure a non-empty value is specified in this ArgumentsProcessor. If
     *         not, add it to a UniqueGenericParameterList. Returns the list of unspecified arguments.
     */
    public UniqueGenericParameterList buildListOfUnspecifiedArguments(final UniqueGenericParameterList checkArguments)
    {
        final UniqueGenericParameterList unspecified = new UniqueGenericParameterList();
        for(int i = 0; i < checkArguments.getParameters().size(); i++)
        {
            final GenericParameter check = this.getArgument(checkArguments.getParameters().get(i).getName());
            //TODO OLD WAY -- if((check == null) || (check.getValue().length() == 0)
            //    || (check.getValue().equals(getUndefinedArgumentValue())))
            if((check == null) || (check.getValue().equals(getUndefinedArgumentValue())))
            {
                unspecified.addParameter(checkArguments.getParameters().get(i));
            }
        }
        return unspecified;
    }

    /**
     * @param arg Name of argument to check for.
     * @return True if the argument is found.
     */
    public boolean doesArgumentExist(final String arg)
    {
        return (_arguments.getParameterWithName(arg) != null);
    }

    public boolean isArgumentValueUndefined(final String value)
    {
        return ((value.length() == 0) || (value.equals(getUndefinedArgumentValue())));
    }

    /**
     * Call to change the character used to separate arguments immediately after construction and initialization.
     * 
     * @param argumentChar The new argument char, replacing {@link #_argumentChar}.
     */
    public void setArgumentChar(final char argumentChar)
    {
        _argumentChar = argumentChar;
    }

    protected char getArgumentChar()
    {
        return _argumentChar;
    }

    /**
     * Note that, when this is called, its assumed that the argument is needed somewhere and is recorded in the list of
     * accessed (needed) arguments.
     * 
     * @param arg The argument to get.
     * @return GenericParameter containing argument name and value. Note that if the argument is found in the _arguments
     *         list, its value cannot be empty. However, if the argument is a predefined argument, it can be empty.
     */
    public GenericParameter getArgument(final String arg)
    {
        return getArgument(arg, false);
    }

    /**
     * Note that, when this is called, its assumed that the argument is needed somewhere and is recorded in the list of
     * accessed (needed) arguments.
     * 
     * @param arg The argument to get.
     * @param ignorePredefinedArguments If true, the predefined args are ignored. This is useful if you want to limit
     *            your search to overridden arguments only.
     * @return GenericParameter containing argument name and value. Note that if the argument is found in the _arguments
     *         list, its value cannot be empty. However, if the argument is a predefined argument, it can be empty.
     */
    public GenericParameter getArgument(final String arg, final boolean ignorePredefinedArguments)
    {
        GenericParameter param = _arguments.getParameterWithName(arg);

        //If parameter is not found, try to get it from the _higherLevelProcessor.  If still not
        //found create a new parameter using the _predefinedArguments list.
        if(param == null)
        {
            if(this._higherLevelProcessor != null)
            {
                param = _higherLevelProcessor.getArgument(arg);
            }
            else if(!ignorePredefinedArguments)
            {
                param = new GenericParameter(arg, getPredefinedArguments().getDefaultValue(arg));
            }
            else
            {
                param = new GenericParameter(arg, null);
            }
            if((param == null) || (param.getValue() == null)) //param == null can occur when _higherLevelProcessor != null.
            {
                if(isRecording())
                {
                    _argumentsAccessedThusFar.addParameter(arg, null);
                }
                return null;
            }
        }
        if(isRecording())
        {
            _argumentsAccessedThusFar.addParameter(param);
        }
        return param;
    }

    public UniqueGenericParameterList getArguments()
    {
        return _arguments;
    }

    public UniqueGenericParameterList getAllArgumentsIncludingPredefined()
    {
        final UniqueGenericParameterList results = new UniqueGenericParameterList();
        for(int i = 0; i < this.getPredefinedArguments().getListOfPredefinedArgumentNames().size(); i++)
        {
            final String name = getPredefinedArguments().getListOfPredefinedArgumentNames().get(i);
            results.addParameter(name, getPredefinedArguments().getDefaultValue(name));
        }
        for(int i = 0; i < this._arguments.getParameters().size(); i++)
        {
            final GenericParameter parm = _arguments.getParameters().get(i);
            results.addParameter(parm);
        }
        return results;
    }

    public String getDescriptiveName()
    {
        return "Arguments";
    }

    /**
     * @return A String which indicates that the argument is undefined.
     */
    public abstract String getUndefinedArgumentValue();

    /**
     * @return A populated {@link PredefinedArguments} object, supplied by a subclass to be used with this.
     */
    protected abstract PredefinedArguments getPredefinedArguments();

    /**
     * @return Allowable function names.
     */
    protected abstract String[] getFunctionNames();

    /**
     * @param functionName Name of a function
     * @return Array of strings specifying parameter descriptors for the expected parameters.
     */
    protected abstract String[] getFunctionParameterNames(String functionName);

    /**
     * Any subclass should override this method if it wishes to specify function values.
     * 
     * @param argument An Argument object containing a name and function parameters.
     * @return The string representing the value of the argument. Do not include any arguments in the returned String.
     */
    protected abstract String evaluateFunctionValue(Argument argument);

    /**
     * @param functionName The name of the argument for which to build a function parameter editing panel.
     * @return
     */
    public abstract ArgumentFunctionParameterEditingPanel buildEditingPanel(String functionName);

    /**
     * @return The provided parameter name surrounded by {@link #DEFAULT_ARGUMENT_CHAR}.
     */
    public static String argumentizeParameterName(final String parameterName)
    {
        return DEFAULT_ARGUMENT_CHAR + parameterName + DEFAULT_ARGUMENT_CHAR;
    }

    /**
     * @return The provided parameter name surrounded by the specified argument character.
     */
    public static String argumentizeParameterName(final String parameterName, final char argumentChar)
    {
        return argumentChar + parameterName + argumentChar;
    }
}
