package ohd.hseb.hefs.utils.arguments;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.plugins.UniqueGenericParameterList;
import ohd.hseb.util.misc.MiscTools;

/**
 * Default implementation of ArgumentsProcessor in which the evaluateFunctionValue always returns null.
 * 
 * @author herrhd
 */
public class DefaultArgumentsProcessor extends ArgumentsProcessor
{

    private DefaultPredefinedArguments _predefinedArguments = null;

    private final List<String> _functionNames = new ArrayList<String>();

    /**
     * Empty constructor calls {@link #initialize(UniqueGenericParameterList, ProductSpecificArguments)} with an empty
     * {@link UniqueGenericParameterList} and empty {@link ProductSpecificArguments}. It then sets
     * {@link #_predefinedArguments} to be an empty {@link DefaultPredefinedArguments}.<br>
     * <br>
     * This is useful if you wish to define an {@link ArgumentsProcessor} on the fly by constructing an empty instance
     * and adding arguments. To add an argument, call {@link UniqueGenericParameterList#addParameter(String, String)} on
     * the return from the method {@link #getArguments()}.
     */
    public DefaultArgumentsProcessor()
    {
        initialize(new UniqueGenericParameterList(), new ProductSpecificArguments());
        _predefinedArguments = new DefaultPredefinedArguments();
    }

    /**
     * @param arguments List of arguments present.
     * @param requiredArguments List of product specific arguments, each of which is required.
     * @param predefinedArgs Predefined arguments that the product specific arguments may override.
     */
    public DefaultArgumentsProcessor(final UniqueGenericParameterList arguments,
                                     final ProductSpecificArguments requiredArguments,
                                     final DefaultPredefinedArguments predefinedArgs)
    {
        initialize(arguments, requiredArguments);
        _predefinedArguments = predefinedArgs;
    }

    /**
     * @param arguments List of arguments present.
     * @param requiredArguments List of product specific arguments, each of which is required.
     * @param predefinedArgs Predefined arguments that the product specific arguments may override.
     * @param higherLevelProcessor See ArgumentsProcessor; the higher level processor that will be searched for
     *            arguments if not found in this.
     */
    protected DefaultArgumentsProcessor(final UniqueGenericParameterList arguments,
                                        final ProductSpecificArguments requiredArguments,
                                        final DefaultPredefinedArguments predefinedArgs,
                                        final ArgumentsProcessor higherLevelProcessor)
    {
        initialize(arguments, requiredArguments, higherLevelProcessor);
        _predefinedArguments = predefinedArgs;
    }

    public void clearMemory()
    {
    }

    protected void addFunctionName(final String name)
    {
        _functionNames.add(name);
    }

    public static DefaultArgumentsProcessor createEmpty()
    {
        return new DefaultArgumentsProcessor(new UniqueGenericParameterList(),
                                             new ProductSpecificArguments(),
                                             new DefaultPredefinedArguments());
    }

    @Override
    protected String evaluateFunctionValue(final Argument argument)
    {
        return null;
    }

    @Override
    protected PredefinedArguments getPredefinedArguments()
    {
        return _predefinedArguments;
    }

    @Override
    protected String[] getFunctionNames()
    {
        return MiscTools.stringArrayFromList(_functionNames);
    }

    @Override
    protected String[] getFunctionParameterNames(final String name)
    {
        return null;
    }

    @Override
    public String getUndefinedArgumentValue()
    {
        return "";
    }

    @Override
    public ArgumentFunctionParameterEditingPanel buildEditingPanel(final String argumentName)
    {
        return null;
    }
}
