package ohd.hseb.hefs.utils.arguments;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.plugins.DefaultPlugInParameters;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.plugins.UniqueGenericParameterList;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.util.misc.HString;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

/**
 * List of arguments specific for a product. Concept is used in Graphics Generator, but may be applicable elsewhere.
 * These arguments are intended to override elements in the PredefinedArguments list, or be fore new arguments that are
 * not predefined.
 * 
 * @author hank.herr
 */
public class ProductSpecificArguments extends DefaultPlugInParameters
{

    /**
     * The names of arguments that must be specified for this product.
     */
    private final List<String> _requiredArguments = new ArrayList<String>();

    public ProductSpecificArguments()
    {
        setXMLTagName("requiredArguments");
    }

    public void clearParameters()
    {
        this._requiredArguments.clear();
    }

    public void addRequiredArgument(final String arg)
    {
        this._requiredArguments.add(arg);
    }

    public void removeRequiredArgument(final String arg)
    {
        this._requiredArguments.remove(arg);
    }

    public int getNumberOfArguments()
    {
        return this._requiredArguments.size();
    }

    public List<String> getRequiredArguments()
    {
        return this._requiredArguments;
    }

    public String getRequiredArgument(final int index)
    {
        return this._requiredArguments.get(index);
    }

    public boolean isArgumentRequired(final String arg)
    {
        return (this._requiredArguments.indexOf(arg) >= 0);
    }

    public void areAllRequiredArgumentsPresent(final UniqueGenericParameterList arguments,
                                               final PredefinedArguments predefinedArguments) throws RequiredArgumentsException
    {
        for(int i = 0; i < _requiredArguments.size(); i++)
        {
            if((!arguments.doesParameterExist(_requiredArguments.get(i)))
                && (!predefinedArguments.isPredefinedArgumentDefined(_requiredArguments.get(i))))
            {
                throw new RequiredArgumentsException("Required argument '" + _requiredArguments.get(i) + "' not found.");
            }

        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return "Required Args: " + HString.buildStringFromList(_requiredArguments, ",");
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            this.clearParameters();
        }
        else if(!elementName.equals("argument"))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '"
                + elementName + "'.");
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
        }
        else if(elementName.equals("argument"))
        {
            this._requiredArguments.add(value);
        }
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());

        for(int i = 0; i < this._requiredArguments.size(); i++)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "argument", _requiredArguments.get(i)));
        }

        return mainElement;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public Object clone()
    {
        final ProductSpecificArguments results = new ProductSpecificArguments();
        results.copyFrom(this);
        return results;
    }

    @Override
    public boolean equals(final Object obj)
    {
        final ProductSpecificArguments other = (ProductSpecificArguments)obj;
        if(this.getNumberOfArguments() != other.getNumberOfArguments())
        {
            return false;
        }
        for(int i = 0; i < getNumberOfArguments(); i++)
        {
            if(other.getRequiredArguments().indexOf(_requiredArguments.get(i)) < 0)
            {
                return false;
            }
        }
        for(int i = 0; i < getNumberOfArguments(); i++)
        {
            if(_requiredArguments.indexOf(other.getRequiredArguments().get(i)) < 0)
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString()
    {
        String results = "RequiredArgumentsParameters: ";
        results += "xmlTagName = " + this.getXMLTagName() + "; ";
        results += "arguments = {";
        for(int i = 0; i < this._requiredArguments.size(); i++)
        {
            results += "argument" + i + " = " + this._requiredArguments.get(i);
            if(i < this._requiredArguments.size() - 1)
            {
                results += "; ";
            }
        }
        results += "}.";
        return results;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        final ProductSpecificArguments base = (ProductSpecificArguments)parameters;
        this.clearParameters();
        for(int i = 0; i < base.getRequiredArguments().size(); i++)
        {
            addRequiredArgument(base.getRequiredArgument(i));
        }
    }

}
