package ohd.hseb.hefs.utils.collect;

import java.awt.Component;
import java.util.Collection;
import java.util.List;
import java.util.NavigableSet;

import javax.swing.JList;
import javax.swing.ListModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import ohd.hseb.hefs.utils.SplitCollection;
import ohd.hseb.hefs.utils.gui.components.SplitListModel;
import ohd.hseb.hefs.utils.gui.tools.ListCellRendererModifier;
import ohd.hseb.hefs.utils.notify.collect.CollectionModifiedNotice;
import ohd.hseb.hefs.utils.notify.collect.NotifyingCollection;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ForwardingCollection;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * Wraps a {@link Collection} as a {@link ListModel} as well, keeping track of changes to the collection for you.<br/>
 * If this receives a {@link ListDataEvent}, it will send all listeners a new event indicating all values have changed.
 * This is for internal use with the {@link #filter(Predicate)} function, so filtered views respond to changes in this.<br/>
 * If the supplied {@code delegate} implements {@link NotifyingCollection}, then this will automatically be notified
 * when it updates.
 * 
 * @author alexander.garbarino
 * @param <E> the element this collection holds
 */
public class CollectionListModel<E> extends ForwardingCollection<E> implements ListModel, ListDataListener,
ListCellRendererModifier
{
    private final Collection<E> _delegate;
    private final List<ListDataListener> _listeners;
    private final List<ListCellRendererModifier> _modifiers;

    public CollectionListModel(final Collection<E> delegate)
    {
        this(delegate, null);
    }

    protected CollectionListModel(final Collection<E> delegate, final List<ListCellRendererModifier> modifiers)
    {
        _delegate = delegate;
        _listeners = Lists.newArrayList();
        if(modifiers == null)
        {
            _modifiers = Lists.newArrayList();
        }
        else
        {
            _modifiers = modifiers;
        }

        if(_delegate instanceof NotifyingCollection<?>)
        {
            ((NotifyingCollection)_delegate).register(new CollectionModifiedNotice.Subscriber()
            {
                @Override
                @Subscribe
                public void reactToCollectionModified(final CollectionModifiedNotice evt)
                {
                    notifyChanged();
                }
            });
        }
    }

    @Override
    public int getSize()
    {
        return super.size();
    }

    @Override
    public E getElementAt(final int index)
    {
        return Iterators.get(this.iterator(), index);
    }

    @Override
    public void addListDataListener(final ListDataListener l)
    {
        _listeners.add(l);
    }

    @Override
    public void removeListDataListener(final ListDataListener l)
    {
        _listeners.remove(l);
    }

    @Override
    protected Collection<E> delegate()
    {
        return _delegate;
    }

    @Override
    public boolean add(final E element)
    {
        if(super.add(element))
        {
            notifyChanged(element);
            return true;
        }
        return false;
    }

    @Override
    public boolean addAll(final Collection<? extends E> collection)
    {
        if(super.addAll(collection))
        {
            notifyChanged();
            return true;
        }
        return false;
    }

    @Override
    public void clear()
    {
        if(!this.isEmpty())
        {
            super.clear();
            notifyChanged();
        }
    }

    @Override
    public boolean remove(final Object object)
    {
        if(super.remove(object))
        {
            notifyChanged();
            return true;
        }
        return false;
    }

    @Override
    public boolean removeAll(final Collection<?> collection)
    {
        if(super.removeAll(collection))
        {
            notifyChanged();
            return true;
        }
        return false;
    }

    @Override
    public boolean retainAll(final Collection<?> collection)
    {
        if(super.retainAll(collection))
        {
            notifyChanged();
            return true;
        }
        return false;
    }

    public Class<? extends Collection> getDelegateClass()
    {
        return _delegate.getClass();
    }

    /**
     * Returns a filtered view of this model. The view will have its own set of listeners which only respond to changes
     * visible to the filter.<br/>
     * See {@link Collections2}{@link #filter(Predicate)}.
     * 
     * @param predicate the filter predicate
     * @return a new view of this collection restricted by the given predicate
     */
    public CollectionListModel<E> filter(final Predicate<E> predicate)
    {
        final CollectionListModel<E> model = new CollectionListModel<E>(Collections2.filter(this, predicate),
                                                                        Lists.newArrayList(_modifiers));
        this.addListDataListener(model);
        return model;
    }

    public <T> CollectionListModel<T> transform(final Function<E, T> function)
    {
        final CollectionListModel<T> model = new CollectionListModel<T>(Collections2.transform(this, function),
                                                                        Lists.newArrayList(_modifiers));
        this.addListDataListener(model);
        return model;
    }

    public SplitListModel<E> split(final Predicate<E> predicate)
    {
        final SplitListModel<E> model = new SplitListModel<E>(new SplitCollection<E>(_delegate, predicate),
                                                              Lists.newArrayList(_modifiers));
        this.addListDataListener(model);
        return model;
    }

    /**
     * Informs listeners that all values may have changed.
     */
    public void notifyChanged()
    {
        notifyChanged(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, 0, Integer.MAX_VALUE));
    }

    /**
     * Notify that the specified element has changed.
     * 
     * @param element the element that has changed
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public void notifyChanged(final E element)
    {
        Integer index = null;
        if(_delegate instanceof List<?>)
        {
            index = ((List<E>)_delegate).indexOf(element);
        }
        else if(_delegate instanceof NavigableSet<?>)
        {
            index = ((NavigableSet)_delegate).headSet(element).size();
        }
        if(index == null)
        {
            notifyChanged(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, 0, Integer.MAX_VALUE));
        }
        else if(index != -1)
        {
            notifyChanged(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, index, index));
        }
    }

    /**
     * Inform listeners that a change occurred.
     */
    private void notifyChanged(final ListDataEvent event)
    {
        for(final ListDataListener listener: _listeners)
        {
            listener.contentsChanged(event);
        }
    }

    @Override
    public void intervalAdded(final ListDataEvent e)
    {
        notifyChanged();
    }

    @Override
    public void intervalRemoved(final ListDataEvent e)
    {
        notifyChanged();
    }

    @Override
    public void contentsChanged(final ListDataEvent e)
    {
        notifyChanged();
    }

    @Override
    public Component modifyListCellRendererComponent(Component component,
                                                     final JList list,
                                                     final Object value,
                                                     final int index,
                                                     final boolean isSelected,
                                                     final boolean cellHasFocus)
    {
        for(final ListCellRendererModifier modifier: _modifiers)
        {
            component = modifier.modifyListCellRendererComponent(component,
                                                                 list,
                                                                 value,
                                                                 index,
                                                                 isSelected,
                                                                 cellHasFocus);
        }
        return component;
    }
}
