package ohd.hseb.hefs.utils.collect;

import java.util.Collection;
import java.util.Iterator;

import ohd.hseb.hefs.utils.Decorator;
import ohd.hseb.hefs.utils.notify.NoticeForwarder;
import ohd.hseb.hefs.utils.notify.Notifier;

import com.google.common.collect.ForwardingIterator;
import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import com.google.common.eventbus.Subscribe;

/**
 * Collection decorator to automatically, if applicable, {@link Notifier#register(Object)} and
 * {@link Notifier#unregister(Object)} itself with its elements as they are added and removed, respectively.<br/>
 * Note that <i>this</i> object is registered, not the {@link #delegate()}. To get around this, this object itself
 * implements {@link Notifier}, and passes any events received from its elements to its listeners.<br/>
 * Any exceptions from {@link Notifier#unregister(Object) unregister} are quietly consumed.<br/>
 * As a decorator, this also forwards any notices received from its delegate.
 * 
 * @author alexander.garbarino
 */
public class ElementListeningCollection<E, C extends Collection<E>> extends NoticeForwarder implements Decorator<C>,
Collection<E>
{
    private final C _delegate;

    public ElementListeningCollection(C delegate)
    {
        _delegate = delegate;
        Notifier.Tools.tryRegister(_delegate, this);
    }

    @Override
    public C delegate()
    {
        return _delegate;
    }

    protected void registerSelf(E e)
    {
        if(e instanceof Notifier)
        {
            ((Notifier)e).register(this);
        }
    }

    protected void registerSelf(Collection<? extends E> c)
    {
        for(E e: c)
        {
            registerSelf(e);
        }
    }

    protected void unregisterSelf(Object o)
    {
        if(o instanceof Notifier)
        {
            try
            {
                ((Notifier)o).unregister(this);
            }
            catch(Exception ex)
            {
                // Silently ignore.
            }
        }
    }

    protected void unregisterSelf(Collection<?> c)
    {
        for(Object o: c)
        {
            unregisterSelf(o);
        }
    }

    @Override
    public boolean add(E element)
    {
        boolean added = _delegate.add(element);
        if(added)
        {
            registerSelf(element);
        }
        return added;
    }

    @Override
    public boolean remove(Object object)
    {
        boolean removed = _delegate.remove(object);
        if(removed)
        {
            unregisterSelf(object);
        }
        return removed;
    }

    @Override
    public boolean addAll(Collection<? extends E> collection)
    {
        boolean added = _delegate.addAll(collection);
        if(added)
        {
            registerSelf(collection);
        }
        return added;
    }

    @Override
    public boolean removeAll(Collection<?> collection)
    {
        boolean removed = _delegate.removeAll(collection);
        if(removed)
        {
            unregisterSelf(collection);
        }
        return removed;
    }

    @Override
    public boolean retainAll(Collection<?> collection)
    {
        for(E e: this)
        {
            if(!collection.contains(e))
            {
                unregisterSelf(e);
            }
        }
        return _delegate.retainAll(collection);
    }

    @Override
    public void clear()
    {
        for(E e: this)
        {
            unregisterSelf(e);
        }
        _delegate.clear();
    }

    @Subscribe
    public void reactToEvent(Object event)
    {
        post(event);
    }

    @Override
    public int size()
    {
        return _delegate.size();
    }

    @Override
    public boolean isEmpty()
    {
        return _delegate.isEmpty();
    }

    @Override
    public boolean contains(Object o)
    {
        return _delegate.contains(o);
    }

    @Override
    public Iterator<E> iterator()
    {
        return new ElementListeningIterator();
    }

    @Override
    public Object[] toArray()
    {
        return _delegate.toArray();
    }

    @Override
    public <T> T[] toArray(T[] a)
    {
        return _delegate.toArray(a);
    }

    @Override
    public boolean containsAll(Collection<?> c)
    {
        return _delegate.containsAll(c);
    }

    private class ElementListeningIterator extends ForwardingIterator<E>
    {
        private final PeekingIterator<E> _delegateIterator = Iterators.peekingIterator(_delegate.iterator());

        @Override
        protected Iterator<E> delegate()
        {
            return _delegateIterator;
        }

        @Override
        public void remove()
        {
            unregisterSelf(_delegateIterator.peek());
            super.remove();
        }

    }
}
