package ohd.hseb.hefs.utils.collect;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.ListIterator;

import ohd.hseb.hefs.utils.notify.Notifier;

import com.google.common.collect.ForwardingListIterator;

/**
 * List decorator to automatically, if applicable, {@link Notifier#register(Object)} and
 * {@link Notifier#unregister(Object)} itself with its elements as they are added and removed, respectively.<br/>
 * Note that <i>this</i> object is registered, not the {@link #delegate()}. To get around this, this object itself
 * implements {@link Notifier}, and passes any events received from its elements to its listeners.<br/>
 * Any exceptions from {@link Notifier#unregister(Object) unregister} are quietly consumed.
 * 
 * @author alexander.garbarino
 */
public class ElementListeningList<E, C extends List<E>> extends ElementListeningCollection<E, C> implements List<E>
{
    public static <E> ElementListeningList<E, ArrayList<E>> create()
    {
        return create(new ArrayList<E>());
    }

    public static <E, C extends List<E>> ElementListeningList<E, C> create(final C delegate)
    {
        return new ElementListeningList<E, C>(delegate);
    }

    public ElementListeningList(final C delegate)
    {
        super(delegate);
    }

    @Override
    public boolean addAll(final int index, final Collection<? extends E> c)
    {
        final boolean added = delegate().addAll(index, c);
        registerSelf(c);
        return added;
    }

    @Override
    public E get(final int index)
    {
        return delegate().get(index);
    }

    @Override
    public E set(final int index, final E element)
    {
        final E result = delegate().set(index, element);
        unregisterSelf(result); // In this order so it works if result == element.
        registerSelf(element);
        return result;
    }

    @Override
    public void add(final int index, final E element)
    {
        delegate().add(index, element);
        registerSelf(element);
    }

    @Override
    public E remove(final int index)
    {
        final E result = delegate().remove(index);
        unregisterSelf(result);
        return result;
    }

    @Override
    public int indexOf(final Object o)
    {
        return delegate().indexOf(o);
    }

    @Override
    public int lastIndexOf(final Object o)
    {
        return delegate().lastIndexOf(o);
    }

    @Override
    public ListIterator<E> listIterator()
    {
        return new ElementListeningListIterator(delegate().listIterator());
    }

    @Override
    public ListIterator<E> listIterator(final int index)
    {
        return new ElementListeningListIterator(delegate().listIterator(index));
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    @Override
    public List<E> subList(final int fromIndex, final int toIndex)
    {
        return new ElementListeningList(delegate().subList(fromIndex, toIndex));
    }

    private class ElementListeningListIterator extends ForwardingListIterator<E>
    {
        private final ListIterator<E> _delegate;
        private E _lastElement;

        private ElementListeningListIterator(final ListIterator<E> delegate)
        {
            _delegate = delegate;
        }

        @Override
        protected ListIterator<E> delegate()
        {
            return _delegate;
        }

        @Override
        public E next()
        {
            final E e = super.next();
            _lastElement = e;
            return e;
        }

        @Override
        public E previous()
        {
            final E e = super.previous();
            _lastElement = e;
            return e;
        }

        @Override
        public void remove()
        {
            super.remove();
            unregisterSelf(_lastElement);
            _lastElement = null;
        }

        @Override
        public void set(final E element)
        {
            super.set(element);
            unregisterSelf(_lastElement);
            registerSelf(element);
        }

        @Override
        public void add(final E element)
        {
            super.add(element);
            registerSelf(element);
        }
    }
}
