package ohd.hseb.hefs.utils.collect;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ohd.hseb.charter.translator.plugins.ComparableTools;
import ohd.hseb.hefs.utils.notify.NotifierBase;
import ohd.hseb.hefs.utils.notify.ObjectModifiedNotice;
import ohd.hseb.hefs.utils.tools.IntegerFunctions;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * Groups the elements of an {@link Enum} into different 'groups', which are a 0-indexed integer. Each enum element may
 * be in 0 or 1 groups.
 * 
 * @author alexander.garbarino
 * @param <E> the enum class being contained
 */
public class Grouping<E extends Enum<E>> extends NotifierBase implements Cloneable
{
    private final Class<E> _enumClass;
    private final List<EnumSet<E>> _groups;

    /**
     * @param enumClass
     */
    public Grouping(final Class<E> enumClass)
    {
        _enumClass = enumClass;
        _groups = Lists.newArrayList();
    }

    /**
     * @param enumClass the enum that this will group
     * @param groupCount the number of empty groups to create
     */
    public Grouping(final Class<E> enumClass, final int groupCount)
    {
        _enumClass = enumClass;
        _groups = Lists.newArrayList();
        for(int i = 0; i < groupCount; i++)
        {
            _groups.add(EnumSet.noneOf(_enumClass));
        }
    }

    public static <E extends Enum<E>> Grouping<E> create(final Class<E> enumClass)
    {
        return new Grouping<E>(enumClass);
    }

    /**
     * @return the current number of groups
     */
    public int getGroupCount()
    {
        return _groups.size();
    }

    /**
     * Get all elements in the specified group.
     */
    public Set<E> getGroup(final int group)
    {
        return Collections.unmodifiableSet(_groups.get(group));
    }

    /**
     * Gets the index of the group that {@code element} is in.
     * 
     * @return the group index, or {@code null} if {@code element} is not in a group
     */
    public Integer getGroupIndex(final E element)
    {
        for(int i = 0; i < _groups.size(); i++)
        {
            if(_groups.get(i).contains(element))
            {
                return i;
            }
        }
        return null;
    }

    /**
     * This sets the number of groups to the specified {@code count}. Any newly created groups are empty. If there are
     * too many groups, the largest index groups are discarded, and all elements in those groups become unassigned.
     * 
     * @param count the number of groups to have
     */
    public void setGroupCount(final int count)
    {
        while(_groups.size() < count)
        {
            addGroup();
        }
        while(_groups.size() > count)
        {
            _groups.remove(_groups.size() - 1);
        }
    }

    /**
     * Tests if {@code element} is in {@code group}.
     * 
     * @param group The group to check for {@code element} in. If {@code null} or {@code -1}, instead call
     *            {@link #isNotInGroup(element)}.
     */
    public boolean isInGroup(final E element, final Integer group)
    {
        if(group == null || group == -1)
        {
            return isNotInGroup(element);
        }

        Preconditions.checkElementIndex(group, _groups.size(), "Specified group does not exist.");
        return _groups.get(group).contains(element);
    }

    /**
     * Returns true if {@code element} is not in any groups.
     */
    public boolean isNotInGroup(final E element)
    {
        for(final Set<E> group: _groups)
        {
            if(group.contains(element))
            {
                return false;
            }
        }
        return true;
    }

    private void clearGroups(final E element)
    {
        for(final EnumSet<E> group: _groups)
        {
            group.remove(element);
        }
    }

    private void clearGroups(final Collection<E> elements)
    {
        for(final EnumSet<E> group: _groups)
        {
            group.removeAll(elements);
        }
    }

    /**
     * Set {@code element} to be in the group with index {@code groupIndex}.
     * 
     * @param element the element to set the group for
     * @param groupIndex The group to place {@code element} in. Must be an already existing group. If {@code null} or
     *            {@code -1}, instead call {@link #removeFromGroups(element)}.
     */
    public void setGroup(final E element, final Integer groupIndex)
    {
        if(groupIndex == null || groupIndex == -1)
        {
            removeFromGroups(element);
            return;
        }

        Preconditions.checkElementIndex(groupIndex, _groups.size(), "Specified group does not exist.");
        clearGroups(element);
        _groups.get(groupIndex).add(element);
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Remove {@code element} from all groups.<br/>
     * Equivalent to {@link #setGroup(Enum, Integer) setGroup(element, null)}
     * 
     * @param element the element to remove from all groups
     */
    public void removeFromGroups(final E element)
    {
        clearGroups(element);
    }

    /**
     * Remove {@code elements} from all groups.
     * 
     * @param elements the elements to remove from all groups
     */
    public void removeFromGroups(final Collection<E> elements)
    {
        clearGroups(elements);
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Creates a new, empty group.
     */
    public void addGroup()
    {
        _groups.add(EnumSet.noneOf(_enumClass));
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Creates a new group containing {@code element}, which is removed from its preexisting group, if any.
     * 
     * @param element the element the new group contains
     */
    public void addGroup(final E element)
    {
        clearGroups(element);
        _groups.add(EnumSet.of(element));
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Creates a new group containing {@code elements}, which are removed from their preexisting group, if any.
     * 
     * @param elements the elements the new group contains
     */
    public void addGroup(final Collection<E> elements)
    {
        clearGroups(elements);
        _groups.add(EnumSet.copyOf(elements));
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Removes the group with index {@code groupIndex} and unassigns all elements in that group.
     */
    public void removeGroup(final int groupIndex)
    {
        Preconditions.checkElementIndex(groupIndex, _groups.size(), "Specified group does not exist.");
        _groups.remove(groupIndex);
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * @return the indices of all empty groups
     */
    public Set<Integer> getEmptyGroupIndices()
    {
        final Set<Integer> empties = Sets.newHashSet();
        for(int i = 0; i < _groups.size(); i++)
        {
            if(_groups.get(i).isEmpty())
            {
                empties.add(i);
            }
        }
        return empties;
    }

    public boolean hasEmptyGroup()
    {
        return !getEmptyGroupIndices().isEmpty();
    }

    /**
     * Removes all groups and unassigns all elements.
     */
    public void clear()
    {
        _groups.clear();
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Set to a single group containing every element.
     */
    public void changeToSingleGroup()
    {
        _groups.clear();
        _groups.add(EnumSet.allOf(_enumClass));
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Set to each element having its own group composed of just itself.
     */
    public void changeToIndividualGroups()
    {
        _groups.clear();
        for(final E element: EnumSet.allOf(_enumClass))
        {
            addGroup(element);
        }
        post(new ObjectModifiedNotice<Grouping<E>>(this));
    }

    /**
     * Creates a map of element to group index.
     */
    public Map<E, Integer> asMap()
    {
        final Map<E, Integer> map = Maps.newEnumMap(_enumClass);
        for(final E element: EnumSet.allOf(_enumClass))
        {
            map.put(element, null);
        }
        for(int i = 0; i < _groups.size(); i++)
        {
            for(final E element: _groups.get(i))
            {
                map.put(element, i);
            }
        }
        return map;
    }

    @Override
    public String toString()
    {
        return Joiner.on(',').useForNull("-1").join(asMap().values());
    }

    /**
     * Recreates a grouping from a previous call to {@link #toString()}.
     */
    public static <E extends Enum<E>> Grouping<E> valueOf(final Class<E> enumClass, final String string)
    {
        final Iterable<String> strings = Splitter.on(',').split(string);
        final Iterable<Integer> integers = Iterables.transform(strings, IntegerFunctions.valueOf);
        final int groupCount = ComparableTools.max(integers) + 1;
        final Grouping<E> grouping = new Grouping<E>(enumClass, groupCount);
        final Iterator<E> elements = EnumSet.allOf(enumClass).iterator();
        for(final Integer group: integers)
        {
            final E element = elements.next();
            grouping.setGroup(element, group);
        }
        return grouping;
    }

    /**
     * Sets this grouping to match another, through {@link #toString()}.
     */
    public void setToString(final String string)
    {
        final Grouping<E> other = valueOf(_enumClass, string);
        _groups.clear();
        _groups.addAll(other._groups);
    }

    @Override
    public boolean equals(final Object other)
    {
        if(this == other)
        {
            return true;
        }

        if(!(other instanceof Grouping<?>))
        {
            return false;
        }

        final Grouping<?> that = (Grouping<?>)other;
        return this._enumClass.equals(that._enumClass) && this._groups.equals(that._groups);
    }

    /**
     * Other than {@link #_enumClass}, this is a true clone: the {@link #_groups} object and its elements are not
     * shared.
     */
    @Override
    public Grouping<E> clone()
    {
        final Grouping<E> group = new Grouping<>(_enumClass, _groups.size());
        for(int i = 0; i < _groups.size(); i++)
        {
            group._groups.set(i, _groups.get(i).clone());
        }
        return group;
    }
}
