package ohd.hseb.hefs.utils.collect;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkElementIndex;
import static com.google.common.base.Preconditions.checkPositionIndex;

import java.lang.reflect.Array;
import java.util.AbstractList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

import net.jcip.annotations.Immutable;

import com.google.common.base.Objects;
import com.google.common.base.Supplier;

/**
 * An immutable list of a single element.
 * 
 * @author alexander.garbarino
 * @param <E>
 */
@Immutable
public class SingleList<E> extends AbstractList<E> implements Supplier<E>
{
    private final E _element;

    public SingleList(final E element)
    {
        _element = element;
    }

    @Override
    public int size()
    {
        return 1;
    }

    @Override
    public boolean isEmpty()
    {
        return false;
    }

    @Override
    public boolean contains(final Object o)
    {
        return Objects.equal(_element, o);
    }

    @Override
    public Iterator<E> iterator()
    {
        return new SingleIterator();
    }

    @Override
    public Object[] toArray()
    {
        return new Object[]{_element};
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T[] toArray(final T[] a)
    {
        final T[] array = ((T[])Array.newInstance(a.getClass().getComponentType(), 1));
        array[0] = ((T)_element);
        return array;
    }

    @Override
    public boolean add(final E e)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean remove(final Object o)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsAll(final Collection<?> c)
    {
        for(final Object o: c)
        {
            if(!Objects.equal(o, _element))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean addAll(final Collection<? extends E> c)
    {
        if(c.size() == 0)
        {
            return false;
        }
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean addAll(final int index, final Collection<? extends E> c)
    {
        if(c.size() == 0)
        {
            return false;
        }
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean removeAll(final Collection<?> c)
    {
        for(final Object o: c)
        {
            if(Objects.equal(o, _element))
            {
                throw new UnsupportedOperationException();
            }
        }
        return false;
    }

    @Override
    public boolean retainAll(final Collection<?> c)
    {
        for(final Object o: c)
        {
            if(!Objects.equal(o, _element))
            {
                throw new UnsupportedOperationException();
            }
        }
        return false;
    }

    @Override
    public void clear()
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public E get(final int index)
    {
        checkElementIndex(index, 1);
        return _element;
    }

    @Override
    public E set(final int index, final E element)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public void add(final int index, final E element)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public E remove(final int index)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public int indexOf(final Object o)
    {
        if(Objects.equal(o, _element))
        {
            return 0;
        }
        else
        {
            return -1;
        }
    }

    @Override
    public int lastIndexOf(final Object o)
    {
        return indexOf(o);
    }

    @Override
    public ListIterator<E> listIterator()
    {
        return new SingleIterator();
    }

    @Override
    public ListIterator<E> listIterator(final int index)
    {
        return new SingleIterator(index);
    }

    @Override
    public List<E> subList(final int fromIndex, final int toIndex)
    {
        checkPositionIndex(fromIndex, 1);
        checkPositionIndex(toIndex, 1);
        checkArgument(fromIndex <= toIndex, "Cannot create sublist from %s to %s.", fromIndex, toIndex);
        if(fromIndex == toIndex)
        {
            return new EmptyList<E>();
        }
        else
        {
            return this;
        }
    }

    @Override
    public E get()
    {
        return _element;
    }

    @Override
    public int hashCode()
    {
        if(_element == null)
        {
            return 31;
        }
        else
        {
            return 31 + _element.hashCode();
        }
    }

    private class SingleIterator implements ListIterator<E>
    {
        private int _index;

        private SingleIterator()
        {
            this(0);
        }

        private SingleIterator(final int index)
        {
            checkPositionIndex(index, 1);
            _index = index;
        }

        @Override
        public boolean hasNext()
        {
            return _index == 0;
        }

        @Override
        public E next()
        {
            if(_index == 0)
            {
                _index++;
                return _element;
            }
            else
            {
                throw new NoSuchElementException();
            }
        }

        @Override
        public boolean hasPrevious()
        {
            return _index == 1;
        }

        @Override
        public E previous()
        {
            if(_index == 1)
            {
                _index--;
                return _element;
            }
            else
            {
                throw new NoSuchElementException();
            }
        }

        @Override
        public int nextIndex()
        {
            return _index;
        }

        @Override
        public int previousIndex()
        {
            return _index - 1;
        }

        @Override
        public void remove()
        {
            throw new UnsupportedOperationException();
        }

        @Override
        public void set(final E e)
        {
            throw new UnsupportedOperationException();
        }

        @Override
        public void add(final E e)
        {
            throw new UnsupportedOperationException();
        }
    }
}
