package ohd.hseb.hefs.utils.datetime;

// -*-Java-*-

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.SimpleTimeZone;

import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HNumber;

/**
 * The JulianDay supports turning Gregorian and Julian (Caesar) dates to/from Julius Scaliger's clever Julian Day
 * Numbers. In 1583, Julius had a great idea, and started to count the number of days starting on -4712-01-01 at noon
 * (Z). This is a wonderful "intermediate form" is still used for calendar calculations today! <br>
 * <br>
 * Hank: Wow... what a load of bs. At any rate, I've taken his code and cleaned it up significantly, so that exceptions
 * are no longer thrown in constructors and internal "sanity check" code is removed.
 * 
 * @author Kleanthes Koniaris
 * @author Hank Herr
 */

public class AstronomicalJulianDay
{
    private static final SimpleTimeZone Z_TZ = new SimpleTimeZone(0, "Z");
    private static final double THIRTYPLUS = 30.6001;
    //private static final long ZERO_DAY_MILLIS = -210866673600000L; + 1 day
    //private static final long ZERO_DAY_MILLIS = -210866846400000L; - 1 day
    private static final long ZERO_DAY_MILLIS = -210866760000000L; //0 day, 12Z
    private static final long ZERO_DAY_MILLIS_0Z_HOUR = ZERO_DAY_MILLIS - 12 * HCalendar.MILLIS_IN_HR;

    private double julianDay;
    private final int y; // year
    private final int mo; // month -- STARTS COUNTING FROM 1 (stupied Kleanthes!)
    private int day; // day
    private int h; // hour
    private int m; // minute
    private int s; // second
    private double d; // the day fraction (d <-> {day, h, m, s})

    private final double secondsInDay = 24 * 60 * 60; // how do I declare this constant?

    /**
     * Returns the Julian Day, i.e., the number of days since 12h GMT in the year -4712, aka 4713 B.C.
     */
    public double getJulianDay()
    {
        return julianDay;
    }

    /**
     * Returns the day as a float that also accounts for H, M and S. For example, the middle (noon) of the 3rd day would
     * be represented as 3.5.
     */
    public double getD()
    {
        return d;
    }

    /** Returns the year. */
    public int getYear()
    {
        return y;
    }

    /**
     * Returns the month. Note that this is the Calendar.JANUARY == 0, but we return one for January, so be careful.
     * What were they thinking at Sun to do such a non-standard thing---and for no good reason, as far as I can tell!
     */
    public int getMonth()
    {
        return mo;
    }

    /** Returns the day of the month. */
    public int getDayOfMonth()
    {
        return day;
    }

    /** Returns the day hour (0..23). */
    public int getHour()
    {
        return h;
    }

    /** Returns the minute of the hour (0..59). */
    public int getMinute()
    {
        return m;
    }

    /** Ruterns the second of the hour. */
    public int getSecond()
    {
        return s;
    }

    private void computeD()
    {
        d = day + ((s + 60 * (m + 60 * h))) / secondsInDay;
    }

    /** Turns the current time into a JulianDay. */

    public AstronomicalJulianDay()
    {
        final GregorianCalendar g = new GregorianCalendar(Z_TZ);
        final Date now = new Date();
        g.setTime(now);
        y = g.get(Calendar.YEAR);
        mo = 1 + g.get(Calendar.MONTH); // +1 cuz January is 0 to Sun!
        day = g.get(Calendar.DAY_OF_MONTH);
        h = g.get(Calendar.HOUR_OF_DAY);
        m = g.get(Calendar.MINUTE);
        s = g.get(Calendar.SECOND);
        computeD();
        computeJulianDay();
    }

    /**
     * Makes a new JulianDay, probably by means of adding offsets to JulianDates generated by other means.
     */

    public AstronomicalJulianDay(final double julianDay)
    {
        this.julianDay = julianDay;
        final double jd = julianDay + 0.5;
        final double z = Math.floor(jd);
        final double f = HNumber.roundDouble(jd - z, 9); // a fraction between zero and one; rounding avoids garbage digits due to casting
        double a = z;
        if(z >= 2299161)
        { // (typo in the book)
            final double alpha = Math.floor((z - 1867216.25) / 36524.25);
            a += 1 + alpha - Math.floor(alpha / 4.0);
        }
        final double b = a + 1524;
        final double c = Math.floor((b - 122.1) / 365.25);
        final double dd = Math.floor(365.25 * c);
        final double e = Math.floor((b - dd) / THIRTYPLUS);
        // and then,
        this.d = HNumber.roundDouble(b - dd - Math.floor(THIRTYPLUS * e) + f, 9);
        computeDayHourMinuteSecond();
        mo = (int)(e - ((e < 14) ? 1 : 13));
        // sometimes the year is too high by one
        y = (int)(c - ((mo > 2) ? 4716 : 4715));
    }

    /**
     * Creates a new JulianDay given a year, month, day, hour (24h), minute and second.
     */

    public AstronomicalJulianDay(final int y, final int mo, final int day, final int h, final int m, final int s)
    {
        this.y = y;
        this.mo = mo;
        this.day = day;
        this.h = h;
        this.m = m;
        this.s = s;
        computeD();
        computeJulianDay();
    }

    /**
     * Creates a new JulianDay given a year, month, day, hour (24h), minute and second.
     */

    public AstronomicalJulianDay(final Calendar cal)
    {
        this.y = cal.get(Calendar.YEAR);
        this.mo = cal.get(Calendar.MONTH) + 1;
        this.day = cal.get(Calendar.DAY_OF_MONTH);
        this.h = cal.get(Calendar.HOUR_OF_DAY);
        this.m = cal.get(Calendar.MINUTE);
        this.s = cal.get(Calendar.SECOND);
        computeD();
        computeJulianDay();
    }

    /**
     * Creates a new JulianDay given a year, month (counting from 1!), and day-fraction, where 12.5 means noon on the
     * 12th day of the month.
     */

    public AstronomicalJulianDay(final int y, final int mo, final double d)
    {
        this.y = y;
        this.mo = mo;
        this.d = d;
        computeDayHourMinuteSecond();
        computeJulianDay();
    }

    /** Sets day, hour, minute and second on the basis of d. */

    private void computeDayHourMinuteSecond()
    {
        final int seconds = (int)Math.round(secondsInDay * d); // how many seconds in the day, d
        s = seconds % 60;
        final int minutes = (seconds - s) / 60;
        m = minutes % 60;
        final int hours = (minutes - m) / 60;
        h = hours % 24;
        final int days = (hours - h) / 24;
        day = days;
    }

    /** Is the point in time after 4 October 1582? */

    public boolean dateIsGregorian()
    {
        // 4 October 1582
        if(y > 1582)
            return true;
        if(y < 1582)
            return false;
        // shit, somebody is messing with us!
        if(mo > 10)
            return true;
        if(mo < 10)
            return false;
        // now it is evident that they're fucking with us---or it is
        // my Monte-Carlo testing.  :)
        if(day > 4)
            return true;
        if(day < 4)
            return false;
        return false; // this is probably the last non-Gregorian date?
    }

    /**
     * Returns the Juilan Day, the number of days since 12h GMT in the year -4712, aka 4713 B.C.
     */

    private void computeJulianDay()
    {
        int year = y;
        int month = mo;

        // here we go with Jean Meeus, p. 61 of 2nd edition of Astronomical Algorithms
        switch(month)
        {
            case 1: // January (don't dare say Calendar.JANUARY here, it will be wrong)
            case 2: // February
                year += -1;
                month += 12;
                break;
        }
        double b = 0;
        if(dateIsGregorian())
        {
            final double a = Math.floor((year) / 100.0);
            b = 2 - a + Math.floor(a / 4.0);
        }
        // finally
        julianDay = Math.floor(365.25 * (year + 4716)) + Math.floor(THIRTYPLUS * (month + 1)) + d + b - 1524.5;
    }

    /**
     * This date as a GregorianCalendar object.
     * 
     * @return
     */
    public GregorianCalendar toGregorian()
    {
        final GregorianCalendar g = new GregorianCalendar(Z_TZ);
        g.set(Calendar.YEAR, y);
        g.set(Calendar.MONTH, mo - 1);
        g.set(Calendar.DAY_OF_MONTH, day);
        g.set(Calendar.HOUR_OF_DAY, h);
        g.set(Calendar.MINUTE, m);
        g.set(Calendar.SECOND, s);
        return g;
    }

    /**
     * @return {@link #ZERO_DAY_MILLIS} + {@link #julianDay} * 24 * {@link HCalendar#MILLIS_IN_HR} rounded to a long.
     *         This should handle fractional days. The hour of the day for ZERO_DAY_MILLIS will be 12Z.
     */
    public long toMillis()
    {
        return ZERO_DAY_MILLIS + (long)(this.julianDay * 24 * HCalendar.MILLIS_IN_HR);
    }

    public long toMillis0ZHour()
    {
        return ZERO_DAY_MILLIS_0Z_HOUR + (long)(this.julianDay * 24 * HCalendar.MILLIS_IN_HR);
    }

    /** Returns an ISO-like date to show what's going on in here. */

    @Override
    public String toString()
    {
        return y + "-" + mo + "-" + d + "(" + day + ")" + "--" + h + ":" + m + ":" + s + "--" + julianDay;
    }

    @Override
    public boolean equals(final Object o)
    {
        if(!(o instanceof AstronomicalJulianDay))
        {
            return false;
        }
        final AstronomicalJulianDay other = (AstronomicalJulianDay)o;

        if((getYear() != other.getYear()) || (getMonth() != other.getMonth()) || (getD() != other.getD())
            || (getDayOfMonth() != other.getDayOfMonth()) || (getHour() != other.getHour())
            || (getMinute() != other.getMinute()) || (getSecond() != other.getSecond())
            || (getJulianDay() != other.getJulianDay()))
        {
            return false;
        }
        return true;
    }

    /**
     * @param julianDay Astronomical julian day.
     * @return Milliseconds assuming a 12Z-12Z daily clock.
     */
    public static long computeMillis(final double julianDay)
    {
        return new AstronomicalJulianDay(julianDay).toMillis();
    }

    /**
     * @param julianDay Astronomical julian day.
     * @return Milliseconds assuming a 0Z-0Z daily clock.
     */
    public static long computeMillis0ZHour(final double julianDay)
    {
        return new AstronomicalJulianDay(julianDay).toMillis0ZHour();
    }

    /**
     * @param year
     * @param mon Counting starts at 1! This is how the {@link AstronomicalJulianDay} works.
     * @param day
     * @return The astronomical julian day. In this case, since days is the smallest unit, the return value will be an
     *         integer.
     */
    public static int computeAstronomicalJulianDay(final int year, final int mon, final double day)
    {
        return (int)(new AstronomicalJulianDay(year, mon, day).getJulianDay());
    }

    /**
     * Computes the astronomical whole julian day from milliseconds.
     * 
     * @param millis The hour, minutes, and seconds of the milliseconds are ignored when computing the whole day.
     * @return Astronomical whole day as an integer. This is an int casting of a double julian day computed based on the
     *         given milliseconds.
     */
    public static int computeAstronomicalJulianDay(final long millis)
    {
        final Calendar cal = HCalendar.computeCalendarFromMilliseconds(millis);
        return computeAstronomicalJulianDay(cal.get(Calendar.YEAR),
                                            cal.get(Calendar.MONTH),
                                            cal.get(Calendar.DAY_OF_MONTH) + 1);
    }
}
