package ohd.hseb.hefs.utils.datetime;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseEvent;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.ToolTipManager;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.arguments.DefaultArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.components.ArrowButton;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInPanel;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.plugins.GenericParameter;
import ohd.hseb.hefs.utils.plugins.GenericParameterList;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.util.misc.HCalendar;

import com.google.common.base.Strings;

/**
 * Panel that displays a date and provides tools for editing that date, which can be fixed or relative.
 * 
 * @author Hank.Herr
 */
public class DateComboGenericParameterPanel extends GeneralPlugInPanel implements ActionListener, HDateChooserListener,
ItemListener, FocusListener
{
    private static final long serialVersionUID = 1L;

    public final static String SET_TO_DEFAULT = "Set to Default";
    public final static String SET_TO_FIXED = "Set to Fixed Date";
    public final static String SET_TO_RELATIVE = "Set to Relative Date";
    public final static String SET_TO_TIMESERIES_ENDTIME = "Set to Time Series End Time";
    public final static String SET_TO_TIMESERIES_STARTTIME = "Set to Time Series Start Time";
    public final static String[] DEFAULT_OPERATION_CHOICES = {SET_TO_DEFAULT, SET_TO_FIXED, SET_TO_RELATIVE,
        SET_TO_TIMESERIES_STARTTIME, SET_TO_TIMESERIES_ENDTIME};

    private TimeSeriesArrays _baseTimeSeries = null;
    private ArgumentsProcessor _argumentsProcessor = null;
    private String[] _dateTypeChoices = null;
    private String _defaultValueIfUnchecked = null;
    private final JCheckBox _includeInListCheckBox = HSwingFactory.createJCheckBox("Override Parameter", null, false);
    private final JButton _increaseButton = new ArrowButton(ArrowButton.NORTH, 1, 6);
    private final JButton _decreaseButton = new ArrowButton(ArrowButton.SOUTH, 1, 6);
    private String _nameOfParameterToManage;
    private String[] _timeOptions = null;
    private JComboBox<String> _timeOptionsBox = null;
    private String _selectedTimeOption = null;
    private final JTextField _valueTextField;
    private boolean _disablePanelIfCheckboxUnchecked = false;
    private boolean _arrowButtonAvailable = false;

    /**
     * This version will create a {@link GenericParameterList} to store the set date with a parameter name of
     * "datetime". Call {@link #getValueTextFieldText()} to acquire the date specified as a {@link String}. No base time
     * series will be used, limiting the options available in the {@link #_timeOptionsBox}. This is useful for a general
     * date editing panel that allows for relative dates. Call {@link #getEditingDisplayPanel()} to get a displayable
     * panel.
     * 
     * @param argumentsProcessor An arguments processor that provides arguments to be used. The only important argument
     *            is the system time override.
     * @param defaultValueIfUnchecked The default value for the text field if left unchanged.
     */
    public DateComboGenericParameterPanel(final DefaultArgumentsProcessor argumentsProcessor,
                                          final String defaultValueIfUnchecked)
    {
        this(new GenericParameterList(), "datetime", argumentsProcessor, defaultValueIfUnchecked);
    }

    public DateComboGenericParameterPanel(final GeneralPlugInParameters parameters,
                                          final String nameOfParameterToManage,
                                          final ArgumentsProcessor argumentsProcessor,
                                          final String defaultValueIfUnchecked)
    {
        this(parameters, nameOfParameterToManage, argumentsProcessor, defaultValueIfUnchecked, null, null);
    }

    public DateComboGenericParameterPanel(final GeneralPlugInParameters parameters,
                                          final String nameOfParameterToManage,
                                          final ArgumentsProcessor argumentsProcessor,
                                          final String defaultValueIfUnchecked,
                                          final String selectedOperationChoice)
    {
        this(parameters,
             nameOfParameterToManage,
             argumentsProcessor,
             defaultValueIfUnchecked,
             null,
             selectedOperationChoice);
    }

    public DateComboGenericParameterPanel(final GeneralPlugInParameters parameters,
                                          final String nameOfParameterToManage,
                                          final ArgumentsProcessor argumentsProcessor,
                                          final String defaultValueIfUnchecked,
                                          final String[] operationChoices,
                                          final String selectedOperationChoice)
    {
        this(parameters,
             nameOfParameterToManage,
             argumentsProcessor,
             defaultValueIfUnchecked,
             operationChoices,
             selectedOperationChoice,
             null);
    }

    public DateComboGenericParameterPanel(final GeneralPlugInParameters parameters,
                                          final String nameOfParameterToManage,
                                          final ArgumentsProcessor argumentsProcessor,
                                          final String defaultValueIfUnchecked,
                                          final String[] operationChoices,
                                          final String selectedOperationChoice,
                                          final String[] dateTypeChoices)
    {
        this(parameters,
             nameOfParameterToManage,
             argumentsProcessor,
             defaultValueIfUnchecked,
             operationChoices,
             selectedOperationChoice,
             dateTypeChoices,
             false);
    }

    public DateComboGenericParameterPanel(final GeneralPlugInParameters parameters,
                                          final String nameOfParameterToManage,
                                          final ArgumentsProcessor argumentsProcessor,
                                          final String defaultValueIfUnchecked,
                                          final String[] operationChoices,
                                          final String selectedOperationChoice,
                                          final String[] dateTypeChoices,
                                          final boolean arrowButtonAvailable)
    {
        //Setup the value text field with its tool tip.  This must be done in a constructor because
        //_valueTextField is final.
        _valueTextField = new JTextField(16)
        {
            private static final long serialVersionUID = 1L;

            @Override
            public String getToolTipText(final MouseEvent e)
            {
                final Calendar cal = determineFixedDateValue();
                if(cal == null)
                {
                    return "";
                }
                return HCalendar.buildDateTimeTZStr(cal);
            }
        };
        ToolTipManager.sharedInstance().registerComponent(_valueTextField);

        setManagedParameters(parameters);
        setNameOfParameterToManage(nameOfParameterToManage);
        setArgumentsProcessor(argumentsProcessor);
        setDefaultValueIfUnchecked(defaultValueIfUnchecked);
        setOperationChoices(operationChoices);
        setSelectedOperationChoice(selectedOperationChoice);
        setDateTypeChoices(dateTypeChoices);
        setArrowButtonAvailable(arrowButtonAvailable);

        createDisplay();
        makePanelReflectParameters();
    }

    /**
     * Will call {@link #adjustDecIncButtonsEnabledness(long)} if appropriate giving it the currently selected date.
     * This will do nothing if the buttons are not available or the date field is empty.
     */
    private void adjustButtonStatus()
    {
        //if the arrow buttons are not visible, then skip and return
        if(_arrowButtonAvailable == false)
        {
            return;
        }

        if((getBaseTimeSeries() == null) || getBaseTimeSeries().isEmpty())
        {
            return;
        }

        //Determines the current date selected.
        final String dateString = _valueTextField.getText().trim();
        if(Strings.isNullOrEmpty(dateString))
        {
            return;
        }
        final long currDate = HEFSDateTools.computeTime(getSystemTime(), getBaseTimeSeries().get(0), dateString);

        adjustDecIncButtonsEnabledness(currDate);
    }

    /**
     * Adjust the enabledness of {@link #_decreaseButton} and {@link #_increaseButton} based on the current date
     * provided as compared with the return value of {@link #getTimeSeriesStartTime()} and
     * {@link #getTimeSeriesEndTime()}.
     * 
     * @param currDate The date to use for comparison.
     */
    private void adjustDecIncButtonsEnabledness(final long currDate)
    {
        if(currDate > getTimeSeriesStartTime())
        {
            this._decreaseButton.setEnabled(true);
        }
        else
        {
            this._decreaseButton.setEnabled(false);
        }

        if(currDate < getTimeSeriesEndTime())
        {
            this._increaseButton.setEnabled(true);
        }
        else
        {
            this._increaseButton.setEnabled(false);
        }

    }

    /**
     * Converts the time provided as an argument to a long value using the {@link HEFSDateTools} functionality to handle
     * relative times.
     * 
     * @return {@link Long} specifying the date or null if a problem was encountered.
     */
    private Long convertStringToDate(final String dateString)
    {
        Long dateValue = null;

        if((getBaseTimeSeries() == null) || getBaseTimeSeries().isEmpty())
        {
            dateValue = HEFSDateTools.computeTime(getSystemTime(), null, dateString);
        }
        else
        {
            dateValue = HEFSDateTools.computeTime(getSystemTime(), getBaseTimeSeries().get(0), dateString);
        }

        return dateValue;
    }

    /**
     * Sets up the GUI components. Call {@link #getEditingDisplayPanel()} to get the panel to display.
     */
    private void createDisplay()
    {
        //Wrapped in html so that it can be disabled while leaving the text black.
        _includeInListCheckBox.setText("<html>Override " + _nameOfParameterToManage + "</html>");

        // _valueTextField.setEditable(false);
        _valueTextField.addActionListener(this);
        _valueTextField.addFocusListener(this);

        this._timeOptionsBox = HSwingFactory.createJComboBox(getOperationChoices(), null);
        if(getSelectedOperationChoice() != null)
        {
            this._timeOptionsBox.setSelectedItem(getSelectedOperationChoice());
        }

        this._timeOptionsBox.addItemListener(this);
        this._timeOptionsBox.addActionListener(this);

        this._increaseButton.addActionListener(this);
        this._decreaseButton.addActionListener(this);
    }

    /**
     * @param b True if the panel enabledness is to depend upon the checkbox {@link #_includeInListCheckBox} being
     *            checked. False if not, which is the default behavior.
     */
    public void setDisablePanelIfCheckboxUnchecked(final boolean b)
    {
        _disablePanelIfCheckboxUnchecked = b;
        determineEnablednessBasedOnCheckbox();
    }

    /**
     * @return Treu if the panel enabledness is to depend upon the {@link #_includeInListCheckBox}.
     */
    public boolean disablePanelIfCheckboxUnchecked()
    {
        return _disablePanelIfCheckboxUnchecked;
    }

    /**
     * Disables the {@link #_includeInListCheckBox}.
     */
    public void disableCheckBox()
    {
        this._includeInListCheckBox.setEnabled(false);
    }

    /**
     * @param type Type to check for.
     * @return True if {@link #getOperationChoices()} includes the provided type.
     */
    private boolean doOperationsIncludeType(final String type)
    {
        for(final String oper: this.getOperationChoices())
        {
            if(oper.equals(type))
            {
                return true;
            }
        }
        return false;
    }

    public boolean isArrowButtonAvailable()
    {
        return _arrowButtonAvailable;
    }

    public void makePanelReflectParameters()
    {
        _includeInListCheckBox.removeItemListener(this);
        _timeOptionsBox.removeItemListener(this);
        _timeOptionsBox.removeActionListener(this);

        final GenericParameter existingParameter = getParameters().getParameterWithName(this._nameOfParameterToManage);
        if(existingParameter != null)
        {
            _includeInListCheckBox.setSelected(true);
            _valueTextField.setText(existingParameter.getValue());
            final String dateType = HEFSDateTools.computeRelativeDateBasisStr(existingParameter.getValue());

            if(dateType == null)
            {
                if(doOperationsIncludeType(SET_TO_DEFAULT))
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_DEFAULT);
                }
            }
            else if(dateType.equals(HEFSDateTools.FIXED_TIME))
            {
                if(doOperationsIncludeType(SET_TO_FIXED))
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_FIXED);
                }
            }
            else if(dateType.equals(HEFSDateTools.SYSTEM_TIME_BASIS_STR))
            {
                if(doOperationsIncludeType(SET_TO_DEFAULT))
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_DEFAULT);
                }
                else
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_RELATIVE);
                }
            }
            else if(dateType.equals(HEFSDateTools.TIMESERIES_START_TIME_BASIS_STR))
            {
                if(doOperationsIncludeType(SET_TO_TIMESERIES_STARTTIME))
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_TIMESERIES_STARTTIME);
                }
                else
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_RELATIVE);
                }
            }
            else if(dateType.equals(HEFSDateTools.TIMESERIES_END_TIME_BASIS_STR))
            {
                if(doOperationsIncludeType(SET_TO_TIMESERIES_ENDTIME))
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_TIMESERIES_ENDTIME);
                }
                else
                {
                    this._timeOptionsBox.setSelectedItem(SET_TO_RELATIVE);
                }
            }
            else
            {
                this._timeOptionsBox.setSelectedItem(SET_TO_RELATIVE);
            }

        }
        else
        {
            _includeInListCheckBox.setSelected(false);
            _valueTextField.setText(getDefaultValueIfUnchecked());
        }

        _timeOptionsBox.addItemListener(this);
        _timeOptionsBox.addActionListener(this);

        _includeInListCheckBox.addItemListener(this);
    }

    /**
     * Update the edited parameters, returned by {@link #getParameters()}, to match the panel settings.
     */
    protected void makeParametersReflectPanel()
    {
        if(!this._includeInListCheckBox.isSelected())
        {
            getParameters().removeParameters(this._nameOfParameterToManage);
        }
        else
        {
            if(getParameters().getParameterWithName(this._nameOfParameterToManage) == null)
            {
                getParameters().addParameter(this._nameOfParameterToManage, this._valueTextField.getText());
            }
            else
            {
                getParameters().getParameterWithName(this._nameOfParameterToManage)
                               .setValue(this._valueTextField.getText());
            }
        }
    }

    /**
     * Processes a date adjust performed via {@link #_decreaseButton} or {@link #_increaseButton} actions.
     * 
     * @param factor
     */
    private void processAdjustDate(final int factor)
    {
        Long origDate = null;

        String dateString = _valueTextField.getText().trim();

        // default to system time
        if(dateString.length() <= 0)
        {
            dateString = "T0";
        }

        final Long currentDate = convertStringToDate(dateString);

        boolean isFixedDate = false;
        final String dateType = dateString.split(" ")[0];

        if((HEFSDateTools.computeFixedDate(dateString)) != null)
        {
            isFixedDate = true;
        }

        if(currentDate != null)
        {
            Date nextDate = null;
            //Default is 6 hours in the factor direction.
            if((getBaseTimeSeries() == null) || getBaseTimeSeries().isEmpty())
            {
                nextDate = HCalendar.computeCalendarFromMilliseconds(currentDate + factor * 6 * HCalendar.MILLIS_IN_HR)
                                    .getTime();
            }
            //Otherwise, find the time series index that is next and go to it.
            else
            {
                final TimeSeriesArray baseTS = getBaseTimeSeries().get(0);
                int currentIndex = baseTS.closestIndexOfTime(currentDate);

                // case 1:
                // the current date is above the last time series,
                // then set the next date to the last time series

                if(currentDate > baseTS.getTime(baseTS.size() - 1))
                {
                    nextDate = HCalendar.computeCalendarFromMilliseconds(baseTS.getTime(baseTS.size() - 1)).getTime();
                }

                // case 2:
                // the current date is below the first time series,
                // then set the next date to the first time series
                else if(currentDate < baseTS.getTime(0))
                {
                    nextDate = HCalendar.computeCalendarFromMilliseconds(baseTS.getTime(0)).getTime();
                }

                // case 3:
                // the current date is between two time series,
                // then set the next date to the closest lower/upper time series
                // based on the arrow button 
                else if(((factor > 0) && (currentDate < baseTS.getTime(currentIndex)))
                    || ((factor < 0) && (currentDate > baseTS.getTime(currentIndex))))
                {
                    nextDate = HCalendar.computeCalendarFromMilliseconds(baseTS.getTime(currentIndex)).getTime();
                }

                // case 4:
                // the current date has a same date value of a time series,
                // then set the next date to the previous/next time series
                // based on the arrow button 
                else
                {
                    if(((factor > 0) && (currentIndex < baseTS.size() - 1)) || ((factor < 0) && (currentIndex > 0)))
                    {
                        if(factor > 0)
                        {
                            currentIndex++;
                        }
                        else
                        {
                            currentIndex--;
                        }
                    }
                    nextDate = HCalendar.computeCalendarFromMilliseconds(baseTS.getTime(currentIndex)).getTime();
                }
            }

            //adjust the increase/decrease button status
            final long nextMillis = HCalendar.computeCalendarFromDate(nextDate).getTimeInMillis();
            adjustDecIncButtonsEnabledness(nextMillis);

            String value = "";

            if(isFixedDate == true)
            {
                value = HCalendar.buildDateStr(HCalendar.computeCalendarFromDate(nextDate),
                                               HEFSDateTools.getGUIDateTimeFormat());
            }
            else
            {
                if((getBaseTimeSeries() == null) || getBaseTimeSeries().isEmpty())
                {
                    origDate = HEFSDateTools.computeTime(getSystemTime(), null, dateType);
                }
                else
                {
                    origDate = HEFSDateTools.computeTime(getSystemTime(), getBaseTimeSeries().get(0), dateType);
                }

                final Date prevDate = HCalendar.computeCalendarFromMilliseconds(origDate).getTime();

                value = HEFSDateTools.buildFormattedRelativeDate(prevDate, nextDate, dateType);
            }

            _valueTextField.setText(value);
        }

        updateTimeOptionsBox();

        this.makeParametersReflectPanel();
        this.fireParametersChanged();

    }

    /**
     * Resets the date to T0. <br>
     * <br>
     * TODO Should this use {@link #_defaultValueIfUnchecked}???
     */
    private void processDefault()
    {
        _valueTextField.setText(HEFSDateTools.SYSTEM_TIME_BASIS_STR);
        this.makeParametersReflectPanel();
        this.fireParametersChanged();

    }

    /**
     * Sets the date to be fixed and equal to the end time of the {@link #getBaseTimeSeries()}.
     */
    private void processEndTime()
    {
        if(getBaseTimeSeries() != null && !getBaseTimeSeries().isEmpty())
        {
            _valueTextField.setText(HCalendar.buildDateStr(HCalendar.computeCalendarFromMilliseconds(getBaseTimeSeries().get(0)
                                                                                                                        .getEndTime()),
                                                           HEFSDateTools.getGUIDateTimeTZFormat()));
            this.makeParametersReflectPanel();
            this.fireParametersChanged();
        }
    }

    /**
     * @return The Calendar specifying the fixed date equivalent of what is within the _valueTextField.
     */
    private Calendar determineFixedDateValue()
    {
        final String value = _valueTextField.getText().trim();
        Calendar cal = null;

        //Case 1: The value is already a fixed date.  In that case, just compute the fixed date.  This separate check
        //is needed now that years of 000# are allowed.  Before, the next if would for the 000# to be processed and the
        //processed date to be displayed.
        if(HEFSDateTools.isFixedDate(value))
        {
            cal = HEFSDateTools.computeFixedDate(value);
        }
        else if(getBaseTimeSeries() != null && !getBaseTimeSeries().isEmpty())
        {
            final Long millis = HEFSDateTools.computeTime(getSystemTime(), getBaseTimeSeries().get(0), value);
            if(millis == null)
            {
                //TODO Check to see if methods that call this can handle a null return.  I don't think one of
                //them can.  
            }
            else
            {
                cal = HCalendar.computeCalendarFromMilliseconds(millis);
            }
        }
        else
        {
            if(value.equalsIgnoreCase("" + Long.MIN_VALUE))
            {
                cal = HCalendar.computeCalendarFromMilliseconds(HEFSDateTools.computeTime(getSystemTime(), null, null));
            }
            else
            {
                final Long millis = HEFSDateTools.computeTime(getSystemTime(), null, value);
                if(millis == null)
                {
                }
                else
                {
                    cal = HCalendar.computeCalendarFromMilliseconds(millis);
                }
            }

        }
        return cal;
    }

    /**
     * Calls {@link HEFSDateTools#createFixedDateChooser(javax.swing.JComponent, HDateChooserListener, Calendar)} to
     * open a dialog for choosing a fixed date.
     */
    private void processFixedDate()
    {
        HEFSDateTools.createFixedDateChooser(this._timeOptionsBox, this, determineFixedDateValue());
    }

    /**
     * Calls
     * {@link HEFSDateTools#createRelativeDateChooser(javax.swing.JComponent, HDateChooserListener, String, String[])}
     * to open a dialog for choosing a relative date.
     */
    private void processRelativeDate()
    {
        HEFSDateTools.createRelativeDateChooser(_timeOptionsBox,
                                                this,
                                                _valueTextField.getText().trim(),
                                                getDateTypeChoices());
    }

    /**
     * Sets the date to the start time of the {@link #getBaseTimeSeries()}.
     */
    private void processStartTime()
    {
        if(getBaseTimeSeries() != null && !getBaseTimeSeries().isEmpty())
        {
            _valueTextField.setText(HCalendar.buildDateStr(HCalendar.computeCalendarFromMilliseconds(getBaseTimeSeries().get(0)
                                                                                                                        .getStartTime()),
                                                           HEFSDateTools.getGUIDateTimeTZFormat()));
            this.makeParametersReflectPanel();
            this.fireParametersChanged();
        }
    }

    /**
     * Processes a change in the text field, updating the {@link #_valueTextField} appropriately.
     */
    protected void processTextFieldChange()
    {
        final GenericParameter parm = getParameters().getParameterWithName(this._nameOfParameterToManage);

        //For an invalid date string...
        if(!HEFSDateTools.isDateStringValid(getValueTextFieldText().trim()))
        {
            if(parm == null)
            {
                //No parameter exists to revert the text field to, so use the default.
                _valueTextField.setText(this.getDefaultValueIfUnchecked());
            }
            else
            {
                //Use the parameter as a source of the old text field value
                _valueTextField.setText(parm.getValue());
            }
        }
        //For a valid date string that differs from what is currently specified...
        else if((parm == null) || (!getValueTextFieldText().trim().equals(parm.getValue())))
        {
            this.makeParametersReflectPanel();
            this.fireParametersChanged();
        }
    }

    /**
     * Updates the selected item in the {@link #_timeOptionsBox} based on a call to
     * {@link #getDateType(String, TimeSeriesArray)} providing it the date string specified by the
     * {@link #_valueTextField}.
     */
    private void updateTimeOptionsBox()
    {
        this._timeOptionsBox.removeItemListener(this);
        this._timeOptionsBox.removeActionListener(this);

        final String dateStr = _valueTextField.getText().trim();
        String dateType = null;
        if(getBaseTimeSeries() != null && !getBaseTimeSeries().isEmpty())
        {
            dateType = getDateType(dateStr, getBaseTimeSeries().get(0));
        }
        else
        {
            dateType = getDateType(dateStr, null);

        }

        final List<String> options = getTimeOptionList();
        if(options.contains(dateType))
        {
            _timeOptionsBox.setSelectedItem(dateType);
        }
        else
        {
            //case 1: date type is "T0", but there is no "set to default" option in the combo box 
            if(dateType.equals(SET_TO_DEFAULT) && options.contains(SET_TO_RELATIVE))
            {
                _timeOptionsBox.setSelectedItem(SET_TO_RELATIVE);
            }
        }

        this._timeOptionsBox.addItemListener(this);
        this._timeOptionsBox.addActionListener(this);
    }

    private ArgumentsProcessor getArgumentsProcessor()
    {
        return _argumentsProcessor;
    }

    /**
     * Initializes {@link #_dateTypeChoices} to {@link HEFSDateTools#RELATIVE_DATE_BASIS_STRINGS} if it is not yet set.
     */
    private String[] getDateTypeChoices()
    {
        if(_dateTypeChoices == null)
        {
            _dateTypeChoices = HEFSDateTools.RELATIVE_DATE_BASIS_STRINGS;
        }
        return _dateTypeChoices;
    }

    /**
     * @return Either {@link #_defaultValueIfUnchecked} or {@link HEFSDateTools#SYSTEM_TIME_BASIS_STR} if that is null.
     */
    private String getDefaultValueIfUnchecked()
    {
        if(_defaultValueIfUnchecked == null)
        {
            _defaultValueIfUnchecked = HEFSDateTools.SYSTEM_TIME_BASIS_STR;
        }

        return _defaultValueIfUnchecked;
    }

    /**
     * @return A panel that displays the used components for editing a date, except for the
     *         {@link #_includeInListCheckBox}, which is acquired separately via {@link #getIncludeInListCheckBox()}.
     */
    public JPanel getEditingDisplayPanel()
    {
        // JPanel optionPanel = new JPanel();
        //optionPanel.add(_timeOptionsBox);

        final JPanel result = new JPanel(new GridBagLayout());
        GridBagConstraints constraints;
        final Insets theinsets = new Insets(2, 2, 2, 2);
        int gridx = 0;

        if(isArrowButtonAvailable())
        {
            constraints = SwingTools.returnGridBagConstraints(gridx,
                                                              0,
                                                              1,
                                                              1,
                                                              0,
                                                              1,
                                                              GridBagConstraints.WEST,
                                                              GridBagConstraints.BOTH,
                                                              new Insets(2, 2, 2, 0),
                                                              0,
                                                              0);
            result.add(_decreaseButton, constraints);
            gridx++;
        }

        constraints = SwingTools.returnGridBagConstraints(gridx,
                                                          0,
                                                          1,
                                                          1,
                                                          0,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          theinsets,
                                                          0,
                                                          0);
        result.add(_valueTextField, constraints);
        gridx++;

        if(isArrowButtonAvailable())
        {
            constraints = SwingTools.returnGridBagConstraints(gridx,
                                                              0,
                                                              1,
                                                              1,
                                                              0,
                                                              1,
                                                              GridBagConstraints.WEST,
                                                              GridBagConstraints.BOTH,
                                                              new Insets(2, 0, 2, 2),
                                                              0,
                                                              0);
            result.add(_increaseButton, constraints);
            gridx++;
        }

        constraints = SwingTools.returnGridBagConstraints(gridx,
                                                          0,
                                                          1,
                                                          1,
                                                          0,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.NONE,
                                                          theinsets,
                                                          0,
                                                          0);
        result.add(_timeOptionsBox, constraints);

        adjustButtonStatus();

        return result;
    }

    public JCheckBox getIncludeInListCheckBox()
    {
        return this._includeInListCheckBox;
    }

    protected String getNameOfParameterToManage()
    {
        return this._nameOfParameterToManage;
    }

    /**
     * Initializes {@link #_timeOptions} to {@link #DEFAULT_OPERATION_CHOICES} if it is not yet set.
     */
    public String[] getOperationChoices()
    {
        if(_timeOptions == null)
        {
            _timeOptions = DEFAULT_OPERATION_CHOICES;
        }
        return _timeOptions;
    }

    /**
     * Convenience wrapper on {@link #getManagedParameters()} converting it to {@link GenericParameterList}.
     */
    protected GenericParameterList getParameters()
    {
        return (GenericParameterList)this.getManagedParameters();
    }

    /**
     * @return The selected time option {@link #_selectedTimeOption}, which should be used to remember a selected time
     *         option so that it can be reselected later.
     */
    public String getSelectedOperationChoice()
    {
        return _selectedTimeOption;
    }

    /**
     * @return The system time to use, acquired via {@link GeneralTools#getSystemTime(ArgumentsProcessor)}, which pulls
     *         the value from {@link #getArgumentsProcessor()}.
     */
    private Calendar getSystemTime()
    {
        return GeneralTools.getSystemTime(getArgumentsProcessor());
    }

    /**
     * @return The list of choices within {@link #_timeOptionsBox}. It makes use of
     *         {@link SwingTools#getListModelList(javax.swing.ListModel)}.
     */
    private List<String> getTimeOptionList()
    {
        return SwingTools.getListModelList(_timeOptionsBox.getModel());
    }

    /**
     * @return The end time of {@link #_baseTimeSeries}. If not provided, the returned value is {@link Long#MAX_VALUE}.
     */
    private long getTimeSeriesEndTime()
    {
        if((getBaseTimeSeries() == null) || getBaseTimeSeries().isEmpty())
        {
            return Long.MAX_VALUE;
        }

        return getBaseTimeSeries().get(0).getEndTime();
    }

    /**
     * @return The start time of {@link #_baseTimeSeries}. If not provided, the returned value is {@link Long#MIN_VALUE}
     *         .
     */
    private long getTimeSeriesStartTime()
    {
        if((getBaseTimeSeries() == null) || getBaseTimeSeries().isEmpty())
        {
            return Long.MIN_VALUE;
        }

        return getBaseTimeSeries().get(0).getStartTime();
    }

    public JTextField getValueTextField()
    {
        return this._valueTextField;
    }

    /**
     * @return The user specified date, as provided by the text within {@link #_valueTextField}. Null is returned only
     *         if the {@link #_valueTextField} is null.
     */
    public String getValueTextFieldText()
    {
        if(_valueTextField == null)
        {
            return null;
        }
        else
        {
            return this._valueTextField.getText();
        }
    }

    /**
     * Sets {@link #_baseTimeSeries} and calls {@link #adjustButtonStatus()}.
     */
    public void setBaseTimeSeries(final TimeSeriesArrays ts)
    {
        this._baseTimeSeries = ts;
        adjustButtonStatus();
    }

    /**
     * Return the time series that provide input to the plugin.
     */
    public TimeSeriesArrays getBaseTimeSeries()
    {
        return this._baseTimeSeries;
    }

    public void setArgumentsProcessor(final ArgumentsProcessor argumentsProcessor)
    {
        this._argumentsProcessor = argumentsProcessor;
    }

    public void setArrowButtonAvailable(final boolean arrowButtonAvailable)
    {
        this._arrowButtonAvailable = arrowButtonAvailable;
    }

    public void setDateTypeChoices(final String[] dateTypeChoices)
    {
        this._dateTypeChoices = dateTypeChoices;
    }

    public void setDefaultValueIfUnchecked(final String defaultValueIfUnchecked)
    {
        this._defaultValueIfUnchecked = defaultValueIfUnchecked;
    }

    protected void setNameOfParameterToManage(final String nameOfParameterToManage)
    {
        this._nameOfParameterToManage = nameOfParameterToManage;
    }

    /**
     * Sets the {@link #_timeOptions} to be the combination of the provided choices and
     * {@link #DEFAULT_OPERATION_CHOICES}. The {@link #_timeOptionsBox} will be updated without triggering any item or
     * action listeners.
     */
    public void setOperationChoices(final String[] operationChoices)
    {
        if(operationChoices == null || operationChoices.length <= 0)
        {
            return;
        }

        final String[] validOptions = StringTools.buildArrayOfStringsFromAcceptableArray(operationChoices,
                                                                                         DEFAULT_OPERATION_CHOICES);

        if(validOptions == null || validOptions.length <= 0)
        {
            return;
        }

        this._timeOptions = validOptions;

        if(this._timeOptionsBox != null)
        {
            this._timeOptionsBox.removeItemListener(this);
            this._timeOptionsBox.removeActionListener(this);

            this._timeOptionsBox.removeAllItems();

            for(final String type: validOptions)
            {
                this._timeOptionsBox.addItem(type);
            }

            this._timeOptionsBox.addItemListener(this);
            this._timeOptionsBox.addActionListener(this);
        }
    }

    /**
     * Specifies {@link #_selectedTimeOption} and selects it within {@link #_timeOptionsBox} without triggering any
     * action or item listener.
     */
    public void setSelectedOperationChoice(final String selectedOperationChoice)
    {
        if(selectedOperationChoice == null || selectedOperationChoice.length() <= 0)
        {
            return;
        }

        this._selectedTimeOption = selectedOperationChoice;

        if(this._timeOptionsBox != null)
        {
            this._timeOptionsBox.removeItemListener(this);
            this._timeOptionsBox.removeActionListener(this);

            this._timeOptionsBox.setSelectedItem(selectedOperationChoice);

            this._timeOptionsBox.addItemListener(this);
            this._timeOptionsBox.addActionListener(this);
        }
    }

    @Override
    public void actionPerformed(final ActionEvent evt)
    {
        if(evt.getSource() == this._timeOptionsBox)
        {
            final String selectedItem = ((String)this._timeOptionsBox.getSelectedItem()).trim();
            if(selectedItem.equals(SET_TO_DEFAULT))
            {
                processDefault();
            }
            else if(selectedItem.equals(SET_TO_FIXED))
            {
                processFixedDate();
            }
            else if(selectedItem.equals(SET_TO_RELATIVE))
            {
                processRelativeDate();
            }
            else if(selectedItem.equals(SET_TO_TIMESERIES_STARTTIME))
            {
                processStartTime();
            }
            else if(selectedItem.equals(SET_TO_TIMESERIES_ENDTIME))
            {
                processEndTime();
            }

            adjustButtonStatus();
        }
        else if(evt.getSource() == this._decreaseButton)
        {
            processAdjustDate(-1);
        }
        else if(evt.getSource() == this._increaseButton)
        {
            processAdjustDate(1);
        }
        else if(evt.getSource() == _valueTextField)
        {
            processTextFieldChange();
        }
    }

    @Override
    public void dateChooseCancelled(final JDialog caller)
    {
        updateTimeOptionsBox();
    }

    @Override
    public void dateChosen(final JDialog caller)
    {
        if(caller instanceof HDateChooserDialog)
        {
            final Calendar dateChosen = HCalendar.computeCalendarFromDate(((HDateChooserDialog)caller).getDate());
            final String dateStr = HCalendar.buildDateStr(dateChosen, HEFSDateTools.getGUIDateTimeTZFormat());
            _valueTextField.setText(dateStr);
            updateTimeOptionsBox();
        }
        else if(caller instanceof HRelativeDateChooserDialog)
        {
            final String dateString = ((HRelativeDateChooserDialog)caller).getDateString();
            _valueTextField.setText(HEFSDateTools.cleanUpAndReformatRelativeDateString(dateString));

            updateTimeOptionsBox();
        }

        this.makeParametersReflectPanel();
        this.fireParametersChanged();
    }

    public void determineEnablednessBasedOnCheckbox()
    {
        //Disable the panel if the checkbox is not checked, but only if the disable flag is set to true.
        if(disablePanelIfCheckboxUnchecked())
        {
            if(_includeInListCheckBox.isSelected())
            {
                enableAllButCheckbox();
            }
            else
            {
                disableAllButCheckbox();
            }
        }
        else
        {
            enableAllButCheckbox();
        }
    }

    /**
     * Disables all components exception {@link #_includeInListCheckBox}.
     */
    public void disableAllButCheckbox()
    {
        this._decreaseButton.setEnabled(false);
        this._valueTextField.setEnabled(false);
        this._increaseButton.setEnabled(false);
        this._timeOptionsBox.setEnabled(false);
    }

    /**
     * Enables all components except {@link #_includeInListCheckBox}.
     */
    public void enableAllButCheckbox()
    {
        this._decreaseButton.setEnabled(true);
        this._valueTextField.setEnabled(true);
        this._increaseButton.setEnabled(true);
        this._timeOptionsBox.setEnabled(true);
    }

    @Override
    public void disableSubPanel()
    {
        this._includeInListCheckBox.setEnabled(false);
        disableAllButCheckbox();
    }

    @Override
    public void enableSubPanel()
    {
        this._includeInListCheckBox.setEnabled(true);
        enableAllButCheckbox();
    }

    @Override
    public void itemStateChanged(final ItemEvent evt)
    {
        //Disable the panel if the checkbox is not checked, but only if the disable flag is set to true.
        if(evt.getSource() == _includeInListCheckBox)
        {
            determineEnablednessBasedOnCheckbox();
        }
        this.makeParametersReflectPanel();
        this.fireParametersChanged();
    }

    @Override
    public void focusGained(final FocusEvent e)
    {
        if(e.getSource() == this._valueTextField)
        {
            this._valueTextField.selectAll();
        }
    }

    @Override
    public void focusLost(final FocusEvent e)
    {
        if(e.getSource() == this._valueTextField)
        {
            this.processTextFieldChange();
        }
    }

    /**
     * @return The constant to use as the date type given the provided string and a time series. The returned constant
     *         reflects a choice in the {@link #_dateTypeChoices}.
     */
    public static String getDateType(final String dateStr, final TimeSeriesArray oneTimeSeries)
    {
        //check if it is empty string, default to system time.
        if(dateStr == null || dateStr.trim().length() <= 0)
        {
            return SET_TO_DEFAULT;
        }

        //check it is system time 
        if(dateStr.equalsIgnoreCase(HEFSDateTools.SYSTEM_TIME_BASIS_STR))
        {
            return SET_TO_DEFAULT;
        }
        //check it is time series start time "tsStartTime" 
        else if(dateStr.equalsIgnoreCase(HEFSDateTools.TIMESERIES_START_TIME_BASIS_STR))
        {
            return SET_TO_TIMESERIES_STARTTIME;
        }
        //check it is time series end time "tsEndTime" 
        else if(dateStr.equalsIgnoreCase(HEFSDateTools.TIMESERIES_END_TIME_BASIS_STR))
        {
            return SET_TO_TIMESERIES_ENDTIME;
        }
        else
        {
            //check if it is relative time

            final String[] dateArray = dateStr.split(" ");
            for(final String type: HEFSDateTools.RELATIVE_DATE_BASIS_STRINGS)
            {
                if(dateArray[0].indexOf(type) >= 0)
                {
                    return SET_TO_RELATIVE;
                }
            }

            //process for fixed time
            //check if it is start time

            //if the time series is null, return fixed date
            if(oneTimeSeries == null)
            {
                return SET_TO_FIXED;
            }

            final Long cal = HEFSDateTools.computeTime(null, oneTimeSeries, dateStr);

            if(cal == oneTimeSeries.getStartTime())
            {
                return SET_TO_TIMESERIES_STARTTIME;
            }
            else if(cal == oneTimeSeries.getEndTime())
            {
                return SET_TO_TIMESERIES_ENDTIME;
            }
            else
            {
                return SET_TO_FIXED;
            }
        }
    }
}
