package ohd.hseb.hefs.utils.datetime;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.TimeZone;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.arguments.DefaultArgumentsProcessor;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.misc.HCalendar;

/**
 * Provides tools related to {@link SimpleDateFormat}, which JFreeChart uses, and other basic tools. For additional
 * tools, see {@link HCalendar}.
 * 
 * @author hankherr
 */
public abstract class DateTools
{
    public final static TimeZone GMT_TIME_ZONE = TimeZone.getTimeZone("GMT");

    /**
     * Jan 1, 1970 at 12Z. This long is used for quick determination if a provided time is a 12Z time.
     */
    public final static long BASIS_LONG_FOR_12Z_DETERMINATION = 43200000L;

    /**
     * Records a map of string date patterns to the SimpleDateFormat used to format those dates.
     */
    static ThreadLocal<HashMap<Dyad<String, TimeZone>, SimpleDateFormat>> _formattersThreadLocal;
    static
    {
        _formattersThreadLocal = new ThreadLocal<HashMap<Dyad<String, TimeZone>, SimpleDateFormat>>()
        {
            @Override
            public HashMap<Dyad<String, TimeZone>, SimpleDateFormat> initialValue()
            {
                final HashMap<Dyad<String, TimeZone>, SimpleDateFormat> formatters = new HashMap<Dyad<String, TimeZone>, SimpleDateFormat>();
                return formatters;
            }
        };
    };

    /**
     * Makes use of {@link ThreadLocal} in order to construct a thread-safe (supposedly) SimpleDateFormat for use in
     * formatting dates.
     * 
     * @param pattern The pattern to use for formatting.
     * @param timeZone The time zone of the date to format.
     * @return A SimpleDateFormat that is ready for use in the current thread.
     */
    public static SimpleDateFormat getThreadSafeSimpleDateFormat(final String pattern, final TimeZone timeZone)
    {
        final Dyad<String, TimeZone> dyad = new Dyad<String, TimeZone>(pattern, timeZone);

        SimpleDateFormat sdf = _formattersThreadLocal.get().get(dyad);
        if(sdf == null)
        {
            sdf = new SimpleDateFormat(pattern);
            sdf.setTimeZone(timeZone);
            _formattersThreadLocal.get().put(dyad, sdf);
        }
        return sdf;
    }

    /**
     * Makes use of {@link ThreadLocal} in order to construct a thread-safe (supposedly) SimpleDateFormat for use in
     * formatting dates. Calls {@link #getThreadSafeSimpleDateFormat(String, TimeZone)}.
     * 
     * @param sdf A SimpleDateFormat specifying the pattern and time zone of the desired thread-safe version.
     * @return A SimpleDateFormat that is ready for use in the current thread.
     */
    public static SimpleDateFormat getThreadSafeSimpleDateFormat(final SimpleDateFormat sdf)
    {
        return getThreadSafeSimpleDateFormat(sdf.toPattern(), sdf.getTimeZone());
    }

    /**
     * @param dateString The date string to check. The String checks formats using {@link HEFSDateTools}, which makes
     *            use of {@link HCalendar} formatting.
     * @param argProc The arguments processor that supplies the system time.
     * @param oneTimeSeries Time series array serving as a base time series for computing relative dates.
     * @return True if the date is valid, false otherwise.
     */
    public static boolean isDateStrValid(final String dateString,
                                         final DefaultArgumentsProcessor argProc,
                                         final TimeSeriesArray oneTimeSeries)
    {
        final Calendar systemTime = ohd.hseb.hefs.utils.tools.GeneralTools.getSystemTime(argProc);

        //check for relative date type
        if(isRelativeDateStr(dateString))
        {
//        for(final String type: HEFSDateTools.RELATIVE_DATE_BASIS_STRINGS)
//        {
//            if(dateString.indexOf(type) >= 0)
//            {
            if(HEFSDateTools.computeTime(systemTime, oneTimeSeries, dateString) != null)
            {
                return true;
            }
            else
            {
                return false;
            }
//            }
        }

        //check for fixed date
        if(HEFSDateTools.computeFixedDate(dateString) != null)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * @param dateString A string that is known to be a date string of some kind; either fixed or relative.
     * @return True if the date string appears to be a relative date string, based on it containing one of
     *         {@link HEFSDateTools#RELATIVE_DATE_BASIS_STRINGS}.
     */
    public static boolean isRelativeDateStr(final String dateString)
    {
        for(final String type: HEFSDateTools.RELATIVE_DATE_BASIS_STRINGS)
        {
            if(dateString.toUpperCase().contains(type.toUpperCase())) //equivalent to checking with ignore case
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Format is [quantity] [unit] where unit is one of period, year(s), month(s), week(s), day(s), hour(s).
     * 
     * @param stepStr the string to check.
     * @return If the string is a valid time step string. True if it contains 'period', 'year', or 'month', or if
     *         HCalendar.computeIntervalValueInMillis returns non-missing value.
     */
    public static boolean isTimeStepStringValid(final String stepStr)
    {
        if(stepStr.contains("period") || stepStr.contains("year") || stepStr.contains("month")) //TODO what about this?  Does it need to be case insensitive?
        {
            return true;
        }

        final long timeStepLong = HCalendar.computeIntervalValueInMillis(stepStr);
        if(timeStepLong == (long)DataSet.MISSING)
        {
            return false;
        }
        return true;
    }

    /**
     * @param value Date string.
     * @param format Format to put the date in.
     * @return If the date is relative, then value is returned without change. Otherwise, the date in the string will be
     *         reformatted into format specified using {@link HCalendar#buildDateStr(Calendar, String)}. NOTE: we may
     *         want to change this to use other tools.
     */
    public static String formatDateStr(final String value, final String format)
    {
        if(value == null || value.length() <= 0)
        {
            return null;
        }

        if(DateTools.isRelativeDateStr(value))
        {
            return value;
        }
        else
        {
            final Long millis = HEFSDateTools.computeTime(null, null, value);
            if(millis == null)
            {
                return null;
            }
            return HCalendar.buildDateStr(millis, format);
//            return HCalendar.buildDateStr(HEFSDateTools.computeTime(null, null, value), format);
        }
    }

    /**
     * @param start Starting time in millis
     * @param end Ending time in millis
     * @return The number of days between, or, more accurately, the number of whole 24-hour periods. This is a straight
     *         subtraction and division (i.e., it does not count the start time, so if you want to know the number of
     *         days in between and inclusive add one to the return value).
     */
    public static int countNumberOfDaysBetweenTwoTimes(final long start, final long end)
    {
        return (int)((end - start) / (24 * HCalendar.MILLIS_IN_HR));
    }

    /**
     * @param initialTime The base time.
     * @param desiredHourOfDay The hour of day that the returned time must specify.
     * @return The milliseconds corresponding to the time that is the first instance of that hour of day AFTER OR ON the
     *         base time. The minutes, seconds, and milliseconds within a second will be set to 0.
     */
    public static long computeTimeWithDesiredHourAfterBase(final long base, final int desiredHourOfDay)
    {
        final Calendar cal = HCalendar.computeCalendarFromMilliseconds(base);

        //If the hour of the day equals the base hour, then this will check to make sure one of minute, second, 
        //and millisecond are not 0 before considering it to be after.
        if((cal.get(Calendar.HOUR_OF_DAY) > desiredHourOfDay)
            || ((cal.get(Calendar.HOUR_OF_DAY) == desiredHourOfDay) && ((cal.get(Calendar.MINUTE) > 0)
                || (cal.get(Calendar.SECOND) > 0) || (cal.get(Calendar.MILLISECOND) > 0))))
        {
            cal.add(Calendar.DAY_OF_YEAR, 1);
        }
        cal.set(Calendar.HOUR_OF_DAY, desiredHourOfDay);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTimeInMillis();
    }

    /**
     * @param initialTime The base time.
     * @param desiredHourOfDay The hour of day that the returned time must specify.
     * @return The milliseconds corresponding to the time that is the first instance of that hour of day BEFORE OR ON
     *         the base time. The minutes, seconds, and milliseconds within a second will be set to 0.
     */
    public static long computeTimeWithDesiredHourBeforeBase(final long base, final int desiredHourOfDay)
    {
        final Calendar cal = HCalendar.computeCalendarFromMilliseconds(base);
        if(cal.get(Calendar.HOUR_OF_DAY) < desiredHourOfDay)
        {
            cal.add(Calendar.DAY_OF_YEAR, -1);
        }
        cal.set(Calendar.HOUR_OF_DAY, desiredHourOfDay);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTimeInMillis();
    }

    /**
     * @param millis Time in millis to check.
     * @return True if the time is a 12Z time, which is determined by computing its difference from
     *         {@link #BASIS_LONG_FOR_12Z_DETERMINATION} and modding it by 24 hours.
     */
    public static boolean is12ZTime(final long millis)
    {
        return ((millis - BASIS_LONG_FOR_12Z_DETERMINATION) % (24 * HCalendar.MILLIS_IN_HR)) == 0;
    }

    /**
     * Calls {@link #is12ZTime(long)} with the millis from the provide calendar.
     */
    public static boolean is12ZTime(final Calendar cal)
    {
        return is12ZTime(cal.getTimeInMillis());
    }

    /**
     * This uses {@link Calendar}, so it may be slowish.
     * 
     * @return Removes the minutes, seconds, and milliseconds (within the second) from the given time so that it is an
     *         exact hour.
     */
    public static long findExactHour(final long time)
    {
        final Calendar cal = HCalendar.computeCalendarFromMilliseconds(time);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTimeInMillis();
    }

    /**
     * This calls {@link #computeTimeWithDesiredHourAfterBase(long, int)} with 12 for the hour.
     * 
     * @param basisTime The time for which to acquire the first 12Z after it.
     * @return The first 12Z time on or after the given basis time, with 0 for minutes, seconds, and milliseconds in a
     *         second.
     */
    public static long findFirstExact12ZTimeAfterOrOn(final long basisTime)
    {
        return computeTimeWithDesiredHourAfterBase(basisTime, 12);
    }
}
