package ohd.hseb.hefs.utils.datetime;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JPanel;

import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.util.misc.HCalendar;

import com.toedter.calendar.JDateChooser;

/**
 * Displays a {@link HDateChooserPanel} inside of a dialog with a Now button (tied to a system time specification) and
 * an OK and Cancel button. Call {@link #getDate()} after use in order to get the time selected by the user. To listen
 * for changes, create an {@link HDateChooserListener} and add it via the {@link #addListener(HDateChooserListener)}
 * method.
 * 
 * @author hankherr
 */
public class HDateChooserDialog extends JDialog implements ActionListener
{

    private static final long serialVersionUID = -7298206933650530721L;
    public final static String CLASSNAME = "HDateChooserDialog";

    protected final static String[] BUTTON_NAMES = {"OK", "Cancel"};
    protected final static int OKAY_BUTTON = 0;
    protected final static int CANCEL_BUTTON = 1;

    protected JButton[] _buttons;
    protected JButton _setSystemTimeButton;

    private HDateChooserPanel _dateChooserPanel = null;

    protected List<HDateChooserListener> _listeners = new ArrayList<HDateChooserListener>();

    private Date systemTime = HCalendar.computeCalendarFromDate(new Date()).getTime();

    public HDateChooserDialog()
    {
        createDisplay();
    }

    public HDateChooserDialog(final Date systemTime)
    {
        setSystemTime(systemTime);
        createDisplay();
    }

    /**
     * Adds the passed in HDateChooserOwner to the Vector of owners.
     * 
     * @param owner A class that implements the HDateChooserOwner interface.
     */
    public void addListener(final HDateChooserListener listener)
    {
        _listeners.add(listener);
    }

    @Override
    public void actionPerformed(final ActionEvent evt)
    {
        if(evt.getSource() == _buttons[OKAY_BUTTON])
        {
            int i;
            for(i = 0; i < _listeners.size(); i++)
            {
                ((_listeners.get(i))).dateChosen(this);
            }
            setVisible(false);
        }
        else if(evt.getSource() == _buttons[CANCEL_BUTTON])
        {
            int i;
            for(i = 0; i < _listeners.size(); i++)
            {
                ((_listeners.get(i))).dateChooseCancelled(this);
            }
            this.dispose();
        }
        else if(evt.getSource() == this._setSystemTimeButton)
        {
            this.setDate(this.getSystemTime());
        }

    }

    /**
     * Creates the display.
     */
    protected void createDisplay()
    {
        this.setTitle("Choose Date/Time");
        _dateChooserPanel = new HDateChooserPanel(null, HDateChooserPanel.OPTION_DATE_TIME, "Date and Time: ");
        _setSystemTimeButton = HSwingFactory.createJButton("Now", null, this);

        //int[] groupsizes = {2};
        _buttons = HSwingFactory.createJButtons(BUTTON_NAMES, null, this);
        //Box buttonPanel = HSwingFactory.createJButtonBox(_buttons, groupsizes, BoxLayout.X_AXIS);

        final JPanel mainPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        mainPanel.add(_dateChooserPanel);
        mainPanel.add(_setSystemTimeButton);

        final JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        buttonPanel.add(_buttons[0]);
        buttonPanel.add(_buttons[1]);

        this.getRootPane().setDefaultButton(_buttons[0]);

        this.setLayout(new BorderLayout());
        this.add(mainPanel, BorderLayout.CENTER);
        this.add(buttonPanel, BorderLayout.SOUTH);

        this.pack();
    }

    /**
     * Returns the date. If the {@link JDateChooser} is started with an empty date and no date is set by the user, null
     * is returned.
     * 
     * @return the current date
     */
    public Date getDate()
    {
        return this._dateChooserPanel.getDate();
    }

    public Date getSystemTime()
    {
        return systemTime;
    }

    public void setDate(final Date date)
    {
        this._dateChooserPanel.setDate(date);
    }

    public void setCalendar(final Calendar cal)
    {
        if(cal != null)
        {
            this._dateChooserPanel.setCalendar(cal);
        }
    }

    public void setSystemTime(final Date systemTime)
    {
        this.systemTime = systemTime;
    }

}
