package ohd.hseb.hefs.utils.datetime;

import java.awt.FlowLayout;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import ohd.hseb.util.misc.HCalendar;

import com.toedter.calendar.JCalendar;
import com.toedter.calendar.JDateChooser;
import com.toedter.calendar.JSpinnerDateEditor;

/**
 * A {@link JPanel} that uses a centered flow layout that includes a label and a {@link JDateChooser}. Apparently
 * {@link JDateChooser} is designed to work in local time, by default, with no apparent way to switch it to GMT (I've
 * never looked into it myself, but Guoxian could not find anything). Hence, the {@link #setDate(Date)} and
 * {@link #getDate()} methods both perform a timezone switch. Hopefully, all time zone concerns will be buried inside
 * this object so that callers do not have to worry about it.
 * 
 * @author hankherr
 */
public class HDateChooserPanel extends JPanel implements PropertyChangeListener
{
    private static final long serialVersionUID = 3214217185229641898L;

    public static int OPTION_DATE_TIME = 0;
    public static int OPTION_DATE = 1;

    private String _labelString = null;
    private int _dateType = OPTION_DATE_TIME;
    private Date _startDate = HCalendar.computeCalendarFromDate(new Date()).getTime();
    private JDateChooser _dateChooser = null;
    private JCheckBox _enabledCheckBox = null;

    //TODO need figure out why could not change the time zone of the JDateChooser
    //10-09-2009
    //the time zone of the JDateChooser is the local time zone and could not be changed.
    //It seems a bug for the JCalendar bean
    //10-13-2009
    private int getTimeZoneHourDifference(final Calendar basisCal)
    {
        final Calendar startDateCal = Calendar.getInstance();
        startDateCal.setTime(basisCal.getTime());

        //This computation is specified in the deprecation method of _startDate.getTimeZoneOffset().
        final int timeZoneOffset = -1
            * (startDateCal.get(Calendar.ZONE_OFFSET) + startDateCal.get(Calendar.DST_OFFSET)) / (60 * 60 * 1000);
        return timeZoneOffset;

//This is the original code.
//        Calendar systemTime = HCalendar.computeCalendarFromDate(new Date());
//
//        TimeZone currTimeZone = TimeZone.getDefault();
//
//        if(this._dateChooser.getCalendar() != null)
//        {
//            currTimeZone = this._dateChooser.getCalendar().getTimeZone();
//        }
//
//        int systemHour = systemTime.get(Calendar.HOUR_OF_DAY);
//        systemTime.setTimeZone(currTimeZone);
//
//        int currHour = systemTime.get(Calendar.HOUR_OF_DAY);
//
//        return systemHour - currHour;
    }

    public HDateChooserPanel()
    {
        this(null, HDateChooserPanel.OPTION_DATE_TIME, null, false);
    }

    public HDateChooserPanel(final Date startDate, final int dateType, final String labelString, final boolean checkBox)
    {
        setName("HDateChooserPane");

        setStartDate(startDate);
        setDateType(dateType);
        setLabelString(labelString);

        createDisplay(checkBox);

        addListeners();
    }

    public HDateChooserPanel(final Date startDate, final int dateType, final String labelString)
    {
        this(startDate, dateType, labelString, false);
//        setName("HDateChooserPanel");
//
//        setStartDate(startDate);
//        setDateType(dateType);
//        setLabelString(labelString);
//
//        createDisplay();
    }

    @SuppressWarnings("serial")
    private void createDisplay(final boolean checkBox)
    {
        setLayout(new FlowLayout(FlowLayout.CENTER));

        if(checkBox)
        {
            _enabledCheckBox = new JCheckBox();
            add(_enabledCheckBox);
        }

        if(getLabelString() != null && getLabelString().trim().length() > 0)
        {
            final JLabel label = new JLabel(getLabelString().trim(), null, JLabel.RIGHT);
            add(label);
        }

        if(getDateType() == OPTION_DATE)
        {
            _dateChooser = new JDateChooser(new JCalendar(HCalendar.computeCalendarFromDate(new Date())),
                                            getStartDate(),
                                            HCalendar.convertToSimpleDateFormat(HEFSDateTools.getGUIDateFormat()),
                                            new JSpinnerDateEditor()
                                            {
                                                @Override
                                                public void focusLost(final FocusEvent arg0)
                                                {
                                                    final String dateString = ((DefaultEditor)getEditor()).getTextField()
                                                                                                          .getText();
                                                    final SimpleDateFormat sdf = DateTools.getThreadSafeSimpleDateFormat(getDateFormatString(),
                                                                                                                         DateTools.GMT_TIME_ZONE);

                                                    try
                                                    {
                                                        sdf.parse(dateString);
                                                        commitEdit();
                                                    }
                                                    catch(final Throwable t)
                                                    {
                                                        ((DefaultEditor)getEditor()).getTextField()
                                                                                    .setText(sdf.format(HDateChooserPanel.this.getDate()));
                                                    }

                                                    super.focusLost(arg0);

//                                                    _dateChooser.setDate(getDate());
//                                                    super.focusLost(arg0);
                                                }

                                                @Override
                                                public void commitEdit() throws ParseException
                                                {
                                                    super.commitEdit();
                                                }
                                            });
            add(_dateChooser);
        }
        else if(getDateType() == OPTION_DATE_TIME)
        {
            _dateChooser = new JDateChooser(new JCalendar(HCalendar.computeCalendarFromDate(new Date())),
                                            getStartDate(),
                                            HCalendar.convertToSimpleDateFormat(HEFSDateTools.getGUIDateTimeFormat()),
                                            new JSpinnerDateEditor()
                                            {
                                                @Override
                                                public void focusLost(final FocusEvent arg0)
                                                {
                                                    final String dateString = ((DefaultEditor)getEditor()).getTextField()
                                                                                                          .getText();
                                                    final SimpleDateFormat sdf = DateTools.getThreadSafeSimpleDateFormat(getDateFormatString(),
                                                                                                                         DateTools.GMT_TIME_ZONE);

                                                    try
                                                    {
                                                        sdf.parse(dateString);
                                                        commitEdit();
                                                    }
                                                    catch(final Throwable t)
                                                    {
                                                        ((DefaultEditor)getEditor()).getTextField()
                                                                                    .setText(sdf.format(HDateChooserPanel.this.getDate()));
                                                    }

                                                    super.focusLost(arg0);
//                                                    getDateFormatString()s
//                                                    _dateChooser.setDate(getDate());
//                                                    setDate(getDate());
                                                }

                                            });
            add(_dateChooser);
        }

        if(_enabledCheckBox != null)
        {
            setEnabled(false);
        }
    }

    private void addListeners()
    {
        if(_enabledCheckBox == null)
        {
            return;
        }
        _enabledCheckBox.addItemListener(new ItemListener()
        {

            public void itemStateChanged(final ItemEvent e)
            {
                setEnabled(_enabledCheckBox.isSelected());
            }
        });
    }

    /**
     * @return Date (in localtime)
     */
    public Date getDate()
    {
        final Calendar cal = this._dateChooser.getCalendar();
        //need convert local time back to system time
        cal.add(Calendar.HOUR_OF_DAY, 0 - getTimeZoneHourDifference(cal));

        return cal.getTime();
    }

    public int getDateType()
    {
        return _dateType;
    }

    public String getLabelString()
    {
        return _labelString;
    }

    public Date getStartDate()
    {
        return _startDate;
    }

    @Override
    public boolean isEnabled()
    {
        return this._dateChooser.isEnabled();
    }

    @Override
    public void propertyChange(final PropertyChangeEvent evt)
    {
        System.err.println(evt.getPropertyName());
        if(evt.getPropertyName().equals("date"))
        {
            setDate((Date)evt.getNewValue());
        }
    }

    public void setDate(final Date date)
    {
        final Calendar cal = HCalendar.computeCalendarFromDate(date);

        //need convert system time to local time 
        cal.add(Calendar.HOUR_OF_DAY, getTimeZoneHourDifference(cal));

        this._dateChooser.setDate(cal.getTime());
    }

    public void setCalendar(final Calendar cal)
    {

        //need convert system time to local time 
        cal.add(Calendar.HOUR_OF_DAY, getTimeZoneHourDifference(cal));

        this._dateChooser.setCalendar(cal);
    }

    public void setDateType(final int dateType)
    {
        this._dateType = dateType;
    }

    @Override
    public void setEnabled(final boolean enabled)
    {
        this._dateChooser.setEnabled(enabled);
    }

    public void setLabelString(final String labelString)
    {
        this._labelString = labelString;
    }

    public void setStartDate(final Date startDate)
    {
        if(startDate != null)
        {

            final Calendar cal = HCalendar.computeCalendarFromDate(startDate);

            cal.add(Calendar.HOUR_OF_DAY, getTimeZoneHourDifference(cal));

            this._startDate = cal.getTime();
        }
    }
}
