package ohd.hseb.hefs.utils.datetime;

import java.awt.Dialog.ModalityType;
import java.util.Calendar;
import java.util.Date;

import javax.swing.JComponent;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.util.misc.HCalendar;

import org.apache.commons.lang.ArrayUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * Abstract class providing tools for handling date strings that can be relative or fixed dates. It allows for going in
 * both directions: from date string to date given relative basis dates and producing date strings based on the date,
 * possibly relative to a base date. This class also provides short-cuts to date choosing dialogs
 * {@link HDateChooserDialog} and {@link HRelativeDateChooserDialog} via methods:
 * {@link #createFixedDateChooser(JComponent, HDateChooserListener, Calendar)},
 * {@link #createRelativeDateChooser(JComponent, HDateChooserListener)},
 * {@link #createRelativeDateChooser(JComponent, HDateChooserListener, String)}, and
 * {@link #createRelativeDateChooser(JComponent, HDateChooserListener, String, String[])}. <br>
 * <br>
 * This tool is designed to be used by applications that allow for user input of relative or fixed dates and saving
 * those dates to XML. It includes the ability to specify XML and GUI date formats to use for date strings, date-time
 * strings, and date-time-tz strings. The formats can be set based on application, but are static in nature, so if you
 * modify one application you modify all applications in that virtual machine. The default values of those formats are
 * typically okay (see the javadoc comments on the static attributes that store the formats for their default values).
 * 
 * @author hank.herr
 */
public abstract class HEFSDateTools
{
    //Default date formats.
    public final static String DEFAULT_XML_DATE_FORMAT = "CCYY-MM-DD";
    public final static String DEFAULT_GUI_DATE_FORMAT = "MM-DD-CCYY";
    public final static String DEFAULT_XML_DATETIME_FORMAT = "CCYY-MM-DD hh:mm:ss";
    public final static String DEFAULT_GUI_DATETIME_FORMAT = "MM-DD-CCYY hh:mm:ss";
    public final static String DEFAULT_XML_DATETIME_TZ_FORMAT = "CCYY-MM-DD hh:mm:ss TZC";
    public final static String DEFAULT_GUI_DATETIME_TZ_FORMAT = "MM-DD-CCYY hh:mm:ss TZC";

    public final static String SYSTEM_TIME_BASIS_STR = "T0";
    public final static String TIMESERIES_FORECAST_TIME_BASIS_STR = "tsT0";
    public final static String TIMESERIES_START_TIME_BASIS_STR = "tsStartTime";
    public final static String TIMESERIES_END_TIME_BASIS_STR = "tsEndTime";
    public final static String FIXED_TIME = "fixedTime";
    public final static String RELATIVE_TIME = "relativeTime";
    public final static String HW_CLOCK_TIME_BASIS_STR = "hwClockTime";

    public final static String[] RELATIVE_DATE_BASIS_STRINGS = {SYSTEM_TIME_BASIS_STR,
        TIMESERIES_FORECAST_TIME_BASIS_STR, TIMESERIES_START_TIME_BASIS_STR, TIMESERIES_END_TIME_BASIS_STR,
        HW_CLOCK_TIME_BASIS_STR};

    public final static String[] DATETIME_FORMATS = new String[]{"MM-DD-CCYY hh:mm:ss TZC", "CCYY-MM-DD hh:mm:ss TZC",
        "MM-DD-CCYY hh:mm:ss", "CCYY-MM-DD hh:mm:ss", "MM-DD-CCYY", "CCYY-MM-DD"};

    public final static String[] DATE_UNITS = {"year", "years", "firstDayOfMonth", "lastDayOfMonth", "month", "months",
        "week", "weeks", "day", "days", "hour", "hours"};
    public final static String[] GUI_DATE_UNITS = {"hours", "days", "weeks", "months", "firstDayOfMonth",
        "lastDayOfMonth", "years"};

    private final static int DEFAULT_BASIS_INDEX = -1;

    private static final Logger LOG = LogManager.getLogger(HEFSDateTools.class);

    //NOTE: These static Strings allow for programmers to override the formats as desired.  However,
    //nothing currently modifies these static variables, leaving them as null so that constants are used.
    //Do we want the ability to override these foramts as static variables?

    /**
     * Format used in XML to store the year, month, and day. Default if null: {@link #DEFAULT_XML_DATE_FORMAT}.
     */
    private static String _xmlDateFormat;

    /**
     * Format used in XML to store the year, month, day and time. Default if null: {@link #DEFAULT_XML_DATETIME_FORMAT}.
     */
    private static String _xmlDateTimeFormat;

    /**
     * Format used in XML to store the year, month, day, time, and time zone. Default if null:
     * {@link #DEFAULT_XML_DATETIME_TZ_FORMAT}.
     */
    private static String _xmlDateTimeTZFormat;

    /**
     * Format used in GUI to store the year, month, and day. Default if null: {@link #DEFAULT_GUI_DATE_FORMAT}.
     */
    private static String _guiDateFormat;

    /**
     * Format used in GUI to store the year, month, day, and time. Default if null: {@link #DEFAULT_GUI_DATETIME_FORMAT}
     * .
     */
    private static String _guiDateTimeFormat;

    /**
     * Format used in GUI to store the year, month, day, time, and time zone. Default if null:
     * {@link #DEFAULT_GUI_DATETIME_TZ_FORMAT}.
     */
    private static String _guiDateTimeTZFormat;

    /**
     * Build a relative date string based on the provided info, excluding the base date type from the results.
     * 
     * @param baseDate The basis date of the relative date.
     * @param newDate The fixed date value of the relative date to build.
     * @return The portion of the relative date AFTER the base date; for example '+ 1 day 2 hour'.
     */
    private static String buildFormattedRelativeDateWithoutBaseDate(final Date baseDate, final Date newDate)
    {
        final StringBuilder sb = new StringBuilder();

        final long baseTime = baseDate.getTime();
        final long newTime = newDate.getTime();
        long diffTime = Math.abs(baseTime - newTime);
        int factor;

        if(baseTime < newTime)
        {
            sb.append("+");
            factor = 1;
        }
        else if(baseTime > newTime)
        {
            sb.append("-");
            factor = -1;
        }
        else
        {
            return "";
        }

        //process the year difference
        int years = 0;
        long diffYear = 0;

        final Calendar currCal = HCalendar.computeCalendarFromDate(baseDate);
        Calendar nextCal = HCalendar.computeCalendarFromDate(baseDate);
        nextCal.add(Calendar.YEAR, factor);
        diffYear = Math.abs(baseTime - nextCal.getTimeInMillis());

        while(diffYear < diffTime)
        {
            years++;
            diffTime -= diffYear;

            currCal.add(Calendar.YEAR, factor);
            nextCal.add(Calendar.YEAR, factor);
            diffYear = Math.abs(currCal.getTimeInMillis() - nextCal.getTimeInMillis());
        }

        if(years > 1)
        {
            sb.append(" " + years + " years");
        }
        else if(years == 1)
        {
            sb.append(" " + years + " year");
        }

        //process the month difference
        int months = 0;
        long diffMonth = 0;

        nextCal = HCalendar.computeCalendarFromDate(currCal.getTime());
        nextCal.add(Calendar.MONTH, factor);
        diffMonth = Math.abs(currCal.getTimeInMillis() - nextCal.getTimeInMillis());

        while(diffMonth < diffTime)
        {
            months++;
            diffTime -= diffMonth;

            currCal.add(Calendar.MONTH, factor);
            nextCal.add(Calendar.MONTH, factor);
            diffMonth = Math.abs(currCal.getTimeInMillis() - nextCal.getTimeInMillis());
        }

        if(months > 1)
        {
            sb.append(" " + months + " months");
        }
        else if(months == 1)
        {
            sb.append(" " + months + " month");
        }

        final int weeks = (int)(diffTime / (7 * 24 * HCalendar.MILLIS_IN_HR));
        if(weeks > 1)
        {
            sb.append(" " + weeks + " weeks");
        }
        else if(weeks == 1)
        {
            sb.append(" " + weeks + " week");
        }

        diffTime -= weeks * (7 * 24 * HCalendar.MILLIS_IN_HR);

        final int days = (int)(diffTime / (24 * HCalendar.MILLIS_IN_HR));
        if(days > 1)
        {
            sb.append(" " + days + " days");
        }
        else if(days == 1)
        {
            sb.append(" " + days + " day");
        }

        diffTime -= days * (24 * HCalendar.MILLIS_IN_HR);

        final int hours = (int)(diffTime / HCalendar.MILLIS_IN_HR);
        if(hours > 1)
        {
            sb.append(" " + hours + " hours");
        }
        else if(hours == 1)
        {
            sb.append(" " + hours + " hour");
        }

        diffTime -= hours * (HCalendar.MILLIS_IN_HR);

        final int minutes = (int)(diffTime / HCalendar.MILLIS_IN_MIN);
        if(minutes > 1)
        {
            sb.append(" " + minutes + " minutes");
        }
        else if(minutes == 1)
        {
            sb.append(" " + minutes + " minute");
        }

        return sb.toString().trim();
    }

    /**
     * Build a relative date string based on the provided info.
     * 
     * @param baseDate The basis date of the relative date.
     * @param newDate The fixed date value of the relative date to build.
     * @param relativeDateBasisStr The type of base date used; see RELATIVE_DATE_BASIS_STRINGS.
     * @return A relative date string representing the difference between the two dates. The basis for the relative date
     *         is specified by relativeDateBasisStr.
     */
    public static String buildFormattedRelativeDate(final Date baseDate, final Date newDate, String relativeDateBasisStr)
    {
        final int typeIndex = determineRelativeDateBasisIndex(relativeDateBasisStr);
        if(typeIndex < 0)
        {
            LOG.warn("Invalid date type: " + relativeDateBasisStr + ". Default to system time");
            relativeDateBasisStr = RELATIVE_DATE_BASIS_STRINGS[0];
        }

        return relativeDateBasisStr.trim() + " " + buildFormattedRelativeDateWithoutBaseDate(baseDate, newDate);
    }

    /**
     * This calls the other computeRelativeDateBasisStr.
     * 
     * @param dateString Date string to check.
     * @return Return the basis date string of the passed in date string. For a fixed date, the constant FIXED_TIME is
     *         returned. For a relative time, RELATIVE_TIME is returned. If the string is invalid, null, or empty, null
     *         is returned.
     */
    public static String computeRelativeDateBasisStr(final String dateString)
    {
        return computeRelativeDateBasisStr(dateString, null, null);
    }

    /**
     * I don't know if I like the algorithm used here. It may require some refactoring.
     * 
     * @param dateString The date string to evaluate.
     * @param systemTime
     * @param baseTimeSeries
     * @return The type of time: {@link #FIXED_TIME} if the time is a fixed date; {@link #RELATIVE_TIME} if the time
     *         includes both a base date type and relative addition (+..., -...); one of
     *         {@link #TIMESERIES_START_TIME_BASIS_STR}, {@link #TIMESERIES_END_TIME_BASIS_STR}, or
     *         {@link #TIMESERIES_FORECAST_TIME_BASIS_STR} if the dateString after evaluation equals the appropriate
     *         time. The value {@link #HW_CLOCK_TIME_BASIS_STR} will never be returned.
     */
    private static String computeRelativeDateBasisStr(final String dateString,
                                                      final Calendar systemTime,
                                                      final TimeSeriesArray baseTimeSeries)
    {

        if(dateString == null || dateString.trim().length() == 0 || !isDateStringValid(dateString))
        {
            return null;
        }

        if(isFixedDate(dateString))
        {
            return FIXED_TIME;
        }
        else if(dateString.equalsIgnoreCase(SYSTEM_TIME_BASIS_STR))
        {
            return SYSTEM_TIME_BASIS_STR;
        }
        else
        {
            if(baseTimeSeries == null || systemTime == null)
            {
                return RELATIVE_TIME;
            }

            final long currTime = computeTime(systemTime, baseTimeSeries, dateString);
            if(currTime == baseTimeSeries.getStartTime())
            {
                return TIMESERIES_START_TIME_BASIS_STR;
            }
            else if(currTime == baseTimeSeries.getEndTime())
            {
                return TIMESERIES_END_TIME_BASIS_STR;
            }
            else if(currTime == baseTimeSeries.getForecastTime())
            {
                return TIMESERIES_FORECAST_TIME_BASIS_STR;
            }
            return RELATIVE_TIME;
        }

    }

    /**
     * Calls {@link #computeFixedDate(Calendar, String)} with null for the system time.
     */
    public static Calendar computeFixedDate(final String dateString)
    {
        return computeFixedDate(null, dateString);
    }

    /**
     * Compute a Calendar object based on the passed-in fixed date string.
     * 
     * @param basisTime If not null, then years starting with 0 will be treated as adjustment factors to the date based
     *            on the systemTime. For example, 0001 is the first date AFTER the provided system time where the rest
     *            of the fields are as specified in the provided string. 0002 is the second such date. etc.
     * @param dateString -- fixed date string like "03-10-2010 12:00:00"
     * @return calendar represents the date string
     */
    public static Calendar computeFixedDate(final Calendar basisTime, final String dateString)
    {
        final int formatIndex = determineFixedDateFormatIndex(dateString);

        Calendar cal = null;
        if(formatIndex != DEFAULT_BASIS_INDEX)
        {
            cal = HCalendar.convertStringToCalendar(dateString, DATETIME_FORMATS[formatIndex]);
        }

        if(basisTime != null)
        {
            //Special formatting for if the first digit of the year is 0.
            final int specifiedYear = cal.get(Calendar.YEAR);
            if(specifiedYear < 1000)
            {
                //Set the year of cal to be appropriate for the year 0001.  That is the first year such
                //that the date is AFTER OR ON the basisTime.  
                cal.set(Calendar.YEAR, basisTime.get(Calendar.YEAR));
                // Change to fix Redmine ticket # 53541
                //if(cal.before(basisTime))
                if (!cal.after(basisTime))
                {
                    cal.set(Calendar.YEAR, cal.get(Calendar.YEAR) + 1);
                }

                //Since 0001 is already accounted for, add to the year the amount specified - 1.
                cal.add(Calendar.YEAR, specifiedYear - 1);
            }
        }

        return cal;
    }

    /**
     * Compute relative date.
     * 
     * @param systemTime Null is not allowed
     * @param baseTimeSeries Null allowed; provides the time series start time and end time.
     * @param dateString The date specifying string, either fixed or relative.
     * @return Milliseconds for date, or null if invalid dateString.
     */
    public static Long computeRelativeTime(final Calendar systemTime,
                                           final TimeSeriesArray baseTimeSeries,
                                           final String dateString)
    {
        //Conditions underwhich null is returned.
        if(systemTime == null || dateString == null || dateString.trim().length() <= 0)
        {
            return null;
        }

        //Check if the date string is valid relative date
        if(!isRelativeDate(dateString))
        {
            return null;
        }

        final String[] words = dateString.trim().split(" ");
        String[] dateArray;
        final int typeIndex = determineRelativeDateBasisIndex(words[0]);

        //SHOULD NOT OCCUR HERE
        //need time series date but the time series is null
        //return null
        if((typeIndex == 1 || typeIndex == 2 || typeIndex == 3) && (baseTimeSeries == null))
        {
            LOG.error("Relative date type depends on base time series which is not available.");
            return null;
        }

        //The date string is pure relative date string without date type such as "1 week"
        //set the date type to system time.
        if(typeIndex < 0)
        {
            final Calendar newCal = processRelativeDate(systemTime, words, false);

            if(newCal == null)
            {
                return null;
            }
            else
            {
                return newCal.getTimeInMillis();
            }
        }

        //Now handle relative date string with date type head info

        //Default the base date to system time
        final Calendar baseCal = HCalendar.computeCalendarFromDate(systemTime.getTime());

        //I think the ints in this switch should be constants.
        switch(typeIndex)
        {
            case 0: //for system time
                break;
            case 1: //for TS forecast time
                baseCal.setTimeInMillis(baseTimeSeries.getForecastTime());
                break;
            case 2: //for TS start time
                baseCal.setTimeInMillis(baseTimeSeries.getStartTime());
                break;
            case 3: //for TS end time
                baseCal.setTimeInMillis(baseTimeSeries.getEndTime());
                break;
            case 4: //for hardware clock time
                baseCal.setTimeInMillis(new Date().getTime());
                break;
            default: //default to system time
                break;
        }

        //if the date string only contains the date type, just return the millis.
        if(words.length == 1)
        {
            return baseCal.getTimeInMillis();
        }

        //Identify if the relative date is - or +.  Default to + if the character is not recognized.
        boolean isMinus = false;
        if(words[1].trim().equals("-"))
        {
            isMinus = true;
        }

        //Populate datearray with the words starting after the minus sign word/character.
        dateArray = new String[words.length - 2];
        for(int i = 0; i < dateArray.length; i++)
        {
            dateArray[i] = words[i + 2];
        }

        //Call processRelativeDate and return null if that method returns null.  Otherwise return the time in millis.
        final Calendar newCal = processRelativeDate(baseCal, dateArray, isMinus);
        if(newCal == null)
        {
            return null;
        }
        else
        {
            return newCal.getTimeInMillis();
        }
    }

    /**
     * Calls {@link #computeTime(Calendar, TimeSeriesArray, String, Calendar)} passing in null for the last arg.
     */
    public static Long computeTime(final Calendar systemTime,
                                   final TimeSeriesArray baseTimeSeries,
                                   final String dateString)
    {
        return computeTime(systemTime, baseTimeSeries, dateString, null);
    }

    /**
     * This can be used to check format of the dateString by passing in dummy information for systemTime and/or
     * baseTimeSeries and checking for a null result.
     * 
     * @param systemTime Null allowed if computing fixed date
     * @param baseTimeSeries Null allowed; provides the time series start time and end time.
     * @param dateString The date specifying string, either fixed or relative.
     * @param otherTime The other time that will be used instead of systemTime if not null -AND- a fixed date is used.
     *            This is only useful if the 0### year feature is used for the fixed dates. See
     *            {@link #computeFixedDate(Calendar, String)}.
     * @return Milliseconds for date, or null if invalid dateString.
     */
    public static Long computeTime(final Calendar systemTime,
                                   final TimeSeriesArray baseTimeSeries,
                                   final String dateString,
                                   final Calendar otherTime)
    {
        if(dateString == null || dateString.trim().length() <= 0)
        {
            //LOG.error("Date string is empty.");
            return null;
        }

        if(isFixedDate(dateString))
        {
            if(otherTime != null)
            {
                return computeFixedDate(otherTime, dateString).getTimeInMillis();
            }
            return computeFixedDate(systemTime, dateString).getTimeInMillis();
        }
        else
        {
            return computeRelativeTime(systemTime, baseTimeSeries, dateString);
        }
    }

    /**
     * Calls other computeTime with systemTime being a long.
     * 
     * @param systemTime
     * @param baseTimeSeries
     * @param dateString
     * @return
     */
    public static Long computeTime(final long systemTime, final TimeSeriesArray baseTimeSeries, final String dateString)
    {
        final Calendar systemCal = HCalendar.computeCalendarFromMilliseconds(systemTime);

        return computeTime(systemCal, baseTimeSeries, dateString);
    }

    /**
     * Creates a dialog for choosing fixed dates.
     * 
     * @param parent
     * @param owner Implements a listener whose method is called when the user clicks on OK.
     * @param presentDate The current value of the date being edited.
     * @return the dialog.
     */
    public static HDateChooserDialog createFixedDateChooser(final JComponent parent,
                                                            final HDateChooserListener owner,
                                                            final Calendar presentDate)
    {
        final HDateChooserDialog dateChooser = new HDateChooserDialog();
        dateChooser.addListener(owner);
        dateChooser.setModalityType(ModalityType.APPLICATION_MODAL);
        dateChooser.setLocationRelativeTo(parent);
        dateChooser.setCalendar(presentDate);
        dateChooser.setVisible(true);

        return dateChooser;
    }

    /**
     * Calls createRelativeDateChooser(parent, owner, null, null);
     * 
     * @param parent
     * @param owner Implements a listener whose method is called when the user clicks on OK.
     * @return The dialog.
     */
    public static HRelativeDateChooserDialog createRelativeDateChooser(final JComponent parent,
                                                                       final HDateChooserListener owner)
    {
        return createRelativeDateChooser(parent, owner, null, null);
    }

    /**
     * Calls createRelativeDateChooser(parent, owner, presentDate, null);
     * 
     * @param parent
     * @param owner Implements a listener whose method is called when the user clicks on OK.
     * @param presentDate The current value of the date being modified.
     * @return The dialog.
     */
    public static HRelativeDateChooserDialog createRelativeDateChooser(final JComponent parent,
                                                                       final HDateChooserListener owner,
                                                                       final String presentDate)
    {
        return createRelativeDateChooser(parent, owner, presentDate, null);
    }

    /**
     * Creates a dialog for editing a relative date.
     * 
     * @param parent
     * @param listener Implements a listener whose method is called when the user clicks on OK.
     * @param presentDate The current value of the date being modified.
     * @param comboOptions List of strings specifying the options for the base date of the relative date dialog, such as
     *            "T0" or "tsStartTime".
     * @return Dialog for editing the relative date.
     */
    public static HRelativeDateChooserDialog createRelativeDateChooser(final JComponent parent,
                                                                       final HDateChooserListener listener,
                                                                       String presentDate,
                                                                       final String[] comboOptions)
    {

        final HRelativeDateChooserDialog dateChooser = new HRelativeDateChooserDialog();
        dateChooser.addListener(listener);
        dateChooser.setModalityType(ModalityType.APPLICATION_MODAL);
        dateChooser.setLocationRelativeTo(parent);

        if(comboOptions != null)
        {
            dateChooser.setRelativeDateBasisStrings(comboOptions);
        }
        if(presentDate == null || presentDate.trim().length() <= 0)
        {
            presentDate = HEFSDateTools.SYSTEM_TIME_BASIS_STR;
        }

        dateChooser.setDate(presentDate);
        dateChooser.setVisible(true);
        return dateChooser;
    }

    /**
     * Cleans up relative date, combining terms in the relative date string appropriately.
     * 
     * @param dateString Relative date string to format.
     * @return Formatted relative date string.
     */
    public static String cleanUpAndReformatRelativeDateString(String dateString)
    {
        if(dateString == null || dateString.trim().length() <= 0)
        {
            return null;
        }

        //only include date type like "T0", "tsStartTime" etc.
        if(dateString.trim().split(" ").length == 1)
        {
            return dateString.trim();
        }

        final StringBuffer sb = new StringBuffer();

        //check if the date string include date type and operator
        if(dateString.contains("+") || dateString.contains("-"))
        {
            final int index = Math.max(dateString.indexOf("+"), dateString.indexOf("-"));
            sb.append(dateString.substring(0, index + 1));
            dateString = dateString.substring(index + 1).trim();
        }

        int numYear = 0;
        int numMonth = 0;
        int numFirstDayOfMonth = 0;
        int numLastDayOfMonth = 0;
        int numWeek = 0;
        int numDay = 0;
        int numHour = 0;
        int value;

        final String[] words = dateString.split(" ");

        for(int i = 0; i < words.length - 1; i += 2)
        {
            try
            {
                value = Integer.parseInt(words[i]);
            }
            catch(final NumberFormatException e)
            {
                LOG.error("INVALID date string: " + dateString);
                return null;
            }
            if(words[i + 1].equalsIgnoreCase("year") || words[i + 1].equalsIgnoreCase("years"))
            {
                numYear += value;
            }
            else if(words[i + 1].equals("firstDayOfMonth"))
            {
                numFirstDayOfMonth += value;
            }
            else if(words[i + 1].equals("lastDayOfMonth"))
            {
                numLastDayOfMonth += value;
            }
            else if(words[i + 1].equalsIgnoreCase("month") || words[i + 1].equalsIgnoreCase("months"))
            {
                numMonth += value;
            }
            else if(words[i + 1].equalsIgnoreCase("week") || words[i + 1].equalsIgnoreCase("weeks"))
            {
                numWeek += value;
            }
            else if(words[i + 1].equalsIgnoreCase("day") || words[i + 1].equalsIgnoreCase("days"))
            {
                numDay += value;
            }
            else if(words[i + 1].equalsIgnoreCase("hour") || words[i + 1].equalsIgnoreCase("hours"))
            {
                numHour += value;
            }
        }

        if(numYear > 1)
        {
            sb.append(" " + numYear + " years");
        }
        else if(numYear > 0)
        {
            sb.append(" " + numYear + " year");
        }

        if(numFirstDayOfMonth > 0)
        {
            sb.append(" " + numFirstDayOfMonth + " firstDayOfMonth");
        }

        if(numLastDayOfMonth > 0)
        {
            sb.append(" " + numLastDayOfMonth + " lastDayOfMonth");
        }

        if(numMonth > 1)
        {
            sb.append(" " + numMonth + " months");
        }
        else if(numMonth > 0)
        {
            sb.append(" " + numMonth + " month");
        }

        if(numWeek > 1)
        {
            sb.append(" " + numWeek + " weeks");
        }
        else if(numWeek > 0)
        {
            sb.append(" " + numWeek + " week");
        }

        if(numDay > 1)
        {
            sb.append(" " + numDay + " days");
        }
        else if(numDay > 0)
        {
            sb.append(" " + numDay + " day");
        }

        if(numHour > 1)
        {
            sb.append(" " + numHour + " hours");
        }
        else if(numHour > 0)
        {
            sb.append(" " + numHour + " hour");
        }

        return sb.toString().trim();
    }

    /**
     * @return Index within {@link #RELATIVE_DATE_BASIS_STRINGS} specifying the basis of the provided relative date
     *         string.
     */
    private static int determineRelativeDateBasisIndex(final String dateBasisStr)
    {
        //default to no date type (fixed date)
        int basisIndex = DEFAULT_BASIS_INDEX;

        if(dateBasisStr == null || dateBasisStr.length() <= 0)
        {
            LOG.error("Date type string is empty!");

            return basisIndex;
        }

        //find out the date type from the head of date string
        int i;
        for(i = 0; i < RELATIVE_DATE_BASIS_STRINGS.length; i++)
        {
            if(dateBasisStr.equalsIgnoreCase(RELATIVE_DATE_BASIS_STRINGS[i]))
            {
                basisIndex = i;
                break;
            }
        }

        if(i == RELATIVE_DATE_BASIS_STRINGS.length)
        {
            LOG.error("Relative date basis string '" + dateBasisStr + "' is not recognized; using default '"
                + RELATIVE_DATE_BASIS_STRINGS[basisIndex] + "'.");
        }

        return basisIndex;
    }

    /**
     * @return Index of the format within {@link #DATETIME_FORMATS} that corresponds to the provided date string.
     *         {@link #DEFAULT_BASIS_INDEX} is returned if the string does not match a recognized format.
     */
    private static int determineFixedDateFormatIndex(final String dateString)
    {
        if(dateString == null || dateString.trim().length() == 0)
        {
            return DEFAULT_BASIS_INDEX;
        }

        final String strDate = dateString.trim();
        String firstString;
        try
        {
            for(int i = 0; i < DATETIME_FORMATS.length; i++)
            {
                if(strDate.length() == DATETIME_FORMATS[i].length())
                {
                    firstString = strDate.split("-")[0];
                    if(firstString.length() == DATETIME_FORMATS[i].split("-")[0].length())
                    {
                        if(Integer.parseInt(firstString) > 0)
                        {
                            return i;
                        }
                    }
                }
            }
        }
        catch(final NumberFormatException e)
        {
            return DEFAULT_BASIS_INDEX;
        }

        return DEFAULT_BASIS_INDEX;
    }

    public static String getGUIDateFormat()
    {
        if(_guiDateFormat == null)
        {
            return DEFAULT_GUI_DATE_FORMAT;
        }
        return _guiDateFormat;
    }

    public static String getGUIDateTimeFormat()
    {
        if(_guiDateTimeFormat == null)
        {
            return DEFAULT_GUI_DATETIME_FORMAT;
        }
        return _guiDateTimeFormat;
    }

    public static String getGUIDateTimeTZFormat()
    {
        if(_guiDateTimeTZFormat == null)
        {
            return DEFAULT_GUI_DATETIME_TZ_FORMAT;
        }

        return _guiDateTimeTZFormat;
    }

    public static String getXmlDateFormat()
    {
        if(_xmlDateFormat == null)
        {
            return DEFAULT_XML_DATE_FORMAT;
        }
        return _xmlDateFormat;
    }

    public static String getXmlDateTimeFormat()
    {
        if(_xmlDateTimeFormat == null)
        {
            return DEFAULT_XML_DATETIME_FORMAT;
        }
        return _xmlDateTimeFormat;
    }

    public static String getXmlDateTimeTZFormat()
    {
        if(_xmlDateTimeTZFormat == null)
        {
            return DEFAULT_XML_DATETIME_TZ_FORMAT;
        }

        return _xmlDateTimeTZFormat;
    }

    /**
     * check if the date string is valid fixed date or relative date
     * 
     * @param dateString
     * @return
     */
    public static boolean isDateStringValid(final String dateString)
    {
        //check if the date string is null or empty
        if(dateString == null || dateString.trim().length() <= 0)
        {
            return false;
        }

        //check if the date string is fixed date
        if(isFixedDate(dateString))
        {
            return true;
        }

        // check if the date string is relative date
        return isRelativeDate(dateString);
    }

    /**
     * @param basisStr The basis to check
     * @return True if the basisStr is recognized by this class.
     */
    public static boolean isRelativeDateBasisStrValid(final String basisStr)
    {
        if(basisStr == null || basisStr.trim().length() <= 0)
        {
            return false;
        }

        for(final String type: RELATIVE_DATE_BASIS_STRINGS)
        {
            if(basisStr.equalsIgnoreCase(type))
            {
                return true;
            }
        }

        return false;
    }

    /**
     * check if the date string is valid fixed date
     * 
     * @param dateString
     * @return boolean value
     */
    public static boolean isFixedDate(final String dateString)
    {
        //check if the date string is null or empty string
        if(dateString == null || dateString.trim().length() <= 0)
        {
            return false;
        }

        //check if the date string is fixed date
        if(determineFixedDateFormatIndex(dateString) == DEFAULT_BASIS_INDEX)
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    /**
     * check if the date string is pure relative date string like "1 day 2 hours"
     * 
     * @param dateString
     * @return
     */
    private static boolean isPureRelativeDatePair(final String[] dates)
    {
        //check if the date string array is null or empty
        if(dates == null || dates.length <= 0 || ((dates.length % 2) != 0))
        {
            return false;
        }

        boolean isDateUnitValid = false;
        int dateValue;
        for(int i = 0; i < dates.length; i += 2)
        {
            //check date number
            try
            {
                dateValue = Integer.parseInt(dates[i]);

                //The date value should be greater than 0
                if(dateValue <= 0)
                {
                    return false;
                }
            }
            catch(final NumberFormatException e)
            {
                return false;
            }

            //check date unit
            isDateUnitValid = false;
            for(int j = 0; j < DATE_UNITS.length; j++)
            {
                if(dates[i + 1].trim().equalsIgnoreCase(DATE_UNITS[j]))
                {
                    isDateUnitValid = true;
                    break;
                }
            }

            if(isDateUnitValid == false)
            {
                return false;
            }
        }

        return true;
    }

    /**
     * @return True if the provided unit is valid; i.e., if it is within {@link #DATE_UNITS}.
     */
    public static boolean isValidDateUnit(final String unit)
    {
        return ArrayUtils.contains(DATE_UNITS, unit);
    }

    /**
     * check if the date string is valid relative date
     * 
     * @param dateString
     * @return True if the date string is relative, false otherwise.
     */
    public static boolean isRelativeDate(final String dateString)
    {
        //check if the date string is null or empty string
        if(dateString == null || dateString.trim().length() <= 0)
        {
            return false;
        }

        final String[] words = dateString.trim().split(" ");

        final boolean validDateBasisStr = isRelativeDateBasisStrValid(words[0]);

        //case 1: only include the date type like "T0", "tsT0"
        if(words.length == 1)
        {
            return validDateBasisStr;
        }
        //case 2: allow for a + or - but nothing after.
        else if(words.length == 2)
        {
            if(words[1].equals("+") || words[1].equals("-"))
            {
                return validDateBasisStr;
            }
            return false;
        }
        else
        {
            // case 3: date string includes date type and operator like "T0 +/- 1 hour"
            if(validDateBasisStr == true)
            {
                //check if the operator is valid
                if(words[1].equals("+") || words[1].equals("-"))
                {
                    //check the pure relative date array
                    final String[] dates = new String[words.length - 2];
                    for(int i = 0; i < dates.length; i++)
                    {
                        dates[i] = words[i + 2];
                    }

                    return isPureRelativeDatePair(dates);
                }
                else
                {
                    return false;
                }
            }
            // case 3: pure relative date string like "1 day 1 hour"
            else
            {
                return isPureRelativeDatePair(words);
            }
        }
    }

    /**
     * This method is used internally once the needed arguments has been determined.
     * 
     * @param baseDate The basis time for the relative date.
     * @param dateArray Array of relative date addend components, each specifying a unit and type.
     * @param isMinus True if the relative date uses a -, or false for a +.
     * @return The value of the relative date.
     */
    private static Calendar processRelativeDate(final Calendar baseDate, final String[] dateArray, final boolean isMinus)
    {
        if(baseDate == null || dateArray == null)
        {
            return null;
        }

        if(dateArray.length == 0)
        {
            return baseDate;
        }

        final Calendar cal = (Calendar)baseDate.clone();

        int value = 0;
        String dateUnit = null;
        for(int i = 0; i < dateArray.length - 1; i += 2)
        {
            try
            {
                value = Integer.parseInt(dateArray[i]);
            }
            catch(final NumberFormatException e)
            {
                return null;
            }

            dateUnit = dateArray[i + 1].trim().toLowerCase();

            //minusSign tracks if the minus flag is true (-1) or false (1).
            //The value is made negative based on the minus flag, and then the minus sign is used later to adjust the value if needed.
            int minusSign = 1;
            if(isMinus == true)
            {
                value = -value;
                minusSign = -1;
            }

            if(dateUnit.equalsIgnoreCase("year") || dateUnit.equalsIgnoreCase("years"))
            {
                cal.add(Calendar.YEAR, value);
            }
            else if(dateUnit.equalsIgnoreCase("firstDayOfMonth"))
            {
                //Subtract one from the absolute value of the 'value' if the current day of the month is 1 -or- whenever the direction is minus.
                //In the latter case, the first firstDayOfMonth is alway the current month's firstDayOfMonth.
                if((cal.get(Calendar.DAY_OF_MONTH) == 1) || isMinus)
                {
                    value = value + -1 * minusSign; //-4 becomes -3; 4 becomes 3
                }
                cal.add(Calendar.MONTH, value);
                cal.set(Calendar.DAY_OF_MONTH, 1);
            }
            else if(dateUnit.equalsIgnoreCase("lastDayOfMonth"))
            {
                //Subtract one from the absolute value of 'value' if the current day of the month is the last day -or- whenever the direction is positive.
                //In the latter case, the first lastDayOfMonth is always the current month's lastDayOfMonth.
                final int lastDay = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
                if((cal.get(Calendar.DAY_OF_MONTH) == lastDay) || !isMinus)
                {
                    value = value + -1 * minusSign;
                }
                cal.add(Calendar.MONTH, value);
                cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
            }
            else if(dateUnit.equalsIgnoreCase("month") || dateUnit.equalsIgnoreCase("months"))
            {
                cal.add(Calendar.MONTH, value);
            }
            else if(dateUnit.equalsIgnoreCase("week") || dateUnit.equalsIgnoreCase("weeks"))
            {
                cal.add(Calendar.WEEK_OF_YEAR, value);
            }
            else if(dateUnit.equalsIgnoreCase("day") || dateUnit.equalsIgnoreCase("days"))
            {
                cal.add(Calendar.DAY_OF_MONTH, value);
            }
            else if(dateUnit.equalsIgnoreCase("hour") || dateUnit.equalsIgnoreCase("hours"))
            {
                cal.add(Calendar.HOUR_OF_DAY, value);
            }
            else if(dateUnit.equalsIgnoreCase("minute") || dateUnit.equalsIgnoreCase("minutes"))
            {
                cal.add(Calendar.MINUTE, value);
            }
            else if(dateUnit.equalsIgnoreCase("second") || dateUnit.equalsIgnoreCase("seconds"))
            {
                cal.add(Calendar.SECOND, value);
            }
            else if(dateUnit.equalsIgnoreCase("millisecond") || dateUnit.equalsIgnoreCase("milliseconds"))
            {
                cal.add(Calendar.MILLISECOND, value);
            }
        }

        return cal;
    }

    public static void setGUIDateFormat(final String guiDateFormat)
    {
        HEFSDateTools._guiDateFormat = guiDateFormat;
    }

    public static void setGUIDateTimeFormat(final String guiDateTimeFormat)
    {
        HEFSDateTools._guiDateTimeFormat = guiDateTimeFormat;
    }

    public static void setGUIDateTimeTZFormat(final String _guiDateTimeTZFormat)
    {
        HEFSDateTools._guiDateTimeTZFormat = _guiDateTimeTZFormat;
    }

    public static void setXmlDateFormat(final String xmlDateFormat)
    {
        HEFSDateTools._xmlDateFormat = xmlDateFormat;
    }

    public static void setXmlDateTimeFormat(final String xmlDateTimeFormat)
    {
        HEFSDateTools._xmlDateTimeFormat = xmlDateTimeFormat;
    }

    public static void setXmlDateTimeTZFormat(final String _xmlDateTimeTZFormat)
    {
        HEFSDateTools._xmlDateTimeTZFormat = _xmlDateTimeTZFormat;
    }

}
