package ohd.hseb.hefs.utils.datetime;

import java.util.ArrayList;
import java.util.List;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import javax.swing.JComboBox;

import ohd.hseb.hefs.utils.gui.components.SeparatableJComboBoxRenderer;

/**
 * Tools related to special time zones created for HEFS and GraphGen. Examples of those time zone ids incude "AST/AKDT",
 * "CST/CDT", and so on. Without such easily identified ids, users would need to use "America/Anchorage" and
 * "America/Chicago" to identify the time zones.
 * 
 * @author hankherr
 */
public abstract class HEFSTimeZoneTools
{

    public static final String DEFAULT_TIMEZONE = "GMT";

    /**
     * Identifiers specific to Graphics Generator. Some overlap Java time zones, which is becuase the java provided one
     * has problems with DST (and have been deprecated, more or less).
     */
    private static final String[] ADDITIONAL_TIME_ZONE_IDENTIFIERS = new String[]{DEFAULT_TIMEZONE, "AST", "AST/AKDT",
        "CST", "CST/CDT", "EST", "EST/EDT", "HST", "HST/HADT", "MST", "MST/MDT", "PST", "PST/PDT"};

    /**
     * null entries are only valid if the DEFAULT_TIMEZONE_DISPLAYED_STRINGS time zone is a valid time zone already. A
     * new time zone will be constructed that has the offset but DST turned off.
     */
    private static final TimeZone[] MAPPED_TIME_ZONES = new TimeZone[]{TimeZone.getTimeZone(DEFAULT_TIMEZONE), null,
        TimeZone.getTimeZone("America/Anchorage"), null, TimeZone.getTimeZone("America/Chicago"), null,
        TimeZone.getTimeZone("America/New_York"), null, TimeZone.getTimeZone("America/Adak"), null,
        TimeZone.getTimeZone("America/Denver"), null, TimeZone.getTimeZone("America/Los_Angeles")};

    public static List<String> VALID_TIME_ZONE_IDENTIFIERS = new ArrayList<String>();

    /**
     * @return The time zone instance corresponding to the provided id. The {@link #ADDITIONAL_TIME_ZONE_IDENTIFIERS}
     *         are searched, first, for the provided id. If found, then the {@link #MAPPED_TIME_ZONES} variable is used
     *         to determine which time zone to return. If not found, then {@link TimeZone#getTimeZone(String)} is called
     *         for the id.
     */
    public static TimeZone retrieveTimeZone(final String id)
    {
        for(int index = 0; index < ADDITIONAL_TIME_ZONE_IDENTIFIERS.length; index++)
        {
            if(ADDITIONAL_TIME_ZONE_IDENTIFIERS[index].equals(id))
            {
                if(MAPPED_TIME_ZONES[index] != null)
                {
                    return MAPPED_TIME_ZONES[index];
                }
                else
                {
                    final TimeZone tz = TimeZone.getTimeZone(id);
                    final SimpleTimeZone returnedZone = new SimpleTimeZone(tz.getRawOffset(), id);
                    MAPPED_TIME_ZONES[index] = returnedZone;
                    return returnedZone;
                }
            }
        }
        return TimeZone.getTimeZone(id);
    }

    /**
     * @return List of valid time zone identifiers, startign with {@link #ADDITIONAL_TIME_ZONE_IDENTIFIERS} followed by
     *         a separator, {@link SeparatableJComboBoxRenderer#getSeparatorItem()}, followed by time zone identifiers
     *         found via {@link TimeZone#getAvailableIDs()}. By using
     *         {@link SeparatableJComboBoxRenderer#getSeparatorItem()}, the returned list is appropriate to use in a
     *         {@link JComboBox} that uses a {@link SeparatableJComboBoxRenderer} renderer. The returned list is stored
     *         as {@link #VALID_TIME_ZONE_IDENTIFIERS}, so that the list must only be built once when this is first
     *         called.
     */
    public static List<String> getValidTimeZoneIdentifiersIncludingSeparators()
    {
        if(VALID_TIME_ZONE_IDENTIFIERS.size() == 0)
        {
            for(int i = 0; i < ADDITIONAL_TIME_ZONE_IDENTIFIERS.length; i++)
            {
                VALID_TIME_ZONE_IDENTIFIERS.add(ADDITIONAL_TIME_ZONE_IDENTIFIERS[i]);
            }
            final List<String> firstPart = new ArrayList<String>();
            firstPart.addAll(VALID_TIME_ZONE_IDENTIFIERS);

            VALID_TIME_ZONE_IDENTIFIERS.add(SeparatableJComboBoxRenderer.getSeparatorItem());

            final String[] javaTZs = TimeZone.getAvailableIDs();
            for(int i = 0; i < javaTZs.length; i++)
            {
                if(firstPart.indexOf(javaTZs[i]) < 0)
                {
                    VALID_TIME_ZONE_IDENTIFIERS.add(javaTZs[i]);
                }
            }
        }
        return VALID_TIME_ZONE_IDENTIFIERS;
    }

    /**
     * @return True if the provided id is within {@link #VALID_TIME_ZONE_IDENTIFIERS}.
     */
    public static boolean isTimeZoneValid(final String id)
    {
        return (getValidTimeZoneIdentifiersIncludingSeparators().indexOf(id) >= 0);
    }

}
