package ohd.hseb.hefs.utils.datetime;

import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Vector;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.ScrollPaneConstants;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.BevelBorder;

import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.util.data.DataSet;

/**
 * A dialog for specifying relative dates. It is possible for an invalid relative date to be specified, so the caller
 * must be able to determine if its valid and act responsibly.
 * 
 * @author hank.herr
 */
public class HRelativeDateChooserDialog extends JDialog implements ActionListener
{

    private static final long serialVersionUID = -6573864883282278519L;

    final static String CLASSNAME = "HRelativeDateChooserDialog";

    private final static String[] BUTTON_NAMES = {"OK", "Cancel", "Reset"};
    private final static String[] OPERATION_CHOICES = {"+", "-"};

    private final static int OKAY_BUTTON = 0;
    private final static int CANCEL_BUTTON = 1;
    private final static int RESET_BUTTON = 2;

    private JPanel _additionPanel;
    private Box _buttonPanel;
    private JButton[] _buttons;
    private JTextField _currentValueTxt;
    private String[] _dateTypes = null;
    private JPanel _fullPanel;
    private JComboBox _operationChoices;

    public List<HDateChooserListener> _listeners = new ArrayList<HDateChooserListener>();

    private JButton _pushButton;
    private JSpinner _quantitySpinner;

    //Attributes
    private JComboBox _symbolChoices;
    private JPanel _symbolPanel;
    private JScrollPane _txtSP;

    private JComboBox _unitChoices;

    public HRelativeDateChooserDialog()
    {
        init(null);
    }

    public HRelativeDateChooserDialog(final String[] dateTypes)
    {
        init(dateTypes);
    }

    @Override
    public void actionPerformed(final ActionEvent evt)
    {
        if(evt.getSource() == _pushButton)
        {
            processAddAction(evt);
        }
        else if(evt.getSource() == _buttons[OKAY_BUTTON])
        {
            final String relDateStr = this.getDateString();

            //Check for relative date validity.  If invalid, the dialog will NOT close!
            if(!HEFSDateTools.isRelativeDate(relDateStr))
            {
                JOptionPane.showMessageDialog(this,
                                              "Invalid relative date specified: " + relDateStr
                                                  + "\nCheck the date component text field.",
                                              "Badly Formatted Relative Date!",
                                              JOptionPane.ERROR_MESSAGE);
            }
            //Otherwise call the listeners and close.
            else
            {
                int i;
                for(i = 0; i < _listeners.size(); i++)
                {
                    ((_listeners.get(i))).dateChosen(this);
                }
                setVisible(false);
            }
        }
        else if(evt.getSource() == _buttons[CANCEL_BUTTON])
        {
            int i;
            for(i = 0; i < _listeners.size(); i++)
            {
                ((_listeners.get(i))).dateChooseCancelled(this);
            }

            setVisible(false);
        }
        else if(evt.getSource() == _buttons[RESET_BUTTON])
        {
            processResetAction(evt);
        }

    }

    /**
     * Adds the passed in HDateChooserOwner to the Vector of owners.
     * 
     * @param owner A class that implements the HDateChooserOwner interface.
     */
    public void addListener(final HDateChooserListener owner)
    {
        _listeners.add(owner);
    }

    /**
     * Create the display.
     */
    public void createDisplay()
    {
        //Create the buttons.
        final int[] groupsizes = {3};
        _buttons = HSwingFactory.createJButtons(BUTTON_NAMES, null, this);
        _buttonPanel = HSwingFactory.createJButtonBox(_buttons, groupsizes, BoxLayout.X_AXIS);

        //Create the symbol panel.
        _symbolChoices = HSwingFactory.createJComboBox(getDateTypes(), null);
        _operationChoices = HSwingFactory.createJComboBox(OPERATION_CHOICES, null);
        _symbolPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        _symbolPanel.add(_symbolChoices);
        _symbolPanel.add(_operationChoices);

        //create the addition panel.
        _pushButton = HSwingFactory.createJButton(">>", null, this);
        final SpinnerNumberModel spinnerModel = new SpinnerNumberModel(1, 0, null, 1);
        _quantitySpinner = new JSpinner(spinnerModel);

        _unitChoices = HSwingFactory.createJComboBox(HEFSDateTools.GUI_DATE_UNITS, null);
        _txtSP = HSwingFactory.createScrollPanedJTextField(null, "", true);
        _txtSP.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
        _currentValueTxt = (JTextField)(_txtSP.getViewport().getView());
        final JPanel unitsPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        unitsPanel.add(_unitChoices);
        unitsPanel.add(_pushButton);

        final JPanel subPanel = new JPanel(new GridBagLayout());
        Insets theinsets = new Insets(5, 5, 5, 5);
        GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                             0,
                                                                             1,
                                                                             1,
                                                                             1,
                                                                             1,
                                                                             GridBagConstraints.WEST,
                                                                             GridBagConstraints.BOTH,
                                                                             theinsets,
                                                                             0,
                                                                             0);
        subPanel.add(_quantitySpinner, constraints);
        theinsets = new Insets(0, 0, 0, 0);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          0,
                                                          1,
                                                          1,
                                                          20,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        subPanel.add(unitsPanel);

        _additionPanel = new JPanel(new GridLayout(1, 1));
        _additionPanel.add(_txtSP);
        _additionPanel.setBorder(new BevelBorder(BevelBorder.RAISED));

        //Create the full panel.
        _fullPanel = new JPanel();
        _fullPanel.setLayout(new GridBagLayout());
        theinsets = new Insets(5, 5, 5, 0);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          10,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        _fullPanel.add(_symbolPanel, constraints);

        theinsets = new Insets(5, 0, 5, 5);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          0,
                                                          1,
                                                          1,
                                                          20,
                                                          10,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        _fullPanel.add(subPanel, constraints);
        theinsets = new Insets(5, 0, 5, 5);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          1,
                                                          3,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        _fullPanel.add(_additionPanel, constraints);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          2,
                                                          3,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        _fullPanel.add(_buttonPanel, constraints);

        getContentPane().add(_fullPanel);
        pack();

        this.getRootPane().setDefaultButton(this._buttons[0]);
    }

    public String getDateString()
    {
        final StringBuilder sb = new StringBuilder();
        sb.append((String)(_symbolChoices.getSelectedItem()));
        final String result = _currentValueTxt.getText().trim();
        if(result.length() == 0)
        {
            return sb.toString().trim();
        }

        sb.append(" " + (String)_operationChoices.getSelectedItem());
        sb.append(" " + result);

        return sb.toString().trim();
    }

    public String[] getDateTypes()
    {
        if(_dateTypes == null)
        {
            _dateTypes = HEFSDateTools.RELATIVE_DATE_BASIS_STRINGS;
        }

        return _dateTypes;
    }

    /**
     * Returns the index of the symbol String passed in the _symbolChoices component. MISSING is returned if it is not
     * found.
     * 
     * @param symbolString
     * @return index
     */
    public int getSymbolIndex(final String symbolString)
    {
        int i;
        if(getDateTypes() == null || getDateTypes().length <= 0)
        {
            return (int)DataSet.MISSING;
        }

        for(i = 0; i < getDateTypes().length; i++)
        {
            if(symbolString.trim().equalsIgnoreCase(getDateTypes()[i]))
            {
                return i;
            }
        }
        return (int)DataSet.MISSING;
    }

    private void init(final String[] dateTypes)
    {
        setTitle("Choose Relative Date/Time");
        _listeners = new Vector<HDateChooserListener>();
        if(dateTypes != null)
        {
            setRelativeDateBasisStrings(dateTypes);
        }
        createDisplay();
    }

    /**
     * Returns true if the passed in String is a valid date string. Valid format is
     * "<date type> <+/-> <quantity> <unit> <quantity> <unit> ...". See the code in
     * HCalendar.computeCalendarRelativeToPresent. new date string is:
     * "TO/tsTO/tsStartTime/tsEndTime +/-<quantity> <unit> +/-<quantity> <unit> ..."
     * 
     * @param dateString
     * @return
     */
    public boolean isValid(final String dateString)
    {
        String working = dateString.trim();
        if(working.length() == 0)
        {
            return true;
        }

        //check for valid date type
        String[] words = working.split(" ");

        if(HEFSDateTools.isRelativeDateBasisStrValid(words[0]) == false)
        {
            return false;
        }

        working = working.substring(words[0].length(), working.length()).trim();

        //only include the date type
        if(working.length() == 0)
        {
            return true;
        }

        //check for valid operator
        final String operator = working.substring(0, 1);
        if(operator.equals("+") || operator.equals("-"))
        {
            working = working.substring(1, working.length()).trim();
        }
        else
        {
            return false;
        }

        //check valid date quantity/date unit pair
        words = working.split(" ");

        for(int i = 0; i < words.length - 1; i += 2)
        {
            try
            {
                Integer.parseInt(words[i]);
            }
            catch(final NumberFormatException e)
            {
                System.out.println("Invalid relative date string: " + dateString);
                return false;
            }

            if(HEFSDateTools.isValidDateUnit(words[i + 1]) == false)
            {
                System.out.println("Invalid relative date string: " + dateString);
                return false;
            }
        }

        return true;

    }

    private void processAddAction(final java.awt.event.ActionEvent evt)
    {

        final StringBuilder sb = new StringBuilder();
        sb.append(this._currentValueTxt.getText().trim());

        //Get the integer value of the quantity text field and get the unit String.
        int quantity = 0;

        //Get the string values.
        quantity = ((Integer)_quantitySpinner.getValue()).intValue();

        sb.append(" " + quantity);
        sb.append(" " + (String)this._unitChoices.getSelectedItem());

        this._currentValueTxt.setText(sb.toString());
    }

    private void processResetAction(final java.awt.event.ActionEvent evt)
    {
        this._currentValueTxt.setText("");
    }

    /**
     * Sets the date displayed to be that in the String passed in.
     * 
     * @param value
     */
    public boolean setDate(final String value)
    {
        if(!isValid(value))
        {
            return false;
        }

        //Do the symbol.
        String working = value.trim();
        if(working.length() == 0)
        {
            _currentValueTxt.setText("");
            return true;
        }

        final String[] words = working.split(" ");

        _symbolChoices.setSelectedIndex(ListTools.indexOfCaseInsensitive(words[0], Arrays.asList(getDateTypes())));

        if(words.length == 1)
        {
            return true;
        }

        //Do the operator
        _operationChoices.setSelectedItem(words[1]);

        //Do the rest
        working = working.substring(words[0].length() + 1 + words[1].length(), working.length()).trim();
        _currentValueTxt.setText(working);

        return true;
    }

    public void setRelativeDateBasisStrings(final String[] datePrefixes)
    {
        final String[] validOptions = StringTools.buildArrayOfStringsFromAcceptableArray(datePrefixes,
                                                                                         HEFSDateTools.RELATIVE_DATE_BASIS_STRINGS);

        if(validOptions.length <= 0)
        {
            return;
        }

        this._dateTypes = validOptions;

        if(this._symbolChoices != null)
        {
            this._symbolChoices.removeActionListener(this);

            this._symbolChoices.removeAllItems();

            for(final String type: validOptions)
            {
                this._symbolChoices.addItem(type);
            }

            this._symbolChoices.addActionListener(this);
        }
    }

}
