/*
 * Created on Mar 25, 2009 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.hefs.utils.dist;

import org.apache.commons.math.distribution.ContinuousDistribution;
import org.apache.commons.math.distribution.DiscreteDistribution;

/**
 * The OHD distribution library highest level interface. It provides basic functionality that any caller needs to be
 * able to access the critical probability distribution functions and interpret the results. In addition to providing
 * basic functions such as {@link #functionCDF(Number)} and {@link #functionPDF(Number)}, it also assumes that the
 * parameters have been indexed, so that callers can, without knowledge of the distribution or its parameters, set its
 * parameters blindly and use it.<br>
 * <br>
 * {@link ContinuousDistribution} and {@link DiscreteDistribution} provide two implementations of this interface.
 * 
 * @author hank.herr
 * @param <E> The type of variable modeled by the distribution; typically Double (see {@link ContinuousDistribution}) or
 *            Integer (see {@link DiscreteDistribution}).
 */
public interface Distribution<E extends Number>
{

    /**
     * @param index Index of the parameter to set.
     * @param value Value of the parameter.
     */
    public void setParameter(final int index, final Number value);

    /**
     * Sets all the parameters based on data in the provided array.
     * 
     * @param parameters The parameter values.
     * @throws IllegalArgumentException if there are not enough parameters provided. It should NOT throw an exception if
     *             too many are provided (just use what it needs).
     */
    public void setParameters(final Number[] parameters) throws IllegalArgumentException;

    /**
     * @param index Index of parameter to acquire.
     * @return The parameter value.
     */
    public Number getParameter(final int index);

    /**
     * @return The double that represents missing values for the returns of {@link #functionCDF(Number)},
     *         {@link #functionExceedance(Number)}, {@link #functionInverseCDF(double)}, and
     *         {@link #functionPDF(Number)}.
     */
    public double getMissing();

    /**
     * @return True if the provided value is equivalent to missing, however it must be checked. It should be the same
     *         value as {@link #getMissing()}, but if NaN is used and direct comparison with {@link #getMissing()}'s
     *         return value is not possible, so this method is provided.
     */
    public boolean isMissing(double value);

    /**
     * @param value The value to evaluate the cdf at.
     * @return The cumulative distribution function evaluated at the passed in value.
     */
    public double functionCDF(E value);

    /**
     * @param value The value to evaluate the pdf at.
     * @return The probability density function evaluated at the passed in value.
     */
    public double functionPDF(E value);

    /**
     * Return the inverse cdf evaluated at the passed in probability. Note that for discrete distributions, the double
     * returned may actually be an integer; just cast it to int to get the appropriate value.
     * 
     * @param prob The probability to evaluate the inverse cdf at.
     * @return The inverse cdf value.
     */
    public double functionInverseCDF(double prob);

    /**
     * @param value The value to evaluate the exceedance function at.
     * @return The exceedance function evaluated at the passed in value.
     */
    public double functionExceedance(E value);
}
