package ohd.hseb.hefs.utils.dist;

import java.util.Collection;
import java.util.EnumSet;
import java.util.List;

import ohd.hseb.hefs.utils.dist.types.ContinuousDist;
import ohd.hseb.hefs.utils.plugins.DefaultFactory;
import ohd.hseb.hefs.utils.plugins.DefaultFactoryException;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.xml.XMLVariable;

import com.google.common.collect.Lists;

/**
 * General distribution enumeration. It provides tools for loading instances of distributions based on the type.
 * 
 * @author hankherr
 */
public enum DistributionType
{
    BETA, BINOMIAL, EMPIRICAL, GAMMA, LOGLOGISTIC, LOGNORMAL, LOGWEIBULL, NORMAL, PARETO, PEARSON_TYPE_3, WAKEBY, WEIBULL, WEIBULL3;

    /**
     * A set of the distributions that can be fit to moments.
     */
    public static EnumSet<DistributionType> MOMENTS_FITTING_DISTRIBUTIONS = EnumSet.noneOf(DistributionType.class);

    /**
     * A set of the distributions that can be fit to moments with bounded lower end and unbounded upper end; good for
     * precipitation, stream flow, and stage variable types.
     */
    public static EnumSet<DistributionType> MOMENTS_FITTING_POSTIVE_BOUNDED_DISTRIBUTIONS = EnumSet.noneOf(DistributionType.class);

    /**
     * A set of the distributions that can be fit to data.
     */
    public static EnumSet<DistributionType> DATA_FITTING_DISTRIBUTIONS = EnumSet.noneOf(DistributionType.class);

    /**
     * A set of the distributions that can be fit to data with bounded lower end and unbounded upper end.
     */
    public static EnumSet<DistributionType> DATA_FITTING_POSITIVE_BOUNDED_DISTRIBUTIONS = EnumSet.noneOf(DistributionType.class);

    /**
     * A set of the distributions that can be fit to l-moments.
     */
    public static EnumSet<DistributionType> LMOMENTS_FITTING_DISTRIBUTIONS = EnumSet.noneOf(DistributionType.class);

    /**
     * Initializes the lists.
     */
    static
    {
        for(final DistributionType type: DistributionType.values())
        {
            try
            {
                final Distribution dist = instantiateDistribution(type);
                if(dist instanceof MomentsFittingDistribution)
                {
                    MOMENTS_FITTING_DISTRIBUTIONS.add(type);
                }
                //I know this could be done within the if check above, but I keep it outside because it is
                //easier to read this way.
                if((dist instanceof ContinuousDist) && (dist instanceof MomentsFittingDistribution))
                {
                    final ContinuousDist contDist = (ContinuousDist)dist;
                    if(contDist.domainHasLowerBound() && !contDist.domainHasUpperBound())
                    {
                        MOMENTS_FITTING_POSTIVE_BOUNDED_DISTRIBUTIONS.add(type);
                    }
                }
                if(dist instanceof DataFittingDistribution)
                {
                    DATA_FITTING_DISTRIBUTIONS.add(type);
                }
                //I know this could be done within the if check above, but I keep it outside because it is
                //easier to read this way.
                if((dist instanceof ContinuousDist) && (dist instanceof DataFittingDistribution))
                {
                    final ContinuousDist contDist = (ContinuousDist)dist;
                    if(contDist.domainHasLowerBound() && !contDist.domainHasUpperBound())
                    {
                        DATA_FITTING_POSITIVE_BOUNDED_DISTRIBUTIONS.add(type);
                    }
                }
                if(dist instanceof LMomentsFittingDistribution)
                {
                    LMOMENTS_FITTING_DISTRIBUTIONS.add(type);
                }
            }
            catch(final DefaultFactoryException e)
            {
                //Assume that the distributions that cannot be loaded will not be processed into an EnumSet.  
//                e.printStackTrace();
            }

        }
    };

    public static class Variable extends XMLVariable<DistributionType>
    {
        public Variable(final String tag)
        {
            super(tag);
        }

        public Variable(final String tag, final DistributionType value)
        {
            super(tag, value);
        }

        @Override
        protected DistributionType readVariable(final String text)
        {
            return valueOf(text.toUpperCase());
        }
    }

    private final static String DIST_PACKAGE_NAME = "ohd.hseb.hefs.utils.dist.types";
    private final static String CLASS_NAME_SUFFIX = "Dist";

    /**
     * @return The index of the {@link Distribution} int the provided {@link Collection} of instances that matches the
     *         provided {@link DistributionType}.
     */
    public static int indexOfDistribution(final Collection<Distribution> distributions,
                                          final DistributionType desiredType)
    {
        final List<Distribution> dists = Lists.newArrayList(distributions);
        for(int i = 0; i < dists.size(); i++)
        {
            if(isDistributionOfType(dists.get(i), desiredType))
            {
                return i;
            }
        }
        return -1;
    }

    /**
     * @return The {@link Distribution} from the provided {@link Collection} of intances that matches the provided
     *         {@link DistributionType}.
     */
    public static Distribution retrieveDistribution(final Collection<Distribution> distributions,
                                                    final DistributionType desiredType)
    {
        for(final Distribution dist: distributions)
        {
            if(isDistributionOfType(dist, desiredType))
            {
                return dist;
            }
        }
        return null;
    }

    /**
     * @return True if the provided {@link Distribution} matches the provided {@link DistributionType}.
     */
    public static boolean isDistributionOfType(final Distribution dist, final DistributionType type)
    {
        return dist.getClass().getSimpleName().toUpperCase().startsWith(type.toString().toUpperCase());
    }

    /**
     * @return {@link DistributionType} instance corresponding to the provided distribution.
     */
    public static DistributionType getDistributionType(final Distribution dist)
    {
        return DistributionType.valueOf(dist.getClass().getSimpleName().replaceAll(CLASS_NAME_SUFFIX, "").toUpperCase());
    }

    /**
     * Instantiates the distribution with the name provided following the rules of {@link DefaultFactory}.
     */
    public static Distribution instantiateDistribution(final DistributionType type) throws DefaultFactoryException
    {
        final String distSubName = StringTools.capitalize(type.toString());

        @SuppressWarnings({"rawtypes", "unchecked"})
        final DefaultFactory<Distribution> classFactory = new DefaultFactory(DIST_PACKAGE_NAME, CLASS_NAME_SUFFIX);
        @SuppressWarnings("rawtypes")
        final Distribution dist = classFactory.loadInstance(distSubName);
        return dist;
    }

    /**
     * Instantiates the distribution with the name provided following the rules of {@link DefaultFactory}. The given
     * parameters are those stored in MEFP
     */
    public static Distribution instantiateDistribution(final DistributionType type, final Number[] parameters) throws DefaultFactoryException
    {
        final Distribution dist = instantiateDistribution(type);
        dist.setParameters(parameters);
        return dist;
    }

}
