package ohd.hseb.hefs.utils.dist;

import ohd.hseb.hefs.utils.dist.types.NormalDist;
import ohd.hseb.util.data.DataSet;

/**
 * General tools developed for use in the distributions.
 * 
 * @author Hank.Herr
 */
public abstract class HMathTools
{

    /**
     * Constant needed for betacf.
     */
    public final static int MAXIT = 100;

    /**
     * Constant needed for betacf.
     */
    public final static double EPS = 3.0e-7;

    /**
     * Constant needed for betacf.
     */
    public final static double FPMIN = 1.0e-30;

    /**
     * Numerical Recipes in C, page 214. This it the ln-Gamma function implementation.
     */
    public static double gammaln(final double value) throws ArithmeticException
    {
        double x, y, tmp, ser;
        final double cof[] = {76.18009172947146, -86.50532032941677, 24.01409824083091, -1.231739572450155,
            0.1208650973866179e-2, -0.5395139384953e-5};

        int j;

        y = x = value;
        tmp = x + 5.5;
        tmp -= (x + 0.5) * Math.log(tmp);
        ser = 1.000000000190015;
        for(j = 0; j <= 5; j++)
        {
            ser += cof[j] / ++y;
        }
        return -1 * tmp + Math.log(2.5066282746310005 * ser / x);
    }

    /**
     * Numerical Recipes in C, page 216. The beta function.
     */
    public static double beta(final double z, final double w) throws ArithmeticException
    {
        return Math.exp(gammaln(z) + gammaln(w) - gammaln(z + w));
    }

    /**
     * The incomplete beta function, which is also the Beta distribution CDF. Numerical Recipes in C, page 227. Note
     * that the ordering of the two shape parameters a and b does not matter.
     */
    public static double incompleteBeta(final double value, final double a, final double b)
    {
        double bt;

        if((value < 0.0) || (value > 1.0))
        {
            return DataSet.MISSING;
        }

        if((value == 0.0) || (value == 1.0))
        {
            bt = 0.0;
        }
        else
        {
            bt = Math.exp(gammaln(a + b) - gammaln(a) - gammaln(b) + a * Math.log(value) + b * Math.log(1.0 - value));
        }
        if(value < (a + 1.0) / (a + b + 2.0))
        {
            return bt * betacf(value, a, b) / a;
        }
        else
        {
            return 1.0 - bt * betacf(1.0 - value, b, a) / b;
        }
    }

    /**
     * The betacf function, as defined in Numerical Reciped in C, page 227. Used in the computation of
     * {@link #incompleteBeta(double, double, double)}.
     */
    public static double betacf(final double value, final double a, final double b)
    {
        final double x = value;
        int m, m2;
        double aa, c, d, del, h, qab, qam, qap;

        qab = a + b;
        qap = a + 1.0;
        qam = a - 1.0;
        c = 1.0;
        d = 1.0 - qab * x / qap;
        if(Math.abs(d) < FPMIN)
        {
            d = FPMIN;
        }
        d = 1.0 / d;
        h = d;
        for(m = 1; m < MAXIT; m++)
        {
            m2 = 2 * m;
            aa = m * (b - m) * x / ((qam + m2) * (a + m2));
            d = 1.0 + aa * d;
            if(Math.abs(d) < FPMIN)
            {
                d = FPMIN;
            }
            c = 1.0 + aa / c;
            if(Math.abs(c) < FPMIN)
            {
                c = FPMIN;
            }
            d = 1.0 / d;
            h = h * d * c;
            aa = -1 * (a + m) * (qab + m) * x / ((a + m2) * (qap + m2));
            d = 1.0 + aa * d;
            if(Math.abs(d) < FPMIN)
            {
                d = FPMIN;
            }
            c = 1.0 + aa / c;
            if(Math.abs(c) < FPMIN)
            {
                c = FPMIN;
            }
            d = 1.0 / d;
            del = d * c;
            h = h * del;
            if(Math.abs(del - 1.0) < EPS)
            {
                break;
            }
        }
        if(m > MAXIT)
        {
            return DataSet.MISSING;
        }
        return h;
    }

    /**
     * The inverse of the imcomplete beta function. From Abramowitz and Stegun (60's version).
     */
    public static double inverseIncompleteBeta(final double prob, final double a, final double b) throws ArithmeticException
    {
        final NormalDist std = new NormalDist();

        //Get the normal qunatile for p.
        final double yp = std.functionInverseCDF(1.0 - prob);

        //Calculate lambda.
        final double lambda = (yp * yp - 3.0) / 6.0;

        //Calculate h.
        final double h = 2.0 * 1.0 / (1.0 / (2.0 * a - 1.0) + 1.0 / (2.0 * b - 1.0));

        //Calculate w.
        final double w = yp * Math.sqrt(h + lambda) / h - (1.0 / (2.0 * b - 1.0) - 1.0 / (2.0 * a - 1.0))
            * (lambda + 5.0 / 6.0 - 2.0 / (3.0 * h));

        //Return the quantile estimate.
        return a / (a + b * Math.exp(2 * w));
    }

    /**
     * The choose function, as in (6 choose 4), which is 6!/4!2!. To avoid number explosion, the factorials are
     * performed within the algorithm below to ensure the numbers stay small.
     * 
     * @param above The number normally at the top of the choose notation; i.e., the total number of values from which
     *            to 'choose'.
     * @param below The number normally at the bottom of hte choose notation; i.e., the number of values to 'choose'.
     * @return The computed choose result.
     */
    public static double choose(final int above, final int below)
    {
        double result = 1.0;

        //If above is less than below, the result is unknown.
        if(above < below)
        {
            return DataSet.MISSING;
        }

        //If above equals below, the result is 1.
        if(above == below)
        {
            return 1.0;
        }

        int i;

        //This is doing factorials the hard way, in order to make sure the numbers
        //don't get too large.
        for(i = 1; i <= above; i++)
        {
            result = result * i;

            //If i <= below, then the we still have the factorial b! to consider
            //in the denominator.
            if(i <= below)
            {
                result = result / i;
            }

            //If i <= (above - below), then we still have the factorial (a-b)! to consider
            //in the denominator.
            if(i <= (above - below))
            {
                result = result / i;
            }
        }

        //Return the result;
        return result;
    }

    /**
     * @return The sum of the array of doubles provided as arguments.
     */
    public static double sumArray(final double... numbers)
    {
        double sum = 0.0;
        for(final double number: numbers)
        {
            sum += number;
        }
        return sum;
    }

    /**
     * @return The sum of the provided numbers starting from the first index through (and including) the last index.
     */
    public static double sumArray(final double[] numbers, final int firstIncludedIndex, final int lastIncludedIndex)
    {
        double sum = 0.0;
        for(int i = firstIncludedIndex; i <= lastIncludedIndex; i++)
        {
            sum += numbers[i];
        }
        return sum;
    }

    /**
     * Scales the array of numbers between the first provided index and the last index multiplying them by the provided
     * scalar.
     */
    public static void scaleArrayInPlace(final double[] numbers,
                                         final int firstIncludedIndex,
                                         final int lastIncludedIndex,
                                         final double scalar)
    {
        for(int i = firstIncludedIndex; i <= lastIncludedIndex; i++)
        {
            numbers[i] = numbers[i] * scalar;
        }
    }

}
