package ohd.hseb.hefs.utils.dist;

import ohd.hseb.util.data.DataSet;

// For a paper on L-moment theory, see:
//
// http://engineering.tufts.edu/cee/people/vogel/publications/lmoment-replace.pdf
//
// Parameters can be calculated from L-moments an order of magnitude faster than regular moments.

public abstract class LMomentsMath
{
    /**
     * dataToLMoments() calculates the sample L-moment ratios from a DataSet. It first calculates the L-moments, and
     * returns the ratios. The code is based on the FORTRAN routine (in lmoments.f) SAMLMU(X,N,XMOM,NMOM) from the:<br>
     * <br>
     * IBM RESEARCH REPORT RC20525<br>
     * 'FORTRAN ROUTINES FOR USE WITH THE METHOD OF L-MOMENTS, VERSION 3'<br>
     * J. R. M. HOSKING <br>
     * IBM RESEARCH DIVISION <br>
     * VERSION 3.04 JULY 2005 <br>
     * <br>
     * 
     * @param data - The data for which to calculate moments. The data should have FitSampleVariable set.
     * @param num_moments - The number of moments to compute. num_moments should be equal to the numbers of distribution
     *            parameters: 2 for shape and scale (shift assumed to be 0), 3 for shape, scale, and shift
     * @return The lmoment ratios.
     */

    public static double[] dataToLMoments(final DataSet data, final int num_moments)
    {
        final double[] lmoments = new double[num_moments];

        double temp, cnst; // constant
        double xi, xii;
        double term, termp, termn; // (p)lus, (m)inus 
        final double[][] coef = new double[2][num_moments];
        double s, s1, s2;
        double sum1, sum2;
        int i, j;
        int nhalf; // = n / 2, truncated down

        // Sort the data in ascending order in place

        data.sortBy(data.getFitSampleVariable());

        final int n = data.getSampleSize();
        final double dn = data.getSampleSize(); // n as a double

        // Init L-Moments

        for(j = 0; j < num_moments; j++)
        {
            lmoments[j] = 0.0;
        }

        // Calculate L-Moments

        if(num_moments <= 2)
        {
            sum1 = 0.0;
            sum2 = 0.0;
            temp = -dn + 1;

            for(i = 0; i < n; i++)
            {
                sum1 = sum1 + data.getValue(i, data.getFitSampleVariable());
                sum2 = sum2 + data.getValue(i, data.getFitSampleVariable()) * temp;
                temp = temp + 2.0;
            }

            lmoments[0] = sum1 / dn;

            if(num_moments <= 1)
            {
                return lmoments;
            }

            lmoments[1] = sum2 / (dn * (dn - 1.0));

            return lmoments;
        }

        // If we got here, we have > 2 L-Moments to find

        for(j = 3; j <= num_moments; j++)
        {
            temp = 1.0 / ((j - 1.0) * (n - j + 1.0));
            coef[0][j - 1] = (j + j - 3.0) * temp;
            coef[1][j - 1] = ((j - 2.0) * (n + j - 2.0)) * temp;
        }

        temp = -dn - 1.0;
        cnst = 1.0 / (dn - 1.0);
        nhalf = n / 2;

        for(i = 1; i <= nhalf; i++)
        {
            temp = temp + 2.0;
            xi = data.getValue(i - 1, data.getFitSampleVariable());
            xii = data.getValue(n - i, data.getFitSampleVariable());
            termp = xi + xii;
            termn = xi - xii;
            lmoments[0] = lmoments[0] + termp;
            s1 = 1.0;
            s = temp * cnst;
            lmoments[1] = lmoments[1] + s * termn;

            // This loop recursively calculates discrete Legendre polynomials, 
            // via eq.(9) of Neuman and Schonbach (1974, int.j.num.meth.eng.)

            for(j = 3; j <= num_moments; j += 2)
            {
                s2 = s1;
                s1 = s;
                s = coef[0][j - 1] * temp * s1 - coef[1][j - 1] * s2;

                lmoments[j - 1] = lmoments[j - 1] + s * termp;

                if(j == num_moments)
                    break;

                s2 = s1;
                s1 = s;
                s = coef[0][j] * temp * s1 - coef[1][j] * s2;

                lmoments[j] = lmoments[j] + s * termn;
            }
        }

        if(n != nhalf + nhalf) // n was odd, do the middle term
        {
            term = data.getValue(nhalf, data.getFitSampleVariable());
            s = 1.0;
            lmoments[0] = lmoments[0] + term;

            for(j = 3; j <= num_moments; j += 2)
            {
                s = -coef[1][j - 1] * s;
                lmoments[j - 1] = lmoments[j - 1] + s * term;
            }
        }

        //  Replace the L-Moments with the L-Moment ratios

        lmoments[0] = lmoments[0] / dn;

        if(lmoments[1] == 0.0) // can't divide by 0
        {
            for(j = 1; j < num_moments; j++)
            {
                lmoments[j] = 0.0;
            }
        }
        else
        {
            for(j = 2; j < num_moments; j++)
            {
                lmoments[j] = lmoments[j] / lmoments[1];
            }

            lmoments[1] = lmoments[1] / dn;
        }

        return lmoments;
    }
}
