package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.HMathTools;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

/**
 * Implementation of the Binomial distribution as a {@link DiscreteDist}. See
 * http://en.wikipedia.org/wiki/Binomial_distribution.
 * 
 * @author hank.herr
 */
public class BinomialDist extends DiscreteDist
{
    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    public final static int PARAMN = 0; //Array index.

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    public final static int PARAMP = 1; //Array index.

    final static int DEFAULT_N = 10;
    final static double DEFAULT_P = 0.5;

    /////////////////////////////////////////////////////////////////////////
    //Constructors
    /////////////////////////////////////////////////////////////////////////
    public BinomialDist()
    {
        this(DEFAULT_N, DEFAULT_P);
    }

    public BinomialDist(final int n, final double p)
    {
        super(new XMLInteger("domain", 0, 0, n), new XMLInteger("paramN", n, 1, null), new XMLDouble("paramP",
                                                                                                     p,
                                                                                                     0d,
                                                                                                     1d));
    }

    public int getParamN()
    {
        return super.getParameter(PARAMN).intValue();
    }

    public void setParamN(final int n)
    {
        super.setParameter(PARAMN, n);
    }

    public double getParamP()
    {
        return super.getParameter(PARAMP).doubleValue();
    }

    public void setParamP(final double p)
    {
        super.setParameter(PARAMP, p);
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
        super.setDomainLowerBound(value.intValue());
    }

    public double functionCDF(final Integer x)
    {
        double temp = 0.0; //Stores the result.

        //Cast the parameter and get the parameters
        final int n = getParamN();

        //If value is not between 0 and n, inclusive, return 0 or 1.
        if(x < 0)
        {
            return 0.0;
        }
        if(x > n)
        {
            return 1.0;
        }

        //Add up the PDF values leading up to value.
        int i;
        for(i = 0; i <= x; i++)
        {
            temp += functionPDF(i);
        }

        return temp;

    }

    public double functionPDF(final Integer x)
    {
//        double temp; //Stores the result.  This makes sure the returned prob is 0 or 1 only when outside bounds.

        //Cast the parameter and get the parameters
        final int n = getParamN();
        final double p = getParamP();

        //If value is not between 0 and n, inclusive, return 0 or 1.
        if((x < 0) || (x > n))
        {
            return 0.0;
        }

        final double chosen = HMathTools.choose(n, x);
        return chosen * Math.pow(p, x) * Math.pow((1 - p), (n - x));
    }

    //For this discrete variable, this returns the value x (cast into a double) that is the first
    //x such that F(x) >= prob.
    @Override
    public double functionInverseCDF(final double prob)
    {
        //Store the cdfs
        double temp = 0;

        //Check Prob
        if((prob < 0.0) || (prob > 1.0))
        {
            return getMissing();
        }

        //Get the parameters
        final int n = getParamN();

        //Loop through every value for x until I find one with the CDF value greater than prob.
        int i;
        for(i = 0; i <= n; i++)
        {
            temp += functionPDF(i);
            if(temp >= prob)
            {
                return i;
            }
        }

        //If I reached this point, something went wrong.  
        return getMissing();
    }

    ////////////////////////////////////////
    public static void main(final String argv[])
    {
        try
        {
            System.out.println("6 Choose 3 = " + HMathTools.choose(6, 3));

            final BinomialDist bin = new BinomialDist(10, 0.5);
            System.out.println("BinPMF  1 = " + bin.functionPDF(1));
            System.out.println("BinPMF  2 = " + bin.functionPDF(2));
            System.out.println("BinPMF  9 = " + bin.functionPDF(9));
            System.out.println("BinPMF  10 = " + bin.functionPDF(10));
            System.out.println("BinDist  1 = " + bin.functionCDF(1));
            System.out.println("BinDist  2 = " + bin.functionCDF(2));
            System.out.println("BinDist  9 = " + bin.functionCDF(9));
            System.out.println("BinDist  10 = " + bin.functionCDF(10));
            System.out.println("BinInv  0    = " + bin.functionInverseCDF(0.0));
            System.out.println("BinINV  0.50 = " + bin.functionInverseCDF(0.50));
            System.out.println("BinINV  0.75 = " + bin.functionInverseCDF(0.75));
            System.out.println("BinINV  1    = " + bin.functionInverseCDF(1.0));

        }
        catch(final ArithmeticException e)
        {
            System.out.println("ERROR");
        }

    }

}
