package ohd.hseb.hefs.utils.dist.types;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ohd.hseb.hefs.utils.dist.DataFittingDistribution;
import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.Distribution;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.hefs.utils.xml.vars.XMLNumber;
import ohd.hseb.util.data.DataSet;

import com.google.common.collect.Lists;

/**
 * Abstract framework for a continuous distribution, providing an attribute {@link #_domainVariableDefinition} that
 * defines the bounds of the variate modeled, and {@link #_parameters} as a list of parameters, all being
 * {@link XMLNumber} instances. It also provides a default implementation of {@link #fitToData(DataSet, double[])} that
 * assumes the distribution has a shift parameter that must be optimized; for more info, read its javadoc.
 * 
 * @author hank.herr
 */
public abstract class ContinuousDist implements Distribution<Double>, DataFittingDistribution
{

    /**
     * Records the domain variable definition, including lower and upper bounds that define the boundedness of the
     * continuous variable modeled.
     */
    private final XMLDouble _domainVariableDefinition;

    /**
     * List of parameters can be any length; its up to the subclass to make sure it is of the right length through
     * intialization via the constructor.
     */
    private final List<XMLNumber> _parameters = new ArrayList<XMLNumber>();

    /**
     * @param domainVariableDefinition Always first, and must define the bounds of variable modeled, even if unbounded.
     *            If unbounded, just pass in null for the upper and lower bound of the {@link XMLDouble}. The value of
     *            the {@link XMLDouble}, that returned by {@link XMLDouble#get()}, is ignored and can be null.
     * @param parameters A listing of the parameters, each with bounds on the parameter specified via the
     *            {@link XMLNumber} provided, typically {@link XMLDouble}.
     */
    protected ContinuousDist(final XMLDouble domainVariableDefinition, final XMLNumber... parameters)
    {
        _domainVariableDefinition = domainVariableDefinition;
        _parameters.addAll(Lists.newArrayList(parameters));
    }

    /**
     * @return The {@link XMLNumber} used to store the parameter at the provided index.
     */
    protected XMLNumber getParameterStorageVar(final int index)
    {
        return _parameters.get(index);
    }

    /**
     * @return True if the domain has a lower bound, as defined in {@link #_domainVariableDefinition}.
     */
    public boolean domainHasLowerBound()
    {
        return getDomainLowerBound() != null;
    }

    /**
     * @return True if the domain has a upper bound, as defined in {@link #_domainVariableDefinition}.
     */
    public boolean domainHasUpperBound()
    {
        return getDomainUpperBound() != null;
    }

    /**
     * @return Returns the lower bound defined in {@link #_domainVariableDefinition}.
     */
    public Double getDomainLowerBound()
    {
        return _domainVariableDefinition.getLowerBound();
    }

    /**
     * Calls {@link #_domainVariableDefinition}'s {@link XMLDouble#setBounds(Double, Double)} method, reseting the
     * bounds, using the current upper bound but replacing the lower bound.
     * 
     * @param value New lower bound.
     */
    public void setDomainLowerBound(final Double value)
    {
        _domainVariableDefinition.setBounds(value, _domainVariableDefinition.getUpperBound());
    }

    /**
     * @return Returns the upper bound defined in {@link #_domainVariableDefinition}.
     */
    public Double getDomainUpperBound()
    {
        return _domainVariableDefinition.getUpperBound();
    }

    /**
     * Calls {@link #_domainVariableDefinition}'s {@link XMLDouble#setBounds(Double, Double)} method, reseting the
     * bounds, using the current lower bound but replacing the upper bound.
     * 
     * @param value New upper bound.
     */
    public void setDomainUpperBound(final Double value)
    {
        _domainVariableDefinition.setBounds(_domainVariableDefinition.getLowerBound(), value);
    }

    /**
     * @return The number of parameters defined for this distribution.
     */
    public int getParameterCount()
    {
        return _parameters.size();
    }

    @SuppressWarnings("unchecked")
    @Override
    public void setParameter(final int index, final Number value)
    {
        _parameters.get(index).set(value);
    }

    /**
     * Copies the contents of parameters into the internal storage {@link #_parameters}. First, {@link #_parameters} is
     * initialized to all null. Then the values from the provided parameters are copied one at a time. If parameters is
     * longer than {@link #_parameters}, it will only copy as many as can fit. If parameters is shorter, then there will
     * be nulls at the end of {@link #_parameters}.
     * 
     * @param parameters The parameter values to use.
     */
    @Override
    public void setParameters(final Number[] parameters)
    {
        for(int i = 0; i < _parameters.size(); i++)
        {
            if(i < parameters.length)
            {
                setParameter(i, parameters[i]);
            }
            else
            {
                setParameter(i, null);
            }
        }
    }

    /**
     * @return The parameters. If an element in the {@link #_parameters} attribute is null, then the corresponding item
     *         in the list will be null.
     */
    public List<Number> getParameters()
    {
        final List<Number> parms = new ArrayList<Number>();
        for(final XMLNumber num: _parameters)
        {
            if(num != null)
            {
                parms.add((Number)num.get());
            }
            else
            {
                parms.add((Number)null);
            }
        }
        return parms;
    }

    public Number getParameter(final int index)
    {
        if(_parameters.get(index) == null)
        {
            return null;
        }
        return (Number)_parameters.get(index).get();
    }

    /**
     * By default returns NaN.
     */
    @Override
    public double getMissing()
    {
        return Double.NaN;
    }

    /**
     * By default, checks for NaN.
     */
    @Override
    public boolean isMissing(final double value)
    {
        return Double.isNaN(value);
    }

    @Override
    public void fitToData(final DataSet data) throws DataFittingDistributionException
    {
        fitToData(data, null);
    }

    /**
     * Default version assumes that the distribution has a shift parameter which needs to be optimized. It calls
     * {@link #estimateParameters(DataSet, double)} providing the fixed shift as the second parameter. If you want to
     * use the default version, the distribution **MUST** override {@link #estimateParameters(DataSet, double)}. <br>
     * <br>
     * If null is passed in for fitParms, then it calls the estimate method assuming a 0 fixed shift.
     * 
     * @param data The data to which to fit. The data must have its fit sample variable attribute specified (see
     *            {@link DataSet#setFitSampleVariable()}). If its fit cdf variable (see
     *            {@link DataSet#setFitCDFVariable()}) is specified, then that variable must store the empirical
     *            probabilities. If the fit cdf variable is negative (not specified), then the CDF will be computed
     *            here. In that case, the DataSet will have a variable added to store the CDF and will be sorted by the
     *            sample variable! The data is NOT copied first!).
     * @param fitParms In the default version, fitParms must be of null (to assume a 0 shift) or of length 1. If length
     *            1, then that number is a lower bound on the shift parameter to find. The upper bound is the smallest
     *            value in the data set.
     * @return False for any of these reasons: fitParms is neither null or 1-length; fit sample variable is invalid; if
     *         a shift is provided in fitParms and it is larger than the smallest value in the data sample; if
     *         {@link #estimateParameters(DataSet, double)} returns false for any reason.
     */
//    @Override
//    public void fitToData(final DataSet data, final double[] fitParms) throws DataFittingDistributionException
//    {
//        //First, if fitparms is null, then I am going to assume that you are fitting a two parameter
//        //distribution.  So, I just estimateRegression with a 0 shift.
//        if(fitParms == null)
//        {
//            estimateParameters(data, 0.0D);
//            return;
//        }
//
//        //Otherwise, make sure fitparms is of length 1.
//        if(fitParms.length != 1)
//        {
//            throw new 
//        }
//
//        //Get the smallest shift from fitparms[0].
//        final double smallestshift = fitParms[0];
//
//        data.resetPtr();
//        double min, max;
//        double lower = 0.0;
//        double higher = 0.0;
//        double lowermse = 0.0;
//        double highermse = 0.0;
//
//        //Some QC checks...
//
//        //Check the sample variable and empirical variable.
//        int samplevar, empvar;
//        samplevar = data.getFitSampleVariable();
//        empvar = data.getFitCDFVariable();
//        if((samplevar < 0) || (samplevar == empvar))
//        {
//            return false;
//        }
//
//        //Estimate the CDF if needed.
//        if(empvar < 0)
//        {
//            data.addNewVariable();
//            empvar = data.getNumberOfVariables() - 1;
//            data.createCDFUsingWeibullPlottingPosition(samplevar, empvar);
//        }
//
//        //Set the min and max on the shift parameter to be the smallestshift and the smallest data value.
//        //Check the values.
//        min = smallestshift;
//        max = data.getSmallest(samplevar);
//        if(min > max)
//        {
//            return false;
//        }
//
//        //At this point, the_data is set up for a search.
//        //Find the best shift for the data set.  This uses Golden Section search to find the smallest MSE value.
//        //End the search when max and min are less than ACCURACY (see Distribution.java) units apart.
//        while((max - min) > FIT_ACCURACY)
//        {
//            data.resetPtr();
//            //Determine the lower guess.  Do not change the coefficient value of 0.618.
//            //This allows me to make only one new fit per iteration.
//            if(lower == 0)
//            {
//                lower = -0.618 * (max - min) + max;
//                if(!estimateParameters(data, lower))
//                {
//                    return false;
//                }
//                lowermse = data.meanSquaredError(samplevar, empvar, this);
//            }
//
//            //Determine the higher guess, and don't change 0.6120.
//            if(higher == 0)
//            {
//                higher = 0.618 * (max - min) + min;
//                if(!estimateParameters(data, higher))
//                {
//                    return false;
//                }
//                highermse = data.meanSquaredError(samplevar, empvar, this);
//            }
//
//            //If the error of the lower guess is less than that for the higher guess,
//            //then the minimum is not between higher and max.  Reset max as higher, 
//            //and adjust the other numbers appropriately.
//            if(lowermse <= highermse)
//            {
//                max = higher;
//                higher = lower;
//                lower = 0;
//                highermse = lowermse;
//            }
//
//            //Otherwise, the minimum must not be between min and lower.  Adjust appropriately.
//            else
//            {
//                min = lower;
//                lower = higher;
//                higher = 0;
//                lowermse = highermse;
//            }
//        }
//
//        return true;
//    }

//    /**
//     * MUST be overridden if the default {@link #fitToData(DataSet, double[])} is used. That method assumes the
//     * distribution has a shift parameter that must be optimized. It will call this method to estimate the parameters
//     * for each shift it tests.
//     * 
//     * @param data Data to which to estimate parameters. The data must have specified fit sample and CDF variables.
//     * @param fixedShift The fixed shift to use.
//     * @return True if successful, false otherwise.
//     */
//    //TODO Change to use exception throwing instead
//    protected boolean estimateParameters(final DataSet data, final double fixedShift)
//    {
//        throw new IllegalStateException("This method has not been overridden for the distribution!");
//    }

    /**
     * @return By default, returns 1 - {@link #functionCDF(Double)}.
     */
    @Override
    public double functionExceedance(final Double value)
    {
        final double temp = functionCDF(value);
        if(isMissing(temp))
        {
            return getMissing();
        }
        return 1 - temp;
    }

    /**
     * @param nf {@link NumberFormat} to apply to each number when outputting.
     */
    public String toString(final NumberFormat nf)
    {
        final String[] parameterValues = new String[this._parameters.size()];
        for(int i = 0; i < parameterValues.length; i++)
        {
            parameterValues[i] = nf.format(_parameters.get(i).get());
        }
        return this.getClass().getSimpleName() + ": parameters = " + Arrays.toString(parameterValues);
    }

    @Override
    public String toString()
    {
        final double[] parameterValues = new double[this._parameters.size()];
        for(int i = 0; i < parameterValues.length; i++)
        {
            parameterValues[i] = ((Number)_parameters.get(i).get()).doubleValue();
        }
        return this.getClass().getSimpleName() + ": parameters = " + Arrays.toString(parameterValues);
    }
}
