package ohd.hseb.hefs.utils.dist.types;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ohd.hseb.hefs.utils.dist.Distribution;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLNumber;

import com.google.common.collect.Lists;

/**
 * Abstract framework for a discrete distribution, providing an attribute {@link #_domainVariableDefinition} that
 * defines the bounds of the variate modeled, and {@link #_parameters} as a list of parameters, all being
 * {@link XMLNumber} instances.
 * 
 * @author hank.herr
 */
public abstract class DiscreteDist implements Distribution<Integer>
{

    /**
     * Defines the domain of the distribution. Specifically, the bounds provided with the {@link XMLInteger} indicate
     * whether it is lower or upper bounded and what those bounds are.
     */
    private final XMLInteger _domainVariableDefinition;

    /**
     * Parameter storage attribute records each parameter as an {@link XMLNumber} within an {@link ArrayList}.
     */
    private final List<XMLNumber> _parameters = new ArrayList<XMLNumber>();

    /**
     * @param domainVariableDefinition Variable defining the properties of the domain to be stored in
     *            {@link #_domainVariableDefinition}.
     * @param parameters The parameters of the distribution to be stored in {@link #_parameters}.
     */
    protected DiscreteDist(final XMLInteger domainVariableDefinition, final XMLNumber... parameters)
    {
        _domainVariableDefinition = domainVariableDefinition;
        _parameters.addAll(Lists.newArrayList(parameters));
    }

    /**
     * @return True if the domain has a lower bound, as defined in {@link #_domainVariableDefinition}.
     */
    public boolean domainHasLowerBound()
    {
        return getDomainLowerBound() != null;
    }

    /**
     * @return True if the domain has a upper bound, as defined in {@link #_domainVariableDefinition}.
     */
    public boolean domainHasUpperBound()
    {
        return getDomainUpperBound() != null;
    }

    /**
     * @return Returns the lower bound defined in {@link #_domainVariableDefinition}.
     */
    public Integer getDomainLowerBound()
    {
        return _domainVariableDefinition.getLowerBound();
    }

    /**
     * Calls {@link #_domainVariableDefinition}'s {@link XMLInteger#setBounds(Double, Double)} method, reseting the
     * bounds, using the current upper bound but replacing the lower bound.
     * 
     * @param value New lower bound.
     */
    public void setDomainLowerBound(final int lb)
    {
        _domainVariableDefinition.setBounds(lb, _domainVariableDefinition.getUpperBound());
    }

    /**
     * @return Returns the upper bound defined in {@link #_domainVariableDefinition}.
     */
    public Integer getDomainUpperBound()
    {
        return _domainVariableDefinition.getUpperBound();
    }

    /**
     * Calls {@link #_domainVariableDefinition}'s {@link XMLInteger#setBounds(Double, Double)} method, reseting the
     * bounds, using the current lower bound but replacing the upper bound.
     * 
     * @param value New lower bound.
     */
    public void setDomainUpperBound(final int ub)
    {
        _domainVariableDefinition.setBounds(_domainVariableDefinition.getLowerBound(), ub);
    }

    /**
     * @return The parameters. If an element in the {@link #_parameters} attribute is null, then the corresponding item
     *         in the list will be null.
     */
    public List<Number> getParameters()
    {
        final List<Number> parms = new ArrayList<Number>();
        for(final XMLNumber num: _parameters)
        {
            if(num != null)
            {
                parms.add((Number)num.get());
            }
            else
            {
                parms.add((Number)null);
            }
        }
        return parms;
    }

    @SuppressWarnings("unchecked")
    @Override
    public void setParameter(final int index, final Number value)
    {
        _parameters.get(index).set(value);
    }

    /**
     * Copies the contents of parameters into the internal storage {@link #_parameters}. First, {@link #_parameters} is
     * initialized to all null. Then the values from the provided parameters are copied one at a time. If parameters is
     * longer than {@link #_parameters}, it will only copy as many as can fit. If parameters is shorter, then there will
     * be nulls at the end of {@link #_parameters}.
     * 
     * @param parameters The parameter values to use.
     */
    @Override
    public void setParameters(final Number[] parameters)
    {
        for(int i = 0; i < _parameters.size(); i++)
        {
            if(i < parameters.length)
            {
                setParameter(i, parameters[i]);
            }
            else
            {
                setParameter(i, null);
            }
        }
    }

    @Override
    public Number getParameter(final int index)
    {
        return (Number)_parameters.get(index).get();
    }

    @Override
    public boolean isMissing(final double value)
    {
        return Double.isNaN(value);
    }

    @Override
    public double getMissing()
    {
        return Double.NaN;
    }

    @Override
    public double functionExceedance(final Integer value)
    {
        final double temp = functionCDF(value);
        if(isMissing(temp))
        {
            return getMissing();
        }
        return 1 - temp;
    }

    @Override
    public String toString()
    {
        final double[] parameterValues = new double[this._parameters.size()];
        for(int i = 0; i < parameterValues.length; i++)
        {
            parameterValues[i] = ((Number)_parameters.get(i).get()).doubleValue();
        }
        return this.getClass().getSimpleName() + ": parameters = " + Arrays.toString(parameterValues);
    }
}
