package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.DistributionTools;
import ohd.hseb.hefs.utils.dist.ShiftOptimizationFittingDistribution;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

/**
 * An implementation of {@link ContinuousDist} for a log-logistic distribution, which is a Logistic distribution applied
 * to a log-transformed variable. This distribution includes a shape, scale, and shift parameter. By default, however,
 * the shift parameter is not fitted; that is, it is not set via the {@link #fitToData(DataSet)} method. Rather, it must
 * be set before fitting. To turn on shift fitting, call {@link #setFitShift(boolean)} with true.
 * 
 * @author hank.herr
 */
public class LoglogisticDist extends ContinuousDist implements ShiftOptimizationFittingDistribution
{
    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SCALE = 0;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHAPE = 1;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHIFT = 2;

    private static final double DEFAULT_SCALE = 1.0D;
    private static final double DEFAULT_SHAPE = 2.0D;
    private static final double DEFAULT_SHIFT = 0.0D;

    private boolean _fitShift = false;

    /**
     * Constructs the Gamma distribution using the defaults provided by the constants: {@link #DEFAULT_SCALE},
     * {@link #DEFAULT_SHAPE}, and {@link #DEFAULT_SHIFT}.
     */
    public LoglogisticDist()
    {
        this(DEFAULT_SCALE, DEFAULT_SHAPE, DEFAULT_SHIFT);
    }

    /**
     * Constructs the distribution given the specified parameters.
     * 
     * @param scale Scale parameter.
     * @param shape Shape parameter.
     * @param shift Shift parameter.
     */
    public LoglogisticDist(final double scale, final double shape, final double shift)
    {
        super(new XMLDouble("domain", shift, shift, null),
              new XMLDouble("scale", scale, 0.0D, null),
              new XMLDouble("shape", shape, 0.0D, null),
              new XMLDouble("shift", shift, null, null));
    }

    public void setFitShift(final boolean b)
    {
        _fitShift = b;
    }

    public void setScale(final double scale)
    {
        setParameter(SCALE, scale);
    }

    public double getScale()
    {
        return getParameter(SCALE).doubleValue();
    }

    public void setShape(final double shape)
    {
        setParameter(SHAPE, shape);
    }

    public double getShape()
    {
        return getParameter(SHAPE).doubleValue();
    }

    /**
     * Ensures {@link #setDomainLowerBound(Double)} is called as well.
     */
    public void setShift(final double shift)
    {
        setParameter(SHIFT, shift);
        setDomainLowerBound(shift);
    }

    public double getShift()
    {
        return getParameter(SHIFT).doubleValue();
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
    }

    @Override
    public double functionCDF(final Double value)
    {
        final Double low = getShift();
        double temp;

        if((low == null) || isMissing(low))
        {
            throw new IllegalArgumentException("The lower bound is not defined, which is invalid.");
        }

        //Check the lower bound constraint.
        if(value < low)
            return 0;

        //Try to perform the computation
        try
        {
            temp = 1 / (1 + Math.pow((value - getShift()) / getScale(), -1 * getShape()));
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        //Make sure the number is within reasonable bounds.
        if(temp == 1)
            temp = 0.99999;
        if(temp == 0)
            temp = 0.00001;

        return temp;
    }

    @Override
    public double functionPDF(final Double value)
    {
        double a, b, g, temp;

        a = getScale();
        b = getShape();
        g = getShift();

        // Since the PDF is the derivative of the CDF, you can check this at Wolfram Alpha using:
        // derivative 1 / (1 + pow((x - g) / a, -1 * b))

        try
        {
            temp = ((b / a) * Math.pow((value - g) / a, b - 1) / Math.pow(1 + Math.pow((value - g) / a, b), 2));
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        return temp;

    }

    @Override
    public double functionInverseCDF(final double prob)
    {
        double temp;

        if((prob >= 1) || (prob <= 0))
            return getMissing();

        try
        {
            temp = getScale() * Math.pow(prob / (1 - prob), 1 / getShape()) + getShift();
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        return temp;
    }

    /**
     * Estimate the parameters of the log-Logistic given data, a location parameter, using Regression.
     */
    private void estimateRegression(final DataSet data) throws DataFittingDistributionException
    {
        //First, do some basic QC checks...

        //Make sure the samplevar and empvar are set.
        int samplevar = data.getFitSampleVariable();
        int empvar = data.getFitCDFVariable();
        if((samplevar < 0) || (empvar < 0) || (samplevar == empvar))
        {
            throw new IllegalArgumentException("Sample variable number, " + samplevar
                + ", or empirical variable number, " + empvar + " is invalid.");
        }

        //The shift cannot be larger than any data value for the fit sample variable.
        if(getShift() > data.getSmallest(samplevar))
        {
            throw new DataFittingDistributionException("Smallest data value provided, " + data.getSmallest(samplevar)
                + ", is smaller than this distribution's lower bound, " + getShift());
        }

        //Make sure I have no missing values in any sample or empirical data
        if((data.isAnyVariableDataMissing(samplevar)) || (data.isAnyVariableDataMissing(empvar)))
        {
            throw new DataFittingDistributionException("Some provided data is  missing.");
        }

        //Make sure all the empirical values are within (0, 1) -- NOT INCLUDING 0 or 1!
        if((data.getSmallest(empvar) <= 0) || (data.getLargest(empvar) >= 1))
        {
            throw new DataFittingDistributionException("The range of computed empirical probabilities, smallest = "
                + data.getSmallest(empvar) + ", largest = " + data.getLargest(empvar) + ", is outside (0, 1).");
        }

        //The local variables.
        DataSet samples;
        double coeff, con;

        //Goto beginning of data set.
        data.resetPtr();

        //Create the regression data set.
        samples = createRegressDataSet(data);
        samplevar = samples.getFitSampleVariable();
        empvar = samples.getFitCDFVariable();

        //Calculate coeff and con, the regression coefficient and constant. 
        //Samples has two variables.  The first is SAMPLE_VALUES and the second is SAMPLE_EMPIRICALS.
        coeff = (samples.sumProduct(samplevar, empvar) - samples.getSampleSize() * samples.mean(samplevar)
            * samples.mean(empvar))
            / (samples.sumSquares(samplevar) - samples.getSampleSize() * Math.pow(samples.mean(samplevar), 2));
        con = samples.mean(empvar) - coeff * samples.mean(samplevar);

        //These are the parameter estimates.  
        setScale(Math.exp(con));
        setShape(1 / coeff);

        data.resetPtr();
    }

    /**
     * Create the data set to be used in the (reversed) Regression log-Weibull fit. CDF must be created before this can
     * work.
     * 
     * @param data Data to fit to
     * @return DataSet to be used in regression.
     */
    private DataSet createRegressDataSet(final DataSet data)
    {
        final DataSet newdata = new DataSet(data.getMaximumSampleSize(), 2);
        boolean notatend = true;
        double[] asample = new double[2];

        int samplevar, empvar;
        samplevar = data.getFitSampleVariable();
        empvar = data.getFitCDFVariable();

        data.resetPtr();

        while(notatend)
        {
            //Set the data point in the new data set according to the equations below.
            //Since the regression is REVERSED, the values and empiricals variables are
            //swapped.
            asample = new double[2];
            if(data.getCurrentValue(samplevar) != -999)
            {
                asample[0] = Math.log(1 / (1 / data.getCurrentValue(empvar) - 1));
                asample[1] = Math.log(data.getCurrentValue(samplevar) - getShift());
                newdata.addSample(asample);
            }

            notatend = data.next();
        }

        //Return the new data set.
        newdata.setFitSampleVariable(0);
        newdata.setFitCDFVariable(1);
        return newdata;
    }

    @Override
    public void fitToData(final DataSet data, double[] fitParms) throws DataFittingDistributionException
    {
        if(!_fitShift)
        {
            estimateRegression(data);
        }
        else
        {
            if(fitParms == null)
            {
                fitParms = new double[]{0.0D};
            }
            DistributionTools.optimizeShiftFitForBoundedBelowDistribution(this, data, fitParms);
        }
    }

    @Override
    public void estimateParameters(final DataSet data, final double shift) throws DataFittingDistributionException
    {
        final LoglogisticDist fitDist = new LoglogisticDist();
        fitDist.setFitShift(false); //Force a two-param fit
        fitDist.setShift(shift);
        fitDist.fitToData(data);

        //COpy the results
        setScale(fitDist.getScale());
        setShape(fitDist.getShape());
        setShift(shift);
    }

//
//    //XXX ALL OF THE BELOW METHODS ARE CURRENTLY NOT USED FOR THIS DISTRIBUTION.  THEY ARE LEFT IN SOLELY TO HAVE A COMPLETE RECORD!
//
//    /**
//     * Call this method to use a generic optimization tool for fitting the data: {@link MultiDirectional}.
//     * 
//     * @param data Standard fitting data set, with both {@link DataSet#getFitSampleVariable()} and
//     *            {@link DataSet#getFitCDFVariable()} returning values.
//     * @param fitparms If not null, an array containing a single double value specifying the lower bound on the shift
//     *            parameter, typically zero. The upper bound will be the smallest sample value in the provided data. If
//     *            null, the shift is fixed to 0.
//     * @throws Exception
//     */
//    public void optimizeFitToData(final DataSet data, final double[] fitparms) throws Exception
//    {
//        this._fittedDataSet = data;
//
//        final boolean shiftFitted = (fitparms != null);
//        double[] startingPoint;
//        if(!shiftFitted)
//        {
//            _lowerBoundOnShift = null;
//            _upperBoundOnShift = null;
//            setShift(0.0D);
//            startingPoint = new double[]{1.0, 1.0};
//        }
//        else
//        {
//            _lowerBoundOnShift = fitparms[0];
//            _upperBoundOnShift = data.getSmallest(data.getFitSampleVariable());
//            startingPoint = new double[]{1.0, 1.0, (_lowerBoundOnShift + _upperBoundOnShift) / 2.0D};
//        }
//
//        final MultiDirectional optimizer = new MultiDirectional();
//        optimizer.setConvergenceChecker(this);
//        final RealPointValuePair results = optimizer.optimize(this, GoalType.MINIMIZE, startingPoint);
//
//        setScale(results.getPoint()[0]);
//        setShape(results.getPoint()[1]);
//        if(shiftFitted)
//        {
//            setShift(results.getPoint()[2]);
//        }
//    }
//
//    @Override
//    public double value(final double[] currentParametersForOptimization) throws FunctionEvaluationException,
//                                                                        IllegalArgumentException
//    {
//        if((currentParametersForOptimization[0] < 0) || (currentParametersForOptimization[1] < 0))
//        {
//            return Double.MAX_VALUE;
//        }
//
//        //Check the shift, only if the lower bound is non-null.
//        LoglogisticDist dist = null;
//        if(_lowerBoundOnShift != null)
//        {
//            if((currentParametersForOptimization[2] < this._lowerBoundOnShift)
//                || (currentParametersForOptimization[2] > this._upperBoundOnShift))
//            {
//                return Double.MAX_VALUE;
//            }
//            dist = new LoglogisticDist(currentParametersForOptimization[0],
//                                       currentParametersForOptimization[1],
//                                       currentParametersForOptimization[2]);
//        }
//        else
//        {
//            dist = new LoglogisticDist(currentParametersForOptimization[0], currentParametersForOptimization[1], 0.0D);
//        }
//
//        //Return the maximum error of the distribution fit.
//        return _fittedDataSet.maximumError(_fittedDataSet.getFitSampleVariable(),
//                                           _fittedDataSet.getFitCDFVariable(),
//                                           dist);
//    }
//
//    @Override
//    public boolean converged(final int arg0, final RealPointValuePair previous, final RealPointValuePair current)
//    {
//        if((Math.abs(previous.getPoint()[0] - current.getPoint()[0]) < 0.01)
//            && (Math.abs(previous.getPoint()[1] - current.getPoint()[1]) < 0.01))
//        {
//            return true;
//        }
//        return false;
//    }
}
