package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.DistributionTools;
import ohd.hseb.hefs.utils.dist.ShiftOptimizationFittingDistribution;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

/**
 * An implementation of {@link ContinuousDist} for a log-normal distribution, which is a Normal distribution applied to
 * a log-transformed variable. This distribution includes a mean, standard deviation, and shift parameter. By default,
 * however, the shift parameter is not fitted; that is, it is not set via the {@link #fitToData(DataSet)} method.
 * Rather, it must be set before fitting. To turn on shift fitting, call {@link #setFitShift(boolean)} with true.
 * 
 * @author hank
 */
public class LognormalDist extends ContinuousDist implements ShiftOptimizationFittingDistribution
{
    final String CLASSNAME = "Normal";

    public final static int MEAN = 0;
    public final static int STDEV = 1;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHIFT = 2;

    protected static final double DEFAULT_MEAN = 0.0D;
    protected static final double DEFAULT_STDEV = 1.0D;
    private static final double DEFAULT_SHIFT = 0.0D;

    private boolean _fitShift = false;

    /**
     * Wrapped {@link NormalDist} used in the computations.
     */
    private final NormalDist _wrappedNormal = new NormalDist();

    /**
     * Default is a standard normal.
     */
    public LognormalDist()
    {
        super(new XMLDouble("domain", DEFAULT_SHIFT, DEFAULT_SHIFT, null),
              new XMLDouble("mean", DEFAULT_MEAN, null, null),
              new XMLDouble("stdev", DEFAULT_STDEV, 0.0, null),
              new XMLDouble("shift", DEFAULT_SHIFT, null, null));
        updateWrappedNormal();
    }

    public LognormalDist(final double mean, final double stdev)
    {
        this();
        setMean(mean);
        setStandardDeviation(stdev);
        updateWrappedNormal();
    }

    public LognormalDist(final double mean, final double stdev, final double shift)
    {
        this();
        setMean(mean);
        setStandardDeviation(stdev);
        setShift(shift);
        updateWrappedNormal();
    }

    public void setFitShift(final boolean b)
    {
        _fitShift = b;
    }

    private void updateWrappedNormal()
    {
        _wrappedNormal.setMean(getMean());
        _wrappedNormal.setStandardDeviation(getStandardDeviation());
    }

    public void setMean(final double mean)
    {
        setParameter(MEAN, mean);
        _wrappedNormal.setMean(mean);
    }

    public double getMean()
    {
        return getParameter(MEAN).doubleValue();
    }

    public void setStandardDeviation(final double stdev)
    {
        setParameter(STDEV, stdev);
        _wrappedNormal.setStandardDeviation(stdev);
    }

    public double getStandardDeviation()
    {
        return getParameter(STDEV).doubleValue();
    }

    /**
     * Ensures {@link #setDomainLowerBound(Double)} is called as well.
     */
    public void setShift(final double shift)
    {
        setParameter(SHIFT, shift);
        setDomainLowerBound(shift);
    }

    public double getShift()
    {
        return getParameter(SHIFT).doubleValue();
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
    }

    //Overrides Distribution class
    @Override
    public double functionCDF(final Double value)
    {
        return _wrappedNormal.functionCDF(Math.log(value - getShift()));
    }

    @Override
    public double functionPDF(final Double value)
    {
        return _wrappedNormal.functionPDF(Math.log(value - getShift()));
    }

    @Override
    public double functionInverseCDF(final double prob)
    {
        //Uses Apache package.  The normal distribution I normally use is malfunctioning within its 
        //functionInverseCDF.
        final double tmp = _wrappedNormal.functionInverseCDF(prob);
        if(isMissing(tmp))
        {
            return getMissing();
        }
        return Math.exp(tmp) + getShift();
    }

    /**
     * This creates a copy of the provided data in order to construct a log-transformed version of the data. This
     * estimates the two-parameter version.
     * 
     * @param data Be sure to set the fit sample variable via {@link DataSet#setFitSampleVariable(int)}.
     * @return True if successful, false if not.
     */
    private void estimateMaximumLikelihood(final DataSet data) throws DataFittingDistributionException
    {
        //Make sure the samplevar and empvar are set.
        final int samplevar = data.getFitSampleVariable();
        if(samplevar < 0)
        {
            throw new IllegalArgumentException("Sample variable number, " + samplevar + ", is invalid.");
        }

        //The shift cannot be larger than any data value for the fit sample variable.
        if(getShift() > data.getSmallest(samplevar))
        {
            throw new DataFittingDistributionException("Smallest data value provided, " + data.getSmallest(samplevar)
                + ", is smaller than this distribution's lower bound, " + getShift());
        }

        //Make sure I have no missing values in any sample or empirical data
        if(data.isAnyVariableDataMissing(samplevar))
        {
            throw new DataFittingDistributionException("Some provided data is  missing.");
        }

        final DataSet logData = new DataSet(data);
        logData.applyShiftTransform(data.getFitSampleVariable(), getShift()); //Shift the data.
        logData.applyLogTransform(data.getFitSampleVariable()); //Log transform it.

        setMean(logData.mean(logData.getFitSampleVariable()));
        setStandardDeviation(Math.sqrt(logData.sampleVariance(logData.getFitSampleVariable())));
    }

    /**
     * Overridden to force {@link #estimateMaximumLikelihood(DataSet)} to be called. There is no shift parameter to
     * optimize for... at least, not yet.
     */
    @Override
    public void fitToData(final DataSet data, double[] fitParms) throws DataFittingDistributionException
    {
        if(!_fitShift)
        {
            estimateMaximumLikelihood(data);
        }
        else
        {
            if(fitParms == null)
            {
                fitParms = new double[]{0.0D};
            }
            DistributionTools.optimizeShiftFitForBoundedBelowDistribution(this, data, fitParms);
        }
    }

    @Override
    public void estimateParameters(final DataSet data, final double shift) throws DataFittingDistributionException
    {
        final LognormalDist fitDist = new LognormalDist();
        fitDist.setFitShift(false);//Force a two-parameter fit with a fixed shift
        fitDist.setShift(shift);
        fitDist.fitToData(data);

        //Copy the results
        setMean(fitDist.getMean());
        setStandardDeviation(fitDist.getStandardDeviation());
        setShift(shift);
    }

}
