package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.DistributionTools;
import ohd.hseb.hefs.utils.dist.ShiftOptimizationFittingDistribution;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

/**
 * An implementation of {@link ContinuousDist} for a log-Weibull distribution, which is a Weibull distribution applied
 * to a log-transformed variable shifted by +1 (see {@link #functionCDF(Double)}, for example). This distribution
 * includes a shape, scale, and shift parameter. By default, however, the shift parameter is not fitted; that is, it is
 * not set via the {@link #fitToData(DataSet)} method. Rather, it must be set before fitting. To turn on shift fitting,
 * call {@link #setFitShift(boolean)} with true.
 * 
 * @author Hank.Herr
 */
public class LogweibullDist extends ContinuousDist implements ShiftOptimizationFittingDistribution
{
    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SCALE = 0;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHAPE = 1;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHIFT = 2;

    private static final double DEFAULT_SCALE = 1.0D;
    private static final double DEFAULT_SHAPE = 2.0D;
    private static final double DEFAULT_SHIFT = 0.0D;

    private boolean _fitShift = false;

    /**
     * Constructs the Gamma distribution using the defaults provided by the constants: {@link #DEFAULT_SCALE},
     * {@link #DEFAULT_SHAPE}, and {@link #DEFAULT_SHIFT}.
     */
    public LogweibullDist()
    {
        this(DEFAULT_SCALE, DEFAULT_SHAPE, DEFAULT_SHIFT);
    }

    /**
     * Constructs the distribution given the specified parameters.
     * 
     * @param scale Scale parameter.
     * @param shape Shape parameter.
     * @param shift Shift parameter.
     */
    public LogweibullDist(final double scale, final double shape, final double shift)
    {
        super(new XMLDouble("domain", shift, shift, null),
              new XMLDouble("scale", scale, 0.0D, null),
              new XMLDouble("shape", shape, 0.0D, null),
              new XMLDouble("shift", shift, null, null));
    }

    public void setFitShift(final boolean b)
    {
        _fitShift = b;
    }

    public void setScale(final double scale)
    {
        setParameter(SCALE, scale);
    }

    public double getScale()
    {
        return getParameter(SCALE).doubleValue();
    }

    public void setShape(final double shape)
    {
        setParameter(SHAPE, shape);
    }

    public double getShape()
    {
        return getParameter(SHAPE).doubleValue();
    }

    /**
     * Ensures {@link #setDomainLowerBound(Double)} is called as well.
     */
    public void setShift(final double shift)
    {
        setParameter(SHIFT, shift);
        setDomainLowerBound(shift);
    }

    public double getShift()
    {
        return getParameter(SHIFT).doubleValue();
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
    }

    @Override
    public double functionCDF(final Double value)
    {
        final Double low = getShift();
        double temp;

        if((low == null) || isMissing(low))
        {
            throw new IllegalArgumentException("The lower bound is not defined, which is invalid.");
        }

        //Check the lower bound constraint.
        if(value < low)
        {
            return 0.0D;
        }

        //Try to perform the computation
        try
        {
            temp = 1 - Math.exp(-1 * Math.pow(Math.log(value + 1 - getShift()) / getScale(), getShape()));
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        return DistributionTools.returnTrimmedProbability(temp);
    }

    @Override
    public double functionPDF(final Double value)
    {
        double a, b, g, temp;

        a = getScale();
        b = getShape();
        g = getShift();

        // Since the PDF is the derivative of the CDF, you can check this at Wolfram Alpha using:
        // derivative 1 - exp(-1 * pow(log(x + 1 - g) / a, b))

        try
        {
            temp = ((b / a) * (1 / (value + 1 - g)) * Math.pow(Math.log(value + 1 - g) / a, b - 1) * Math.exp(-1
                * Math.pow(Math.log(value + 1 - g) / a, b)));
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        return temp;

    }

    @Override
    public double functionInverseCDF(final double prob)
    {
        double temp;
        if((prob >= 1) || (prob <= 0))
        {
            return getMissing();
        }
        try
        {
            temp = Math.exp(getScale() * Math.pow(-1 * Math.log(1 - prob), 1 / getShape())) - 1 + getShift();
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        return temp;

    }

    /**
     * @param data Be sure to call {@link DataSet#setFitSampleVariable(int)} and {@link DataSet#setFitCDFVariable(int)}
     *            prior to calling this routine.
     * @return True if successful, or false if a problem occurs.
     */
    private void estimateRegression(final DataSet data) throws DataFittingDistributionException
    {
        //First, do some basic QC checks...

        //Make sure the samplevar and empvar are set.
        int samplevar = data.getFitSampleVariable();
        int empvar = data.getFitCDFVariable();
        if((samplevar < 0) || (empvar < 0) || (samplevar == empvar))
        {
            throw new IllegalArgumentException("Sample variable number, " + samplevar
                + ", or empirical variable number, " + empvar + " is invalid.");
        }

        //The shift cannot be larger than any data value for the fit sample variable.
        if(getShift() > data.getSmallest(samplevar))
        {
            throw new DataFittingDistributionException("Smallest data value provided, " + data.getSmallest(samplevar)
                + ", is smaller than this distribution's lower bound, " + getShift());
        }

        //Make sure I have no missing values in any sample or empirical data
        if((data.isAnyVariableDataMissing(samplevar)) || (data.isAnyVariableDataMissing(empvar)))
        {
            throw new DataFittingDistributionException("Some provided data is  missing.");
        }

        //Make sure all the empirical values are within (0, 1) -- NOT INCLUDING 0 or 1!
        if((data.getSmallest(empvar) <= 0) || (data.getLargest(empvar) >= 1))
        {
            throw new DataFittingDistributionException("The range of computed empirical probabilities, smallest = "
                + data.getSmallest(empvar) + ", largest = " + data.getLargest(empvar) + ", is outside (0, 1).");
        }

        //The local variables.
        DataSet samples;
        double coeff, con;

        //Reset to the beginning of the data set. 
        data.resetPtr();

        //Create the regression data set.
        samples = createRegressDataSet(data, getShift());
        samplevar = samples.getFitSampleVariable();
        empvar = samples.getFitCDFVariable();

        //Calculate coeff and con, the regression coefficient and constant
        coeff = (samples.sumProduct(samplevar, empvar) - samples.getSampleSize() * samples.mean(samplevar)
            * samples.mean(empvar))
            / (samples.sumSquares(samplevar) - samples.getSampleSize() * Math.pow(samples.mean(samplevar), 2));
        con = samples.mean(empvar) - coeff * samples.mean(samplevar);

        //These are the parameter estimates.  
        setScale(Math.exp(con));
        setShape(1 / coeff);

        data.resetPtr();
    }

    /**
     * @param data Original data.
     * @param shift Shift parameter to assume.
     * @return A {@link DataSet} in which variable 0 are the sample values and 1 are the sample empiricals.
     */
    private DataSet createRegressDataSet(final DataSet data, final double shift)
    {
        final DataSet newdata = new DataSet(data.getMaximumSampleSize(), 2);
        boolean notatend = true;
        double[] asample = new double[2];

        int samplevar, empvar;
        samplevar = data.getFitSampleVariable();
        empvar = data.getFitCDFVariable();

        data.resetPtr();

        while(notatend)
        {
            //Set the data point in the new data set according to the equations below.
            //Since the regression is REVERSED, the values and empiricals variables are
            //swapped.
            if(data.getCurrentValue(samplevar) != -999)
            {
                asample = new double[2];
                asample[0] = Math.log(Math.log(1 / (1 - data.getCurrentValue(empvar))));
                asample[1] = Math.log(Math.log(data.getCurrentValue(samplevar) + 1 - shift));
                newdata.addSample(asample);
            }

            notatend = data.next();
        }

        //Return the new data set.
        newdata.setFitSampleVariable(0);
        newdata.setFitCDFVariable(1);
        return newdata;
    }

    @Override
    public void fitToData(final DataSet data, double[] fitParms) throws DataFittingDistributionException
    {
        if(!_fitShift)
        {
            estimateRegression(data);
        }
        else
        {
            if(fitParms == null)
            {
                fitParms = new double[]{0.0D};
            }
            DistributionTools.optimizeShiftFitForBoundedBelowDistribution(this, data, fitParms);
        }
    }

    @Override
    public void estimateParameters(final DataSet data, final double shift) throws DataFittingDistributionException
    {
        final LogweibullDist fitDist = new LogweibullDist();
        fitDist.setFitShift(false); //force regression fit
        fitDist.setShift(shift);
        fitDist.fitToData(data);

        //Copy the results.
        setScale(fitDist.getScale());
        setShape(fitDist.getShape());
        setShift(shift);
    }
}
