package ohd.hseb.hefs.utils.dist.types;

import org.apache.commons.math.distribution.NormalDistribution;
import org.apache.commons.math.distribution.NormalDistributionImpl;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

/**
 * This subclass of {@link ContinuousDist} defines a normal (Gaussian) distribution. The parameters of the normal are
 * the mean and standard deviation (NOT variance). {@link #fitToData(DataSet)} uses the maximum likelihood estimators
 * (the sample mean and sample standard deviation) to estimate the distribution parameters.<br>
 * <br>
 * Approximations to the non-closed form CDF are provided by Abramowitz and Stegun, which allows for fast computation.
 * The inverse CDF is computed via commons math {@link NormalDistribution}.
 * 
 * @author hank
 */
public class NormalDist extends ContinuousDist
{
    public static NormalDist STD_NORM_DIST = new NormalDist();

//    private static final int BOUNDS_POWER = 10;
//
//    private static final double PROBABILITY_UPPER_BOUND = 1.0
//        - Math.pow(10.0, -1.0 * BOUNDS_POWER);
//    private static final double PROBABILITY_LOWER_BOUND =
//                                                        Math.pow(10.0,
//                                                                 -1.0 * BOUNDS_POWER);

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public final static int MEAN = 0;

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public final static int STDEV = 1;

    private NormalDistribution _wrappedInstance;

    /**
     * Default is a standard normal.
     */
    public NormalDist()
    {
        super(new XMLDouble("domain", 0.0D, null, null),
              new XMLDouble("mean", 0.0D, null, null),
              new XMLDouble("stdev", 1.0D, 0.0, null));
        updateWrappedInstance();
    }

    /**
     * @param mean The mean variable to use.
     * @param stdev The standard normal variable to use.
     */
    public NormalDist(final double mean, final double stdev)
    {
        this();
        setMean(mean);
        setStandardDeviation(stdev);
        updateWrappedInstance();
    }

    private void updateWrappedInstance()
    {
        _wrappedInstance = new NormalDistributionImpl(getMean(),
                                                      getStandardDeviation());
    }

    public void setMean(final double mean)
    {
        setParameter(MEAN, mean);
        updateWrappedInstance();
    }

    public double getMean()
    {
        return getParameter(MEAN).doubleValue();
    }

    public void setStandardDeviation(final double stdev)
    {
        setParameter(STDEV, stdev);
        updateWrappedInstance();
    }

    public double getStandardDeviation()
    {
        return getParameter(STDEV).doubleValue();
    }

    //TODO Compare the CDF method below with that used in Apache commons math.  Is the below significantly faster???  See NormalDistributionImpl below.
    @Override
    public double functionCDF(final Double value)
    {
        // The NormalDistribution below will return a 1 if it's given a NaN.  Very bad!
        if(Double.isNaN(value))
        {
            return Double.NaN;
        }
        try
        {
            return _wrappedInstance.cumulativeProbability(value);
        }
        catch(final Exception e)
        {
            return getMissing();
        }

//        double y, z, number, a, b;
//
//        a = this.getMean();
//        b = this.getStandardDeviation();
//
//        //The polynomial corresponds to standard normal, so first I must transform x.
//        z = (value - a) / b;
//
//        //It also only works on positive values, so if z<0 then make y the reflection of z.
//        if(z < 0)
//            y = -1 * z;
//
//        //Otherwise set y to be z.
//        else
//            y = z;
//
//        //Calculate the number according to the polynomial.
//        number = (1 - 0.5 * Math.pow((1 + 0.0498673470 * y
//            + 0.0211410061 * Math.pow(y, 2) + 0.0032776263 * Math.pow(y, 3)
//            + 0.0000380036 * Math.pow(y, 4) + 0.0000488906 * Math.pow(y, 5)
//            + 0.0000053830 * Math.pow(y, 6)), -16));
//
//        if((number + 0) < PROBABILITY_LOWER_BOUND)
//            number = PROBABILITY_LOWER_BOUND;
//        if(((number + 0) > PROBABILITY_UPPER_BOUND) && ((number + 0) < 1))
//            number = PROBABILITY_UPPER_BOUND;
//
//        //If z is less than 0, then the value above is equal to 1 - Q(z), so return
//        //1 - number.
//        if(z < 0)
//            return 1 - number;
//
//        //Otherwise, just return the number.
//        else
//            return number;
    }

    @Override
    public double functionPDF(final Double value)
    {
        return _wrappedInstance.density(value);
//        double number = 0, a, b;
//
//        a = this.getMean();
//        b = this.getStandardDeviation();
//
//        number = ((1 / (Math.sqrt(2 * Math.PI) * b))
//            * Math.exp(-1 * Math.pow(value - a, 2) / (2 * Math.pow(b, 2))));
//
//        if(number == 0.0)
//            number = Math.pow(10, -20);
//
//        return number;

    }

    @Override
    public double functionInverseCDF(final double prob)
    {
        //This uses the Commons NormalDistributionImpl, which has accuracy to 1E-6.  The Abramowitz and Stegun accuracy is much less.
        // The NormalDistribution below will return a 1 if it's given a NaN.  Very bad!
        if(Double.isNaN(prob))
        {
            return Double.NaN;
        }
        try
        {
            return _wrappedInstance.inverseCumulativeProbability(prob);
        }
        catch(final Exception e)
        {
            return getMissing();
        }

//Abramowitz and Stegun:

//        double number, t, temp, a, b;
//
//        a = getMean();
//        b = getStandardDeviation();
//
//        temp = prob;
//
//        if(prob == 1)
//            prob -= Math.pow(10, -20);
//
//        if(prob == 0)
//            prob += Math.pow(10, -20);
//
//        //The function only works on probabilities less than 0.5.  So, if it is larger
//        //than 0.5, subtract it from 1 and use the new value.
//        if(prob > 0.5)
//            prob = 1 - prob;
//
//        //The following returns the value, number, so that the exceedance prob. for number
//        //is p.  This is only for a standard normal distribution.    
//        t = Math.sqrt(Math.log(1 / (prob * prob)));
//        number = t - (2.515517 + 0.802853 * t + 0.010328 * t * t)
//            / (1 + 1.432788 * t + 0.189269 * t * t + 0.001308 * Math.pow(t, 3));
//
//        //if the original probability is less than 0.5, then I must reflect the value of
//        //number about the y-axis and multiply by the stan.dev. and add the mean to handle
//        //non-standard normal distributions.
//        if(temp < 0.5)
//            return b * -1 * number + a;
//        else
//
//        //If the value is exactly 0.5, then the correct value must be the mean.
//        if(temp == 0.5)
//            return a;
//        else
//
//        //If the value of temp is larger than 0.5, then the number is correct as is and 
//        //must only be transformed to handle non-standard distributions.
//        if(temp > 0.5)
//            return b * number + a;
//
//        return b * number + a;
    }

    /**
     * @param data Make sure the {@link DataSet#setFitSampleVariable(int)} has been called to specify the variable which
     *            contains the data to which to fit the distribution.
     */
    private boolean estimateMaximumLikelihood(final DataSet data)
    {
        setMean(data.mean(data.getFitSampleVariable()));
        setStandardDeviation(Math.sqrt(data.sampleVariance(data.getFitSampleVariable())));
        return true;
    }

    /**
     * Note that fitparms is ignored directly calling {@link #estimateMaximumLikelihood(DataSet)}.
     */
    @Override
    public void fitToData(final DataSet data,
                          final double[] fitparms) throws DataFittingDistributionException
    {
        estimateMaximumLikelihood(data);
    }

}
