package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.DistributionTools;
import ohd.hseb.hefs.utils.dist.ShiftOptimizationFittingDistribution;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

public class ParetoDist extends ContinuousDist implements ShiftOptimizationFittingDistribution
{

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SCALE = 0;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHAPE = 1;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHIFT = 2;

    private static final double DEFAULT_SCALE = 1.0D;
    /**
     * scale parameter is the same as location parameter a at page 119, Statistical Distributions, 2nd ed by M. Evans,
     * N. Hastings, and B. Peacock
     */
    private static final double DEFAULT_SHAPE = 2.0D;
    private static final double DEFAULT_SHIFT = 0.0D;

    private boolean _fitShift = false;

    /**
     * Constructs the Gamma distribution using the defaults provided by the constants: {@link #DEFAULT_SCALE},
     * {@link #DEFAULT_SHAPE}, and {@link #DEFAULT_FIXED_SHIFT}.
     */
    public ParetoDist()
    {
        this(DEFAULT_SCALE, DEFAULT_SHAPE, DEFAULT_SHIFT);
    }

    /**
     * Constructs the distribution given the specified parameters.
     * 
     * @param scale Scale parameter.
     * @param shape Shape parameter.
     * @param shift Shift parameter.
     */
    public ParetoDist(final double scale, final double shape, final double shift)
    {
        super(new XMLDouble("domain", shift, shift, null),
              new XMLDouble("scale", scale, 0.0D, null),
              new XMLDouble("shape", shape, 0.0D, null),
              new XMLDouble("shift", shift, null, null));
    }

    public void setFitShift(final boolean b)
    {
        _fitShift = b;
    }

    public void setScale(final double scale)
    {
        setParameter(SCALE, scale);
    }

    public double getScale()
    {
        return getParameter(SCALE).doubleValue();
    }

    public void setShape(final double shape)
    {
        setParameter(SHAPE, shape);
    }

    public double getShape()
    {
        return getParameter(SHAPE).doubleValue();
    }

    /**
     * Ensures {@link #setDomainLowerBound(Double)} is called as well.
     */
    public void setShift(final double shift)
    {
        setParameter(SHIFT, shift);
        setDomainLowerBound(shift);
    }

    public double getShift()
    {
        return getParameter(SHIFT).doubleValue();
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
    }

    //Overrides Distribution class
    @Override
    public double functionCDF(final Double value)
    {
        double temp;
        temp = 1 - Math.pow((getScale() / (value - getShift())), getShape());
        // Make sure the number is within reasonable bounds.
        return DistributionTools.returnTrimmedProbability(temp);
    }

    @Override
    public double functionPDF(final Double value)
    {
        double a, b, c;
        final double temp;
        a = getScale();
        b = getShift();
        c = getShape();
        temp = c * Math.pow(a, c) / Math.pow((value - b), c + 1);
        return temp;
    }

    @Override
    public double functionInverseCDF(final double prob)
    {
        double a, b, c;
        final double temp;
        a = getScale();
        b = getShift();
        c = getShape();
        temp = a * Math.pow((1 - prob), -1 / c);
        return temp + b;
    }

    @Override
    public void fitToData(final DataSet data, double[] fitParms) throws DataFittingDistributionException
    {
        double c2;
        c2 = 0;
        if(_fitShift)
        {
            if(fitParms == null)
            {
                fitParms = new double[]{0.0D};
            }
            DistributionTools.optimizeShiftFitForBoundedBelowDistribution(this, data, fitParms);
        }
        else
        {
            final DataSet useData = new DataSet(data);
            useData.applyShiftTransform(useData.getFitSampleVariable(), -1 * getShift());
            final double minX = useData.getSmallest(useData.getFitSampleVariable());
            double x;
            for(int i = 0; i < useData.getSampleSize(); i++)
            {
                x = useData.getValue(i, useData.getFitSampleVariable());
                c2 = c2 + Math.log(x / minX);
            }
            c2 = c2 / useData.getSampleSize();
            c2 = 1 / c2;
            setScale(minX);
            setShape(c2);
        }
    }

    @Override
    public void estimateParameters(final DataSet data, final double shift) throws DataFittingDistributionException
    {
        final ParetoDist fitDist = new ParetoDist();
        fitDist.setFitShift(false); //force regression fit
        fitDist.setShift(shift);
        fitDist.fitToData(data);

        //Copy the results.
        setScale(fitDist.getScale());
        setShape(fitDist.getShape());
        setShift(shift);
    }
}
