package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.LMomentsFittingDistribution;
import ohd.hseb.hefs.utils.dist.LMomentsMath;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

import org.apache.commons.math.FunctionEvaluationException;
import org.apache.commons.math.analysis.MultivariateRealFunction;
import org.apache.commons.math.optimization.GoalType;
import org.apache.commons.math.optimization.RealConvergenceChecker;
import org.apache.commons.math.optimization.RealPointValuePair;
import org.apache.commons.math.optimization.direct.NelderMead;

/**
 * See... http://www.mathwave.com/articles/wakeby_distribution.html
 * 
 * @author herrhd
 */

public class WakebyDist extends ContinuousDist implements MultivariateRealFunction, RealConvergenceChecker,
LMomentsFittingDistribution
{
    public static int MAX_ITERATIONS = 10;

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public static final int ALPHA = 0;

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public static final int BETA = 1;

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public static final int GAMMA = 2;

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public static final int DELTA = 3;

    /**
     * Must be kept consistent with the ordering of the parameters in the constructor.
     */
    public static final int SHIFT = 4;

    public static final double DEFAULT_ALPHA = 1d;
    public static final double DEFAULT_BETA = 1d;
    public static final double DEFAULT_GAMMA = 1d;
    public static final double DEFAULT_DELTA = 1d;
    public static final double DEFAULT_SHIFT = 0d;

    private boolean _fitShift = true; // true -> use l-moments

    private DataSet _fittedDataSet = null;
    private Double _lowerBoundOnShift = null;
    private Double _upperBoundOnShift = null;
    private int _numberOfIterationsOfLastOptimization = 0;

    /**
     * Default constructs makes use of the DEFAULT constants.
     */
    public WakebyDist()
    {
        this(new double[]{DEFAULT_ALPHA, DEFAULT_BETA, DEFAULT_GAMMA, DEFAULT_DELTA, DEFAULT_SHIFT});
    }

    /**
     * Note that there are special conditions that dictate the allowed values for the parameters. Thus, we do not worry
     * about the value of the parameters when initializing the variables. Rather, {@link #areParametersValid(double[])}
     * is called given the provided array.
     * 
     * @param parameters Array of parameter values in the same order as implied by the constants.
     */
    public WakebyDist(final double[] parameters)
    {
        super(new XMLDouble("domain", parameters[SHIFT], parameters[SHIFT], null),
              new XMLDouble("alpha", parameters[ALPHA], null, null),
              new XMLDouble("beta", parameters[BETA], null, null),
              new XMLDouble("gamma", parameters[GAMMA], null, null),
              new XMLDouble("delta", parameters[DELTA], null, null),
              new XMLDouble("shift", parameters[SHIFT], null, null));
        if(!areParametersValid())
        {
            throw new IllegalArgumentException("Initial parameter are not valid.");
        }
    }

    public void setFitShift(final boolean b)
    {
        _fitShift = b;
    }

    /**
     * Calls {@link #areParametersValid(double[])}.
     */
    private boolean areParametersValid()
    {
        return areParametersValid(NumberTools.convertNumbersToDoublesArray(getParameters()));
    }

    /**
     * @param parms The parameters to check.
     * @return True if the provided parameters satisfy the conditions for a Wakeby; false if not.
     */
    private boolean areParametersValid(final double[] parms)
    {
        // See: http://www.mathwave.com/articles/wakeby_distribution.html

        if((getAlpha() != 0) || (getGamma() != 0))
        {
            if((getBeta() + getDelta() > 0) || ((getBeta() == 0) && (getGamma() == 0) && (getDelta() == 0)))
            {
                if((getAlpha() == 0) && (getBeta() != 0))
                {
                    return false;
                }
                if((getGamma() == 0) && (getDelta() != 0))
                {
                    return false;
                }
                if((getGamma() >= 0) && (getAlpha() + getGamma() >= 0))
                {
                    return true;
                }
            }
        }

        return false;
    }

    public double getAlpha()
    {
        return getParameter(ALPHA).doubleValue();
    }

    public void setAlpha(final double alpha)
    {
        this.setParameter(ALPHA, alpha);
    }

    public double getBeta()
    {
        return getParameter(BETA).doubleValue();
    }

    public void setBeta(final double beta)
    {
        this.setParameter(BETA, beta);
    }

    public double getGamma()
    {
        return getParameter(GAMMA).doubleValue();
    }

    public void setGamma(final double gamma)
    {
        this.setParameter(GAMMA, gamma);
    }

    public double getDelta()
    {
        return getParameter(DELTA).doubleValue();
    }

    public void setDelta(final double delta)
    {
        this.setParameter(DELTA, delta);
    }

    public double getShift()
    {
        return getParameter(SHIFT).doubleValue();
    }

    /**
     * Ensures {@link #setDomainLowerBound(Double)} is called as well.
     */
    public void setShift(final double shift)
    {
        setParameter(SHIFT, shift);
        setDomainLowerBound(shift);
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
        if(index == SHIFT)
        {
            setDomainLowerBound(value.doubleValue());
        }
    }

    /**
     * Calls {@link #areParametersValid()} before setting the parameters based on the provided array.
     * 
     * @param values
     * @return False if the parameters are not valid.
     */
    public boolean setParameters(final double[] values)
    {
        if(!this.areParametersValid(values))
        {
            return false;
        }

        for(int i = 0; i < values.length; i++)
        {
            ((XMLDouble)getParameterStorageVar(i)).set(values[i]);
        }
        return true;
    }

    public boolean isMaxIterations()
    {
        return (this._numberOfIterationsOfLastOptimization >= MAX_ITERATIONS);
    }

    //
    //    @Override
    //public double functionCDF(final double value)
    //{
    //    final double low = getDomainLowerBound();
    //
    //    if((low == null) || isMissing(low))
    //    {
    //        throw new IllegalArgumentException("The lower bound is not defined, which is invalid.");
    //    }
    //
    //    // Check the lower bound constraint.
    //
    //   if(value < low)
    //  {
    //      return getMissing();
    //  }
    //
    //  if((getDelta() < 0) || (getGamma() <= 0))
    //  {
    //      if(value > getShift() + getAlpha() / getBeta() - getGamma() / getDelta())
    //      {
    //          return getMissing();
    //      }
    //  }
    //
    //  //System.out.println("####>>    computing cdf for ... " + HString.buildStringFromArray(_parameter, ", ") + " -- "
    //  //    + value);
    //
    //  double lowerProb = 0.0D;
    //  double upperProb = 1.0D;
    //  double halfProb;
    //  double testValue;
    //    while(upperProb - lowerProb > 0.001)
    //     {
    //        halfProb = (lowerProb + upperProb) / 2.0D;
    //        testValue = this.functionInverseCDF(halfProb);
    //        //System.out.println("####>> " + halfProb + " -- " + testValue);
    //        if(testValue == value)
    //        {
    //            return halfProb;
    //        }
    //        if(testValue > value)
    //        {
    //            upperProb = halfProb;
    //        }
    //        else
    //        {
    //            lowerProb = halfProb;
    //        }
    //    }
    //     //    System.out.println("####>>       returning " + lowerProb + ", " + upperProb);
    //     return (lowerProb + upperProb) / 2.0D;
    //}

    /**
     * functionQUAwak() computes the quantile for the Wakeby distribution. The code is based on the FORTRAN routine (in
     * lmoments.f) QUAWAK from: http://www.jstor.org/stable/2345653. See also the EPP3 fortran prototype code
     * 
     * @param value - The value to find the quantile for
     */

    @Override
    public double functionInverseCDF(final double value)
    {

        double a, b, c, d, xi;
        double temp, y1, y2, z;
        double quawak = 0.0;
        boolean done = false;

        final double ufl = -170.0;

        a = getAlpha();
        b = getBeta();
        c = getGamma();
        d = getDelta();
        xi = getShift();

        // Test for valid parameters

        if((b + d <= 0.0) && ((b != 0.0) || (c != 0.0) || (d != 0.0)))
        {
            throw new IllegalArgumentException("(b + d <= 0.0) && ((b != 0.0) || (c != 0.0) || (d != 0.0))");
        }
        if((a == 0.0) && (b != 0.0))
        {
            throw new IllegalArgumentException("(a == 0.0) && (b != 0.0)");
        }
        if((c == 0.0) && (d != 0.0))
        {
            throw new IllegalArgumentException("(c == 0.0) && (d != 0.0)");
        }
        if((c < 0.0) || (a + c < 0.0))
        {
            throw new IllegalArgumentException("(c < 0.0) || (a + c < 0.0)");
        }
        if((a == 0.0) && (c == 0.0))
        {
            throw new IllegalArgumentException("(a == 0.0) && (c == 0.0)");
        }

        if((value <= 0.0) || (value >= 1.0))
        {
            if(value == 0.0)
            {
                quawak = xi;
            }
            else if(value == 1.0)
            {
                if(d > 0.0)
                {
                    quawak = 0.0;
                }
                else if(d < 0.0)
                {
                    quawak = xi + a / b - c / d;
                }
                else
                // d == 0.0
                {
                    if(c > 0.0)
                    {
                        quawak = 0.0;
                    }
                    else if((c == 0.0) && (b == 0.0))
                    {
                        quawak = 0.0;
                    }
                    else if((c == 0.0) && (b > 0.0))
                    {
                        quawak = xi + a / b;
                    }
                }
            }

            done = true;
        }

        if(done == false)
        {
            z = -Math.log(1.0 - value);
            y1 = z;

            if(b != 0.0)
            {
                temp = -b * z;

                if(temp < ufl)
                {
                    y1 = 1.0 / b;
                }

                if(temp >= ufl)
                {
                    y1 = (1.0 - Math.exp(temp)) / b;
                }
            }

            y2 = z;

            if(d != 0.0)
            {
                y2 = (1.0 - Math.exp(d * y2)) / (-d);
            }

            quawak = xi + a * y1 + c * y2;
        }

        return quawak;
    }

    /**
     * functionCDFwak() computes the CDF for the Wakeby distribution. The code is based on the FORTRAN routine (in
     * lmoments.f) CDFWAK from: http://www.jstor.org/stable/2345653. See also the EPP3 fortran prototype code
     * 
     * @param value - The value to find the CDF for
     */

    @Override
    public double functionCDF(final Double value)
    {
        double a, b, c, d, xi;
        double bz, eb, ed, gb, gd, znew;
        double z = 0.7;
        double xest, func;
        double deriv1, deriv2;
        double temp, zinc;

        double cdfwak = 0.0;
        boolean done = false;

        final double eps = 1.0e-8;
        final int maxit = 20;
        final double zincmx = 3.0;
        final double zmult = 0.2;
        final double ufl = -170.0;

        a = getAlpha();
        b = getBeta();
        c = getGamma();
        d = getDelta();
        xi = getShift();

        // Test for valid parameters

        if((b + d <= 0.0) && ((b != 0.0) || (c != 0.0) || (d != 0.0)))
        {
            throw new IllegalArgumentException("(b + d <= 0.0) && ((b != 0.0) || (c != 0.0) || (d != 0.0))");
        }
        if((a == 0.0) && (b != 0.0))
        {
            throw new IllegalArgumentException("(a == 0.0) && (b != 0.0)");
        }
        if((c == 0.0) && (d != 0.0))
        {
            throw new IllegalArgumentException("(c == 0.0) && (d != 0.0)");
        }
        if((c < 0.0) || (a + c < 0.0))
        {
            throw new IllegalArgumentException("(c < 0.0) || (a + c < 0.0)");
        }
        if((a == 0.0) && (c == 0.0))
        {
            throw new IllegalArgumentException("(a == 0.0) && (c == 0.0)");
        }

        if(value <= xi)
        {
            return 0.0;
        }

        //  Special cases

        if((b == 0.0) && (c == 0.0) && (d == 0.0))
        {
            // Wakeby is exponential

            z = (value - xi) / a;
            done = true;
        }
        else if(c == 0.0)
        {
            // Wakeby is generalized Pareto, bounded above

            if(value >= xi + a / b)
            {
                return 1.0;
            }
            else
            {
                z = -Math.log(1.0 - (value - xi) * b / a) / b;
                done = true;
            }
        }
        else if(a == 0.0)
        {
            // Wakeby is generalized Pareto, no upper bound

            z = Math.log(1.0 + (value - xi) * d / c) / d;
            done = true;
        }

        // General case 

        cdfwak = 1.0;

        if((d < 0.0) && (value >= xi + a / b - c / d))
        {
            return cdfwak;
        }

        if(done == false)
        {
            // Initial values for iteration:
            //  
            // If x is in the lowest decile of the distribution, start at z=0 (f=0);
            //  
            // If x is in the highest percentile of the distribution,
            // starting value is obtained from asymptotic form of the
            // distribution for large z (f near 1);
            //  
            // Otherwise start at z = 0.7 (close to f = 0.5).

            z = 0.7;

            if(value < functionInverseCDF(0.1))
            {
                z = 0.0;
            }
            else if(value < functionInverseCDF(0.99))
            {
                z = 0.7;
            }
            else if(d < 0.0)
            {
                z = Math.log((value - xi - a / b) * d / c + 1.0) / d;
            }
            else if(d == 0.0)
            {
                z = (value - xi - a / b) / c;
            }
            else if(d > 0.0)
            {
                z = Math.log((value - xi) * d / c + 1.0) / d;
            }

            // Halley's method, with modifications:
            //  
            // If Halley iteration would move in wrong direction
            // (temp <= 0.0), use ordinary Newton-Raphson instead;
            //
            // If step goes too far (zinc .> zincmx || znew <= 0.0), limit its length.

            for(int it = 1; it <= maxit; it = it + 1)
            {
                eb = 0.0;
                bz = -b * z;

                if(bz >= ufl)
                {
                    eb = Math.exp(bz);
                }

                gb = z;

                if(Math.abs(b) > eps)
                {
                    gb = (1.0 - eb) / b;
                }

                ed = Math.exp(d * z);
                gd = -z;

                if(Math.abs(d) > eps)
                {
                    gd = (1.0 - ed) / d;
                }

                xest = xi + a * gb - c * gd;
                func = value - xest;

                deriv1 = a * eb + c * ed;
                deriv2 = -a * b * eb + c * d * ed;

                temp = deriv1 + 0.5 * func * deriv2 / deriv1;

                if(temp <= 0.0)
                {
                    temp = deriv1;
                }

                zinc = func / temp;

                if(zinc > zincmx)
                {
                    zinc = zincmx;
                }

                znew = z + zinc;

                if(znew <= 0.0)
                {
                    z = z * zmult;
                }
                else
                {
                    z = znew;

                    if(Math.abs(zinc) <= eps)
                    {
                        done = true;
                    }
                }

                if(done == true)
                {
                    break;
                }
            }
        }

        //if(done == false)
        //{
        // If we got here, the function didn't converge
        // ... could set an error flag
        //}

        // Convert the z value to a probability

        if(-z < ufl)
        {
            cdfwak = 1.0;
        }
        else
        {
            cdfwak = 1.0 - Math.exp(-z);
        }

        return cdfwak;
    }

    @Override
    public double functionPDF(final Double value)
    {
        double F, t, temp;

        // See: http://www.itl.nist.gov/div898/software/dataplot/refman2/auxillar/wakpdf.htm

        F = functionCDF(value - getShift());

        t = Math.pow(1.0 - F, getBeta() + getDelta());

        try
        {
            temp = Math.pow(1.0 - F, getDelta() + 1) / ((getAlpha() * t) + getGamma());
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        return temp;
    }

    // @Override
    // public double functionInverseCDF(final double prob)
    // {
    //    // See http://www.mathwave.com/articles/wakeby_distribution.html
    //
    //     return getShift() + (getAlpha() / getBeta()) * (1 - Math.pow(1 - prob, getBeta())) - (getGamma() / getDelta())
    //         * (1 - Math.pow(1 - prob, -1.0 * getDelta()));
    // }

    /**
     * Wraps {@link #optimizeFitToData(DataSet, double[])} basing its return on whether an exception is thrown.
     */
    @Override
    public void fitToData(final DataSet data, final double[] fitparms) throws DataFittingDistributionException
    {
        if(!_fitShift)
        {
            try
            {
                optimizeFitToData(data, fitparms);
            }
            catch(final Exception e)
            {
                throw new DataFittingDistributionException("Unable to fit Wakeby: " + e.getMessage(), e);
            }
        }
        else
        {
            fitToLMoments(data);
        }
    }

    // XXX See the paper here: http://dspace.mit.edu/bitstream/handle/1721.1/31278/mit-el-77-033wp-04146753.pdf?sequence=1
    // It explains a fitting algorithm at the end.  Perhaps that would work better than the optimizer shown below.

    /**
     * Call this method to use a generic optimization tool for fitting the data: {@link NelderMead}.
     * 
     * @param data Standard fitting data set, with both {@link DataSet#getFitSampleVariable()} and
     *            {@link DataSet#getFitCDFVariable()} returning values.
     * @param fitparms If not null, an array containing a single double value specifying the lower bound on the shift
     *            parameter, typically zero. The upper bound will be the smallest value to fit. If null, the shift is
     *            fixed to 0.
     * @throws Exception
     */
    public void optimizeFitToData(final DataSet data, final double[] fitparms) throws Exception
    {
        this._fittedDataSet = data;

        final boolean shiftFitted = (fitparms != null);
        double[] startingPoint;
        if(!shiftFitted)
        {
            _lowerBoundOnShift = null;
            _upperBoundOnShift = null;
            setShift(0.0D);
            startingPoint = new double[]{1, 1, 1, 1};
        }
        else
        {
            _lowerBoundOnShift = fitparms[0];
            _upperBoundOnShift = data.getSmallest(data.getFitSampleVariable());
            startingPoint = new double[]{1, 1, 1, 1, 0};
        }

        //MultiDirectional optimizer = new MultiDirectional();
        final NelderMead optimizer = new NelderMead();
        optimizer.setConvergenceChecker(this);
        double previousObjFuncValue = 10.0;
        double currentObjFuncValue = 1.0;
        RealPointValuePair results = null;
        _numberOfIterationsOfLastOptimization = 0;
        while(Math.abs(previousObjFuncValue - currentObjFuncValue) > 0.01)
        {
            previousObjFuncValue = currentObjFuncValue;
            results = optimizer.optimize(this, GoalType.MINIMIZE, startingPoint);
            currentObjFuncValue = results.getValue();
            startingPoint = results.getPoint();
            _numberOfIterationsOfLastOptimization++;
            if(isMaxIterations())
            {
                break;
            }
        }
        setAlpha(results.getPoint()[ALPHA]);
        setBeta(results.getPoint()[BETA]);
        setGamma(results.getPoint()[GAMMA]);
        setDelta(results.getPoint()[DELTA]);
        if(shiftFitted)
        {
            setShift(results.getPoint()[SHIFT]);
        }
        else
        {
            setShift(0.0D);
        }
    }

    /**
     * Used in the optimization.
     */
    @Override
    public double value(final double[] currentParametersForOptimization) throws FunctionEvaluationException,
                                                                        IllegalArgumentException
    {
        if(!this.areParametersValid(currentParametersForOptimization))
        {
            return Double.MAX_VALUE;
        }

        double[] parametersToUse = currentParametersForOptimization;
        if(parametersToUse.length == 4)
        {
            parametersToUse = new double[5];
            for(int i = 0; i < 4; i++)
            {
                parametersToUse[i] = currentParametersForOptimization[i];
            }
            parametersToUse[SHIFT] = 0.0D;
        }

        //Check the shift, only if the lower bound is non-null.
        if(_lowerBoundOnShift != null)
        {
            if((currentParametersForOptimization[SHIFT] < this._lowerBoundOnShift)
                || (currentParametersForOptimization[SHIFT] > this._upperBoundOnShift))
            {
                return Double.MAX_VALUE;
            }
        }

        //Return the maximum error of the distribution fit.
        final WakebyDist dist = new WakebyDist(parametersToUse);

//        System.out.println("####>> calculating... "
//            + HString.buildStringFromArray(parametersToUse, ", ")
//            + " -- "
//            + _fittedDataSet.maximumError(_fittedDataSet.getFitSampleVariable(),
//                                          _fittedDataSet.getFitCDFVariable(),
//                                          dist));

        final double error = _fittedDataSet.maximumError(_fittedDataSet.getFitSampleVariable(),
                                                         _fittedDataSet.getFitCDFVariable(),
                                                         dist);
        return error;
    }

    /**
     * Used in the optimization.
     */
    @Override
    public boolean converged(final int arg0, final RealPointValuePair previous, final RealPointValuePair current)
    {
        if((Math.abs(previous.getPoint()[ALPHA] - current.getPoint()[ALPHA]) < 0.001)
            && (Math.abs(previous.getPoint()[BETA] - current.getPoint()[BETA]) < 0.001)
            && (Math.abs(previous.getPoint()[GAMMA] - current.getPoint()[GAMMA]) < 0.001)
            && (Math.abs(previous.getPoint()[DELTA] - current.getPoint()[DELTA]) < 0.001)
            && ((previous.getPoint().length == 4) || (Math.abs(previous.getPoint()[SHIFT] - current.getPoint()[SHIFT]) < 0.001)))
        {
            return true;
        }
        return false;
    }

    /**
     * fitToLMoments() computes the 5 Wakeby parameters from the 5 L-moment ratios. The code is based on the FORTRAN
     * routine (in lmoments.f) PELWAK from: http://www.jstor.org/stable/2345653. See also the EPP3 fortran prototype
     * code
     * 
     * @param lmoments - The 5 lmoments for which to calculate the 5 Wakeby parameters.
     */

    @Override
    public void fitToLMoments(final double[] lmoments)
    {
        double alam1, alam2, alam3, alam4, alam5;
        double xn1, xn2, xn3;
        double xc1, xc2, xc3;
        double xa, xb, xc;
        double zn1, zn2, zn3;
        double zc1, zc2, zc3;
        double za, zb, zc;
        double disc;
        double root1, root2;
        double a = DEFAULT_ALPHA;
        double b = DEFAULT_BETA;
        double c = DEFAULT_GAMMA;
        double d = DEFAULT_DELTA;
        double xi = DEFAULT_SHIFT;
        boolean done = false;

        if(lmoments[1] <= 0.0)
        {
            throw new IllegalArgumentException("lmoments[1] <= 0.0.");
        }
        if(Math.abs(lmoments[2]) >= 1.0)
        {
            throw new IllegalArgumentException("lmoments[2] >= 1.0.");
        }
        if(Math.abs(lmoments[3]) >= 1.0)
        {
            throw new IllegalArgumentException("lmoments[3] >= 1.0.");
        }
        if(Math.abs(lmoments[4]) >= 1.0)
        {
            throw new IllegalArgumentException("lmoments[4] >= 1.0.");
        }

        alam1 = lmoments[0];
        alam2 = lmoments[1];
        alam3 = lmoments[2] * alam2;
        alam4 = lmoments[3] * alam2;
        alam5 = lmoments[4] * alam2;

        xn1 = 3.0 * alam2 - 25.0 * alam3 + 32.0 * alam4;
        xn2 = -3.0 * alam2 + 5.0 * alam3 + 8.0 * alam4;
        xn3 = 3.0 * alam2 + 5.0 * alam3 + 2.0 * alam4;

        xc1 = 7.0 * alam2 - 85.0 * alam3 + 203.0 * alam4 - 125.0 * alam5;
        xc2 = -7.0 * alam2 + 25.0 * alam3 + 7.0 * alam4 - 25.0 * alam5;
        xc3 = 7.0 * alam2 + 5.0 * alam3 - 7.0 * alam4 - 5.0 * alam5;

        xa = xn2 * xc3 - xc2 * xn3;
        xb = xn1 * xc3 - xc1 * xn3;
        xc = xn1 * xc2 - xc1 * xn2;

        disc = xb * xb - 4.0 * xa * xc;

        if(disc >= 0.0)
        {
            disc = Math.sqrt(disc);

            root1 = 0.5 * (-xb + disc) / xa;
            root2 = 0.5 * (-xb - disc) / xa;

            b = Math.max(root1, root2);
            d = -Math.min(root1, root2);

            if(d < 1.0)
            {
                a = (1.0 + b) * (2.0 + b) * (3.0 + b) / (4.0 * (b + d)) * ((1.0 + d) * alam2 - (3.0 - d) * alam3);

                c = -(1.0 - d) * (2.0 - d) * (3.0 - d) / (4.0 * (b + d)) * ((1.0 - b) * alam2 - (3.0 + b) * alam3);

                xi = alam1 - a / (1.0 + b) - c / (1.0 - d);

                if((c >= 0.0) && (a + c >= 0.0))
                {
                    done = true;
                }
            }
        }

        if(done == false) // try xi = 0
        {
            xi = 0.0;

            zn1 = 4.0 * alam1 - 11.0 * alam2 + 9.0 * alam3;
            zn2 = -alam2 + 3.0 * alam3;
            zn3 = alam2 + alam3;

            zc1 = 10.0 * alam1 - 29.0 * alam2 + 35.0 * alam3 - 16.0 * alam4;
            zc2 = -alam2 + 5.0 * alam3 - 4.0 * alam4;
            zc3 = alam2 - alam4;

            za = zn2 * zc3 - zc2 * zn3;
            zb = zn1 * zc3 - zc1 * zn3;
            zc = zn1 * zc2 - zc1 * zn2;

            disc = zb * zb - 4.0 * za * zc;

            if(disc >= 0.0)
            {
                disc = Math.sqrt(disc);

                root1 = 0.5 * (-zb + disc) / za;
                root2 = 0.5 * (-zb - disc) / za;

                b = Math.max(root1, root2);
                d = -Math.min(root1, root2);

                if(d < 1.0)
                {
                    a = (1.0 + b) * (2.0 + b) / (b + d) * (alam1 - (2.0 - d) * alam2);
                    b = -(1.0 - d) * (2.0 - d) / (b + d) * (alam1 - (2.0 + b) * alam2);

                    if((c >= 0.0) && (a + c >= 0.0))
                    {
                        done = true;
                    }
                }
            }
        }

        if(done == false) // try generalized Pareto
        {
            d = -(1.0 - 3.0 * lmoments[2]) / (1.0 + lmoments[2]);
            c = (1.0 - d) * (2.0 - d) * lmoments[1];
            b = 0.0;
            a = 0.0;
            xi = lmoments[0] - c / (1.0 - d);

            if(d <= 0.0)
            {
                a = c;
                b = -d;
                c = 0.0;
                d = 0.0;
            }
        }

        setAlpha(a);
        setBeta(b);
        setGamma(c);
        setDelta(d);
        setShift(xi);
    }

    /**
     * fitToLMoments() computes the 5 Wakeby parameters from the data
     * 
     * @param data - The data for which to calculate the 5 parameters.
     */

    @Override
    public void fitToLMoments(final DataSet data)
    {
        double lmoments[];

        // Get the 5 l-moments

        lmoments = LMomentsMath.dataToLMoments(data, 5);

        // Convert the 5 l-moments into 5 parameters

        fitToLMoments(lmoments);
    }

    public static void main(final String argv[])
    {
        final WakebyDist wake = new WakebyDist();

        System.out.println("Wakeby");
        System.out.println("Distribution of 1.5 is..." + wake.functionCDF(1.5));
        System.out.println("Density of 2.5 is..." + wake.functionPDF(2.5));
        System.out.println("Exceedance of 2.5 is..." + wake.functionExceedance(2.5));
        System.out.println("Inverse CDF of 0.5 is..." + wake.functionInverseCDF(0.5));
        System.out.println("Inverse CDF of 0.4 is..." + wake.functionInverseCDF(0.4));
        System.out.println("Inverse CDF of 0.1 is..." + wake.functionInverseCDF(0.1));
    }
}
