package ohd.hseb.hefs.utils.dist.types;

import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.DistributionTools;
import ohd.hseb.hefs.utils.dist.HMathTools;
import ohd.hseb.hefs.utils.dist.LMomentsFittingDistribution;
import ohd.hseb.hefs.utils.dist.LMomentsMath;
import ohd.hseb.hefs.utils.dist.MomentsFittingDistribution;
import ohd.hseb.hefs.utils.dist.ShiftOptimizationFittingDistribution;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.util.data.DataSet;

/**
 * An implementation of {@link ContinuousDist} for a Weibull distribution. This distribution includes a shape, scale,
 * and shift parameter. By default, however, the shift parameter is not fitted; that is, it is not set via the
 * {@link #fitToData(DataSet)} method. Rather, it must be set before fitting. To turn on shift fitting, call
 * {@link #setFitShift(boolean)} with true.
 * 
 * @author hank.herr
 */
public class WeibullDist extends ContinuousDist implements MomentsFittingDistribution, LMomentsFittingDistribution,
ShiftOptimizationFittingDistribution
{
    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SCALE = 0;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHAPE = 1;

    /**
     * Must be kept consistent with the position of the parameter in the constructors.
     */
    private static final int SHIFT = 2;

    private static final double DEFAULT_SCALE = 1.0D;
    private static final double DEFAULT_SHAPE = 2.0D;
    private static final double DEFAULT_SHIFT = 0.0D;

    private boolean _fitShift = false;

    /**
     * Constructs the Weibull distribution using the defaults provided by the constants: {@link #DEFAULT_SCALE},
     * {@link #DEFAULT_SHAPE}, and {@link #DEFAULT_FIXED_SHIFT}.
     */
    public WeibullDist()
    {
        this(DEFAULT_SCALE, DEFAULT_SHAPE, DEFAULT_SHIFT);
    }

    /**
     * Constructs the distribution given the specified parameters.
     * 
     * @param scale Scale parameter.
     * @param shape Shape parameter.
     * @param shift Shift parameter.
     */
    public WeibullDist(final double scale, final double shape, final double shift)
    {
        super(new XMLDouble("domain", shift, shift, null),
              new XMLDouble("scale", scale, 0.0D, null),
              new XMLDouble("shape", shape, 0.0D, null),
              new XMLDouble("shift", shift, null, null));
    }

    public void setFitShift(final boolean b)
    {
        _fitShift = b;
    }

    public void setScale(final double scale)
    {
        setParameter(SCALE, scale);
    }

    public double getScale()
    {
        return getParameter(SCALE).doubleValue();
    }

    public void setShape(final double shape)
    {
        setParameter(SHAPE, shape);
    }

    public double getShape()
    {
        return getParameter(SHAPE).doubleValue();
    }

    /**
     * Ensures {@link #setDomainLowerBound(Double)} is called as well.
     */
    public void setShift(final double shift)
    {
        setParameter(SHIFT, shift);
        setDomainLowerBound(shift);
    }

    public double getShift()
    {
        return getParameter(SHIFT).doubleValue();
    }

    @Override
    public void setParameter(final int index, final Number value)
    {
        super.setParameter(index, value);
    }

    /**
     * From MEFP fortran code.
     * 
     * @param shape The shape for which to calculate the coefficient of variation.
     * @return The computed coefficient of variation.
     */
    private double computeCoefficientOfVariationForShape(final double shape)
    {
        if(shape < 0)
        {
            return 0.01;
        }
        else
        {
            // The coefficient of variation be computed from:
            // http://en.wikipedia.org/wiki/Weibull_distribution
            // where mu is the mean. Note: The coefficient of variation does not depend
            // on the scale. For a given shape you can check it at Wolfram Alpha using:
            // Sqrt[Abs[Gamma[1. + 2./shape]/(Gamma[1. + 1./shape] * Gamma[1. + 1./shape]) - 1.]]

            final double g1 = Math.exp(HMathTools.gammaln(1.0 + 2d / shape));
            final double g2 = Math.exp(HMathTools.gammaln(1.0 + 1d / shape));
            return Math.sqrt(Math.abs((g1 / (g2 * g2)) - 1d));
        }
    }

    /**
     * From MEFP fortran code. This performs a search to find the shape to use. The maximum iterations is fixed to 100,
     * per the fortran code.
     * 
     * @param coefficientOfVariation The coefficient of variation for which to compute the shape.
     * @return The shape.
     */
    private double computeShapeGivenCoefficientOfVariation(final double coefficientOfVariation)
    {
        if(coefficientOfVariation <= 0)
        {
            return 0;
        }

        double shape2 = Math.pow(coefficientOfVariation, -0.875);
        double shape1 = 0.95 * shape2;
        double cv1 = computeCoefficientOfVariationForShape(shape1);
        double cv2 = 0;
        double deltaCV = 0;
        double dcvdb = 0;
        int iterationCount = 0;
        while(iterationCount < 100)
        {
            cv2 = computeCoefficientOfVariationForShape(shape2);
            deltaCV = cv2 - cv1;
            if(Math.abs(deltaCV) < 0.0001)
            {
                break;
            }
            dcvdb = deltaCV / (shape2 - shape1);
            shape1 = shape2;
            cv1 = cv2;
            shape2 = shape2 - (cv2 - coefficientOfVariation) / dcvdb;
            iterationCount++;
        }

        return shape2;
    }

    //Overrides Distribution class
    @Override
    public double functionCDF(final Double value)
    {
        final Double low = getDomainLowerBound();
        double temp;

        if((low == null) || isMissing(low))
        {
            throw new IllegalArgumentException("The lower bound is not defined, which is invalid.");
        }

        // Check the lower bound constraint.

        if(value < low)
        {
            return 0.0D;
        }

        // Try to perform the computation

        try
        {
            temp = 1 - Math.exp(-1 * Math.pow((value - getShift()) / getScale(), getShape()));
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        // Compare to Apache
        // 
        //final WeibullDistribution wrapped = new WeibullDistributionImpl(getShape(), getScale());
        //
        //try
        //{
        //    temp = wrapped.cumulativeProbability(value - getShift());
        //}
        //catch(final MathException e)
        //{
        //    return getMissing();
        //}

        // Make sure the number is within reasonable bounds.
        return DistributionTools.returnTrimmedProbability(temp);
    }

    @Override
    public double functionPDF(final Double value)
    {
        double a, b, g, temp;

        a = getScale();
        b = getShape();
        g = getShift();

        // Since the PDF is the derivative of the CDF, you can check this at Wolfram Alpha using:
        // derivative 1 - exp(-1 * pow((x - g) / a, b))

        try
        {
            temp = (b / a) * Math.pow((value - g) / a, b - 1) * Math.exp(-1 * Math.pow((value - g) / a, b));
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        // Compare to Apache ... for some reason WeibullDistributionImpl.density() is missing?
        //
        //final WeibullDistribution wrapped = new WeibullDistributionImpl(getShape(), getScale());
        //
        //try
        //{
        //    temp = wrapped.density(value - getShift());
        //}
        //catch(final MathException e)
        //{
        //    return getMissing();
        //}

        return temp;
    }

    @Override
    public double functionInverseCDF(final double prob)
    {
        double temp;

        if((prob >= 1) || (prob <= 0))
        {
            return getMissing();
        }

        try
        {
            temp = getScale() * Math.pow(-1 * Math.log(1 - prob), 1 / getShape()) + getShift();
        }
        catch(final ArithmeticException except)
        {
            return getMissing();
        }

        // Compare to Apache
        //
        //final WeibullDistribution wrapped = new WeibullDistributionImpl(getShape(), getScale());
        //
        //try
        //{
        //    temp = wrapped.inverseCumulativeProbability(prob) + getShift();
        //}
        //catch(final MathException e)
        //{
        //    return getMissing();
        //}

        return temp;
    }

    /**
     * @param data Be sure to call {@link DataSet#setFitSampleVariable(int)} and {@link DataSet#setFitCDFVariable(int)}
     *            prior to calling this routine.
     * @param shift The fixed shift parameter to assume in estimating the parameters.
     * @return True if successful, or false if a problem occurs.
     */
    public void estimateRegression(final DataSet data) throws DataFittingDistributionException
    {
        //First, do some basic QC checks...

        //Make sure the samplevar and empvar are set.
        int samplevar = data.getFitSampleVariable();
        int empvar = data.getFitCDFVariable();
        if((samplevar < 0) || (empvar < 0) || (samplevar == empvar))
        {
            throw new IllegalArgumentException("Sample variable number, " + samplevar
                + ", or empirical variable number, " + empvar + " is invalid.");
        }

        //The shift cannot be larger than any data value for the fit sample variable.
        if(getShift() > data.getSmallest(samplevar))
        {
            throw new DataFittingDistributionException("Smallest data value provided, " + data.getSmallest(samplevar)
                + ", is smaller than this distribution's lower bound, " + getShift());
        }

        //Make sure I have no missing values in any sample or empirical data
        if((data.isAnyVariableDataMissing(samplevar)) || (data.isAnyVariableDataMissing(empvar)))
        {
            throw new DataFittingDistributionException("Some provided data is  missing.");
        }

        //Make sure all the empirical values are within (0, 1) -- NOT INCLUDING 0 or 1!
        if((data.getSmallest(empvar) <= 0) || (data.getLargest(empvar) >= 1))
        {
            throw new DataFittingDistributionException("The range of computed empirical probabilities, smallest = "
                + data.getSmallest(empvar) + ", largest = " + data.getLargest(empvar) + ", is outside (0, 1).");
        }

        //The local variables.
        DataSet samples;
        double coeff, con;

        //Reset to the beginning of the data set. 
        data.resetPtr();

        //Create the regression data set.
        samples = createRegressDataSet(data);
        samplevar = samples.getFitSampleVariable();
        empvar = samples.getFitCDFVariable();

        //Calculate coeff and con, the regression coefficient and constant. 
        coeff = (samples.sumProduct(samplevar, empvar) - samples.getSampleSize() * samples.mean(samplevar)
            * samples.mean(empvar))
            / (samples.sumSquares(samplevar) - samples.getSampleSize() * Math.pow(samples.mean(samplevar), 2));
        con = samples.mean(empvar) - coeff * samples.mean(samplevar);

        //These are the parameter estimates.  
        setScale(Math.exp(con));
        setShape(1 / coeff);

        data.resetPtr();
    }

    /**
     * @param data Original data.
     * @return A {@link DataSet} in which variable 0 are the sample values and 1 are the sample empiricals.
     */
    private DataSet createRegressDataSet(final DataSet data)
    {
        final DataSet newdata = new DataSet(data.getMaximumSampleSize(), 2);
        boolean notatend = true;
        double[] asample = new double[2];

        int samplevar, empvar;
        samplevar = data.getFitSampleVariable();
        empvar = data.getFitCDFVariable();

        data.resetPtr();

        while(notatend)
        {
            //Set the data point in the new data set according to the equations below.
            //Since the regression is REVERSED, the values and empiricals variables are
            //swapped.
            if(data.getCurrentValue(samplevar) != DataSet.MISSING)
            {
                asample = new double[2];
                asample[0] = Math.log(Math.log(1 / (1 - data.getCurrentValue(empvar))));
                asample[1] = Math.log(data.getCurrentValue(samplevar) - getShift());
                newdata.addSample(asample);
            }

            notatend = data.next();
        }

        //Return the new data set.
        newdata.setFitSampleVariable(0);
        newdata.setFitCDFVariable(1);
        return newdata;
    }

    /**
     * fitToLMoments(data) fits the data to a 3 parameter Weibull distribution. The shift can be nonzero.
     * 
     * @param data - The data for which to calculate moments. The data should have FitSampleVariable set.
     */
    @Override
    public void fitToLMoments(final DataSet data)
    {
        final double lmoments[];

        // Get the 3 lmoment ratios

        lmoments = LMomentsMath.dataToLMoments(data, 3);

        // Convert the 3 lmoment ratios into 3 parameters

        fitToLMoments(lmoments);
    }

    /**
     * fitToLMoments(lmonents) fits the L-moment ratios to a 3 parameter Weibull distribution. The shift can be nonzero.
     * 
     * @param lmoments - The lmoment ratios.
     */
    @Override
    public void fitToLMoments(final double[] lmoments)
    {
        // No lmoments fitting routine was provided by HOSKING. The following is from: 
        // 
        // http://journals.tdl.org/icce/index.php/icce/article/viewFile/1154/pdf_42
        //
        // A <-> scale
        // k <-> shape
        // B <-> shift

        final double c0 = 3.516;
        final double c1 = -21.256;
        final double c2 = 98.52;
        final double c3 = -317.2;
        final double c4 = 622.8;
        final double c5 = -658.6;
        final double c6 = 285.3;

        double k, temp;

        k = c0;

        temp = lmoments[2];
        k += c1 * temp;

        temp *= lmoments[2];
        k += c2 * temp;

        temp *= lmoments[2];
        k += c3 * temp;

        temp *= lmoments[2];
        k += c4 * temp;

        temp *= lmoments[2];
        k += c5 * temp;

        temp *= lmoments[2];
        k += c6 * temp;

        if(k == 0.0)
        {
            throw new IllegalArgumentException("k = 0.0");
        }

        setShape(k);

        temp = Math.exp(HMathTools.gammaln(1.0 + 1.0 / k)); // = Gamma(1+1/k)

        setScale(lmoments[1] / ((1.0 - Math.pow(2.0, -1.0 / k)) * temp));

        setShift(lmoments[0] - (getScale() * temp));
    }

    @Override
    public void fitToData(final DataSet data, final double[] fitParms) throws DataFittingDistributionException
    {
        //Fit parms are not used
        if(!_fitShift)
        {
            estimateRegression(data);
        }
//SHIFT OPTIMIZATION CODE (comment out the else below):
//        else
//        {
//            if(fitParms == null)
//            {
//                fitParms = new double[]{0.0D}; //Minimum set to 0!!!
//            }
//            DistributionTools.optimizeShiftFitForBoundedBelowDistribution(this, data, fitParms);
//        }
        else
        {
            fitToLMoments(data);
        }
    }

    @Override
    public void fitToMoments(final double mean, final double coefficientOfVariation)
    {
        setShape(computeShapeGivenCoefficientOfVariation(coefficientOfVariation));
        setScale(mean / Math.exp(HMathTools.gammaln(1 + 1 / getShape())));
    }

    @Override
    public void estimateParameters(final DataSet data, final double shift) throws DataFittingDistributionException
    {
        final WeibullDist fitDist = new WeibullDist();
        fitDist.setFitShift(false); //Force a two-param fit
        fitDist.setShift(shift);
        fitDist.fitToData(data);

        //COpy the results
        setScale(fitDist.getScale());
        setShape(fitDist.getShape());
        setShift(shift);
    }

    ////////////////////////////////////////
    public static void main(final String argv[])
    {
        final WeibullDist weib = new WeibullDist(2.0, 3.0, 1.0);

        System.out.println("Weibull");
        System.out.println("Distribution of 2.5 is " + weib.functionCDF(2.5));
        System.out.println("Density of 2.5 is " + weib.functionPDF(2.5));
        System.out.println("Exceedance of 2.5 is " + weib.functionExceedance(2.5));
        System.out.println("Inverse CDF of 0.5 is " + weib.functionInverseCDF(0.5));
        System.out.println("Coefficient Of Variation is " + weib.computeCoefficientOfVariationForShape(weib.getShape()));
    }
}
