package ohd.hseb.hefs.utils.effect;

import static com.google.common.collect.Lists.newArrayList;

import java.util.List;

/**
 * A series of effects treated as a single one.
 * 
 * @author alexander.garbarino
 * @param <A>
 */
public class CompositeEffect<A> extends Effect<A>
{
    private final List<Effect<? super A>> _components;

    CompositeEffect(final Effect<? super A>... components)
    {
        _components = newArrayList(components);
    }

    private CompositeEffect(final CompositeEffect<A> startEffects, final Effect<? super A> lastEffect)
    {
        _components = newArrayList(startEffects._components);
        _components.add(lastEffect);
    }

    @Override
    public void perform(final A input)
    {
        for(final Effect<? super A> component: _components)
        {
            component.perform(input);
        }
    }

    /**
     * Creates a new effect which first performs {@code this} and then {@code afterEffect}.
     * 
     * @param afterEffect the effect to perform after this one
     */
    @Override
    public Effect<A> then(final Effect<? super A> afterEffect)
    {
        return new CompositeEffect<A>(this, afterEffect);
    }

    /**
     * Creates a new effect which first performs {@code this} and then runs {@code afterRunnable}.
     * 
     * @param afterRunnable the runnable to run after this effect is performed
     */
    @Override
    public Effect<A> then(final Runnable afterRunnable)
    {
        return new CompositeEffect<A>(this, new RunnableEffect(afterRunnable));
    }
}
